/** \file functions.cpp
 * Part of GLISSANDO
 * 
 */


/*! \mainpage
                                                                                         
            GLISSANDO - GLauber Initial State Simulation AND mOre...         \n
                        ver. 2.07, 25 August 2010                            \n
                                                                             \n 
  Authors: 
           - Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              
           - Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              
           - Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                    \n  
                                                                             \n
  
  Modification of the code to ver. 2 by WB.

  For the detailed description of ver. 1 program and further references         
  to the description of the model, please, refer to our                                                                                                   
  Computer Physics Communications 180(2009)69, arXiv:0710.5731 [nucl-th]     \n
  accessibile from: http://arxiv.org.abs/0710.5731                           \n
                                                                             \n
  Implementation of nuclear correlations in ver. 2 as described in 
  Phys. Rev. C81(2010)064909                                                 \n
                                                                             \n
  Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      \n
  
  GLISSANDO is a Glauber Monte-Carlo generator for early-stages of relativistic 
  heavy-ion collisions, written in c++ and interfaced to ROOT. Several models 
  are implemented: the wounded-nucleon model, the binary collisions model, 
  the mixed model, and the model with hot-spots. The original geometric 
  distribution of sources (i.e., wounded nucleons or binary collisions) in 
  the transverse plane can be superimposed with a statistical distribution 
  simulating the dispersion in the generated transverse energy in each individual 
  collision. The program generates inter alia the fixed axes (standard) 
  and variable-axes (participant) two-dimensional profiles of the density 
  of sources in the transverse plane and their Fourier components. These profiles 
  can be used in further analyses of physical phenomena, such as the the jet 
  quenching, event-by-event hydrodynamics, or analyses of the elliptic flow 
  and its fluctuations. Characteristics of the event (multiplicities, eccentricities, 
  Fourier shape coefficients, etc.) are evaluated and stored in a ROOT file 
  for further off-line studies. A number of scripts is provided for that purpose. 
  The code can also be used for the proton-nucleus and deuteron-nucleus collisions. \n
                                                                             \n

  Version 2 of GLISSANDO offers much more functionality than version 1, 
  moreover, it is fully object-oriented, providing the user with the flexibility 
  of inspecting and, if needed, modyfying the code in a simple manner. New 
  features involve:                                                          \n
                                                                             \n
- The possibility of feeding into the simulations the nuclear distributions 
  accounting for the two-body NN correlations (read from external files, see
  Alvioli, Drescher and Strikman, [Phys. Lett. B680, 225, 2009], the distributions 
  can be found at http://www.phys.psu.edu/~malvioli/eventgenerator/ )  
- The use of the Gaussian NN wounding profile (which is more realistic than 
  the commonly-used hard-core wounding profile, see the analysis by Bialas 
  and Bzadak [Acta Phys. Polon. B38, 159,2007]) 
- The generation of the core-mantle (core-corona) distributions (see Bozek 
  [Acta Phys. Polon. B36, 3071,2005] and Werner [Phys. Rev. Lett. 98, 152301, 
  2007], see also Becattini and Manninen [Phys. Lett. B673, 19, 2009] and Bozek 
  [Phys. Rev. C79, 054901, 2009]) 
- The analysis of the triangular shape deformation parameter and profiles, 
  relevant for the triangular flow, see Alver and Roland, [Phys. Rev. C81, 054905, 
  2010] and Alver, Gombeaud, Luzum,and Ollitrault, [arXiv:1007.5469] 
- Generation of rapidity distributions in the wounded-nucleon picture according 
  to the model of Bialas and Czyz [Acta Phys.Polon.B36:905-918,2005], as implemented 
  by Bozek [arXiv:1002.4999]. This allows to obtain the fully 3-dimensional
  distribution of matter in the early Glauber phase of the collision.        \n 
                                                                          \n \n
  The reference manual for ver. 2, generated by Doxygen, is supplied at the home
  page. The full write-up of ver. 2 is under preparation.                    \n
                                                                             \n
  The code can be freely used and redistributed. However, if you decide to  
  make modifications, the authors would appreciate notification for the record.             
  Any publication or display of results obtained using GLISSANDO must        
  include a reference to our published paper.                                

*/

#include <math.h>
#include <time.h>
#include <string.h>
#include <iostream>
#include <fstream>
#include <cstdlib>

#include <TH1D.h>
#include <TH2D.h>
#include <TFile.h>
#include <TTree.h>
#include <TRandom3.h>

#include "functions.h"

using namespace std;

extern float ver; //!< version of the code

/**********************************
parameters and their default values
**********************************/
int        EVENTS=50000,    //!< number of generated events 
           NBIN=40,         //!< number of bins for histogramming in x, y, and r 
           FBIN=72,         //!<  number of bins for histogramming in the azimuthal angle
           NUMA=197,        //!< mass number of nucleus A 
           NUMB=197,        //!< mass number of nucleus B 
           WMIN=2,          //!< minimum number of wounded nucleons to record the event 
           MODEL=0,         //!< switch for the superimposed multiplicity distribution: 0 - uniform, 1 - Poisson, 2 - Gamma
           DOBIN=0,         //!< 1 - count binary collisions even in the pure wounded-nucleon model. 0 - do not
           W0=2,            //!< minimum allowed number of wounded nucleons in the acceptance window
           W1=1000,         //!< maximum allowed number of wounded nucleons in the acceptance window
           SHIFT=1,         //!< 1 - shift the coordinates of the fireball to c.m. in the fixed-axes case (preferred), 0 - do not
           RET=0,           //!< 0 - fix-last algorithm (preferred), 1 - return-to-beginning algorithm for the generation of the nuclear distribution
           FULL=0,          //!< 1 - generate the full event tree (large output file), 0 - do not
           FILES=0,         //!< 1 - read distribution from files, 0 - do not
           GAUSS=0,         //!< 1 - Gaussian wounding profile, 0 - hard-sphere wounding profile
           NUMRAP=10;       //!< number of particles per unit weight generated in the whole rapidity range
           

UInt_t        ISEED,        //!< read seed for the ROOT random number generator, if 0 - random seed generated
              ISEED1;       //!< copy of ISEED

float        BMIN=0.,              //!< minimum value of the impact parameter in the acceptance window
             BMAX=25.,             //!< maximum value of the impact parameter in the acceptance window
             RDS0=0.,              //!< minimum value of the relative deposited strength (RDS) in the acceptance window 
             RDS1=100000,          //!< maximum value of the relative deposited strength (RDS) in the acceptance window 

       BTOT = fmax(RWSA,RWSB)+AWSA+AWSB,    //!< maximum impact parameter value for histogramming

             RWSA=6.43,            //!< Woods-Saxon radius for nucleus A
             AWSA=0.45,            //!< Woods-Saxon width for nucleus A
             RWSB=6.43,            //!< Woods-Saxon radius for nucleus B
             AWSB=0.45,            //!< Woods-Saxon width for nucleus B
             WFA=0.,               //!< the w parameter for the Fermi distribution for nucleus A
             WFB=0.,               //!< the w parameter for the Fermi distribution for nucleus B
             SNN=42.,              //!< NN "wounding" cross section in millibarns
             SBIN=42.,             //!< NN binary cross section in millibarns
             ALPHA=0.145,          //!< the mixing parameter: 0 - wounded, 1 - binary, 0.145 - mixed (PHOBOS)
             Uw=2.,                //!< Poisson or Gamma parameters for superimposed distribution, wounded nucleons
             Ubin=2.,              //!< Poisson or Gamma parameters for superimposed distribution, binary collisions
             PI=4.*atan(1.),       //!< the number pi
             CD=0.9,               //!< closest allowed distance (expulsion distance) between nucleons in the nucleus in fm (simulation of repulsion)
             DW=0.,                //!< dispersion of the location of the source for wounded nucleons (in fm)
             DBIN=0.,              //!< dispersion of the location of the source for binary collisions (in fm)
             GA=0.92,              //!< Gaussian wounding profile parameter (hight at the origin)
             RAPRANGE=5.,          //!< range in rapidity
             ETA0=1.,		   //!< 2*ETA0 is the width of the plateau in eta
             ETAM=3.36,            //!< parameter of the Bialas-Czyz-Bozek model
             SIGETA=1.3,           //!< parameter controling the witdth of the rapidity distribution
             MAXYRAP=1000;         //!< maximum absolute value of the y coordinate in the x-y-rapidity histogram  

/*************************************
 declaration of counters and variables 
*************************************/

counter2 
estd,		  //!< counter for epsilon standard (fixed-axes), <r^2 cos(2 phi)>/<r^2>
epart,            //!< counter for epsilon participant (variable-axes), <r^2 cos(2 phi)>/<r^2>
estd3,            //!< counter for fixed-axes <r^2 cos(3 phi)>/<r^2>
epart3,           //!< counter for variable-axes <r^2 cos(3 phi)>/<r^2>
estd4,            //!< counter for fixed-axes <r^2 cos(4 phi)>/<r^2>
epart4,           //!< counter for variable-axes <r^2 cos(4 phi)>/<r^2>
estd5,            //!< counter for fixed-axes <r^2 cos(5 phi)>/<r^2>
epart5,           //!< counter for variable-axes <r^2 cos(5 phi)>/<r^2>
estd6,            //!< counter for fixed-axes <r^2 cos(6 phi)>/<r^2>
epart6,           //!< counter for variable-axes <r^2 cos(6 phi)>/<r^2>
nwounded,         //!< counter for number of wounded nucleons
nbinary,          //!< counter for number of binary collisions
nhot,             //!< counter for number of hot-spots
nweight;          //!< counter for relative deposited strength (RDS)

int evall=0,      //!< number of all attempted event
    kk;           //!< number of the current event

float 
d,             //!< the wounding distance
dbin;          //!< the binary-collision distance 

Float_t 
b,         //!< impact parameter 
sitot,     //!< the total A+B cross section in the acceptance window
sirad,     //!< equivalent hard-sphere radius for the cross section
rwA,       //!< number of wounded nucleons in A
rwB,       //!< number of wounded nucleons in B
rwAB,      //!< number of all wounded nucleons
rbin,      //!< number of binary collisions
rhotspot,  //!< number of hot-spots
rpa,       //!< relative deposited strength (RDS)
sizeav,	   //!< size
es,        //!< epsilon standard (fixed-axes), <r^2 cos(2 phi)>/<r^2>
ess,       //!< fixed-axes sine moment, <r^2 sin(2 phi)>/<r^2>
ep,        //!< epsilon participant (variable-axes), <r^2 cos(2 phi)>/<r^2>
eps,       //!< variable-axes sine moment, <r^2 sin(2 phi)>/<r^2> 
es3,       //!< fixed-axes <r^2 cos(3 phi)>/<r^2>
es3s,      //!< fixed-axes sine moment, <r^2 sin(3 phi)>/<r^2>
ep3,       //!< variable-axes <r^2 cos(3 phi)>/<r^2>
ep3s,      //!< variable-axes sine moment, <r^2 sin(3 phi)>/<r^2>
es4,       //!< fixed-axes <r^2 cos(4 phi)>/<r^2>
es4s,      //!< fixed-axes sine moment, <r^2 sin(4 phi)>/<r^2>
ep4,       //!< variable-axes <r^2 cos(4 phi)>/<r^2>
ep4s,      //!< variable-axes sine moment, <r^2 sin(4 phi)>/<r^2>
es5,       //!< fixed-axes <r^2 cos(5 phi)>/<r^2>
es5s,      //!< fixed-axes sine moment, <r^2 sin(5 phi)>/<r^2>
ep5,       //!< variable-axes <r^2 cos(5 phi)>/<r^2>
ep5s,      //!< variable-axes sine moment, <r^2 sin(5 phi)>/<r^2>
es6,       //!< fixed-axes <r^2 cos(6 phi)>/<r^2>
es6s,      //!< fixed-axes sine moment, <r^2 sin(6 phi)>/<r^2>
ep6,       //!< variable-axes <r^2 cos(6 phi)>/<r^2>
ep6s,      //!< variable-axes sine moment, <r^2 sin(6 phi)>/<r^2>
phirot,    //!< rotation angle maximizing the second Fourier moment 
phirot3,   //!< rotation angle maximizing the third Fourier moment 
phirot4,   //!< rotation angle maximizing the fourth Fourier moment 
phirot5,   //!< rotation angle maximizing the fifth Fourier moment 
phirot6,   //!< rotation angle maximizing the sixth Fourier moment 
xx,        //!< center-of-mass x coordinate
yy,        //!< center-of-mass y coordinate
xeps,      //!< average es 
xseps,     //!< standard deviation of es
xepp,      //!< average ep 
xsepp;     //!< standard deviation of ep


// specific functions

//! random number generator using the built-in ROOT generator, uniform on (0,1)
float los() {return raa.Uniform(0,1);}; 

//! random number generator for the Woods-Saxon (or Fermi) distribution - nucleus A
float rlosA(){int x=0;while(x==0){float rr=3.*RWSA*los();float b=RWSA*RWSA*los();
	       if(b<rr*rr*(1.+WFA*rr*rr/RWSA/RWSA)/(exp((rr-RWSA)/AWSA)+1)){x=1;return rr;};};}; // Woods-Saxon/Fermi

//! random number generator for the Woods-Saxon (or Fermi) distribution - nucleus B
float rlosB(){int x=0;while(x==0){float rr=3.*RWSB*los();float b=RWSB*RWSB*los();
	       if(b<rr*rr*(1.+WFB*rr*rr/RWSB/RWSB)/(exp((rr-RWSB)/AWSB)+1)){x=1;return rr;};};}; //Woods-Saxon/Fermi

//! random number generator for the Hulthen distribution
/*! The Hulthen distribution used to generate the distance between nucleons in the deuteron */
float rlos_hult(){double const ah=.228;double const bh=1.118;int x=0;while(x==0){float rr=15.*los();
float b=.25*los(); // 0.25 is an upper bound of the probability distribution
 if(b< 2*ah*bh*(ah+bh)*(exp(-2.*ah*rr)+exp(-2.*bh*rr)-2.*exp(-(ah+bh)*rr))/(ah-bh)/(ah-bh)){x=1;return rr;};};}; //Hulthen

//! rapidity distribution with the plateau 
float fg(float rr){return exp(-(sqrt(rr*rr)-ETA0)*(sqrt(rr*rr)-ETA0)/(2*SIGETA*SIGETA)*max(0., sqrt(rr*rr)-ETA0) );};

//! function f +/- from Bozek, arXiv:1002.4999v2 [nucl-th]
float fpm(float rr) { if(rr<-ETAM) return 0; else if(rr<ETAM) return (rr+ETAM)/(2*ETAM); else return 1.; }

//! random number generator for the rapidity distribution - wounded nucleons from nucleus A
float los_rap_A(){
int x=0;while(x==0) {float rr=raa.Uniform(-RAPRANGE,RAPRANGE); float b=raa.Uniform(0.,1.);
	       if( b < fg(rr)*fpm(rr) ) {x=1;return rr;};};
                   };

//! random number generator for the rapidity distribution - wounded nucleons from nucleus B
/*!< Formula for the wounded quark rapidity profile from Bozek, arXiv:1002.4999v2 [nucl-th]. */
float los_rap_B(){
int x=0;while(x==0) {float rr=raa.Uniform(-RAPRANGE,RAPRANGE); float b=raa.Uniform(0.,1.);
	       if( b < fg(rr)*fpm(-rr) ) {x=1;return rr;};};
                   };

//! random number generator for the rapidity distribution - binary collisions
/*!< Formula for the wounded quark rapidity profile from Bozek, arXiv:1002.4999v2 [nucl-th]. */
float los_rap_bin(){
int x=0;while(x==0) {float rr=raa.Uniform(-RAPRANGE,RAPRANGE); float b=raa.Uniform(0.,1.);
	       if( b < fg(rr) ) {x=1;return rr;};};
                   };

//! random number generator for the Gamma distribution
float gamgen(
            float a //!< the parameter in f(x) = x^(a-1) exp(-x)/Gamma(a)
            ){
int k = int(a);
float b = a - k;
float x1 = 0.0;
float u=0.0;
float v=0.0;
float x2=0.0;
float s;

if(k==0) goto blok;
s = 1.0 ;
for (int i=1;i<=k;i++)
s=s*los();
x1 = -log(s);

if (b>1e-6) goto blok;
return x1;
 
blok:x2=-log(los());

bloka:u=pow(los(),1.0/b);
v=u+pow(los(),1.0/(1.0-b));
if (v>1.0) goto bloka;
return (x1+x2*u/v);
}//gamgen


//negative binomial distribution
//int negbin(float m,float v) {
//float p=1.0-(m/v);
//float r=m*(1.0-p)/p;
//float x=gamgen(r)*p/(1.0-p);
//int n=ipois(x);
//return n;
//}//negbin


//! overlaid distribution
/*! The sources are supplied with a random weight, generated according to the distribution selected with m. */  
float dist(
           int m,  //!< case: 0 - uniform, 1 - Poisson, 2 - Gamma 
           float u //!< average of the distribution in cases 1 and 2
          ){
 switch(m) {
 case 0: { return 1; break; }
 case 1: { return raa.Poisson(u)/u;  break; } 
 case 2: { return gamgen(u)/u; break; }
}
}

//! random shift of the source location
/*! The location of the source may by shifted randomly when DW>0 od DBIN>0, with the Gaussian distribution of width w. */
float disp(
          float w //! average shift, Gaussian distribution 
          ){return raa.Gaus(0.,w);};

/**************************
        Blocks
**************************/

//! print the help
void helper(int argc, char* str)
{if(argc > 1){
 if(strcmp(str,"-v")==0){cout << "GLISSANDO ver: " << ver << endl; exit(0);};
 if(strcmp(str,"-h")==0){cout << endl << "Usage:" << endl 
                                 << "glissando.exe <input_file> <output_Root_file>" << endl
                                 << "glissando_files <input_file> <output_Root_file> <nucl_dictribution_A> <nucl_distribution_B>" 
                                 << endl << endl
                                 << "see Comp. Phys. Comm. 180(2009)69, http://arxiv.org.abs/0710.5731, " << endl
                                 << "and Phys. Rev. C81(2010)064909 for more details"
                                 << endl << endl; exit(0);};
};};

//! print the header in the output
void header(){
cout << endl << "*******************************************************" << endl;
cout << "GLISSANDO ver. " << ver << endl << 
"Computer Physics Communications 180(2009)69, http://arxiv.org.abs/0710.5731" << endl <<
"and Phys. Rev. C81(2010)064909 for implementation of the NN correlations" << endl; 
cout << "(tested with ROOT ver. 5.08--5.21)" << endl;
cout << "********************************************************" << endl;
cout << "Simulation of nucleus-nucleus collisions in Glauber models" << endl; 

cout << "--------------------------------------" << endl;
#if(_profile_) 
cout << "generation of the nuclear profile" << endl; 
cout << "--------------------------------------" << endl;
#endif
#if(_weight_) 
cout << "generation of the superimposed distribution" << endl; 
cout << "--------------------------------------" << endl;
#endif
};

//! start the time measurement
int time_start(){
time_t rawtime; 
struct tm * timeinfo;
time ( &rawtime ); 
timeinfo = localtime ( &rawtime ); 
int ti=int(rawtime);
cout << "Start: " << asctime (timeinfo); // time stamp
cout << "--------------------------------------" << endl;
return ti;
};

//! stop the time measurement
void time_stop(
              int ts //!< time at start
              ){
time_t rawtime; 
struct tm * timeinfo;
cout << endl; time ( &rawtime ); timeinfo = localtime ( &rawtime ); 
int ti=int(rawtime) - ts; int tig=ti/3600; ti=ti-tig*3600; int tim=ti/60; ti=ti-tim*60;
cout << "Finish: " << asctime (timeinfo) << "(" << tig << "h:" << tim <<"m:" << ti << "s)" << endl;
cout << "**************************************" << endl << endl;
}

//! process the input file with parameters
void readpar(
            TString inpfile //!< name of the input file
            ){ 

FILE *in; 
in = fopen(inpfile,"rt");

char t[30];

char s1[]={"EVENTS"},
     s2[]={"NBIN"},
     s3[]={"NUMA"},
     s4[]={"NUMB"},
     s5[]={"WMIN"},
     s6[]={"BMIN"},
     s7[]={"BMAX"},
     s8[]={"RWSA"},

     s9[]={"AWSA"},
     s10[]={"RWSB"},
     s11[]={"AWSB"},
     s12[]={"SNN"},
     s13[]={"SBIN"},
     s14[]={"ALPHA"},
     s15[]={"Uw"},
     s16[]={"Ubin"},
     s17[]={"CD"},
     s18[]={"MODEL"},
     s19[]={"ISEED"},
     s20[]={"BTOT"},
     s21[]={"W0"},
     s22[]={"W1"},
     s23[]={"RDS0"},
     s24[]={"RDS1"},
     s25[]={"SHIFT"},
     s26[]={"RET"},
     s27[]={"DW"},
     s28[]={"DBIN"},
     s29[]={"WFA"},
     s30[]={"WFB"},
     s31[]={"FULL"},
     s32[]={"FBIN"},
     s33[]={"DOBIN"},
     s34[]={"GA"},
     s35[]={"FILES"};

int GAUSS; // 1 - compute the shape of the wounding profile, 0 - do not
	#if(_gauss_!=1)
             GAUSS=0;
	#else
	     GAUSS=1;
	#endif


cout << "parameters reset from default in " << inpfile << " :" << endl;

//! scan the input file for the parameters reset from the default values
int fBTOT=0; double v; 
while (!feof(in)){
  fscanf(in,"%s %lf",t,&v);
  if ((!feof(in)) && (t[0]!='#') && (t[0]!='I')) cout << t << "\t"<< v <<endl;

  if ((!feof(in)) && (t[0]!='#') && (t[0]=='I')) cout << t << "\t"<< UInt_t(v) <<endl;
  if (!strcmp(t,s1)) {EVENTS=int(v);goto dalej;} // number of events
  if (!strcmp(t,s2)) {NBIN=int(v);goto dalej;}   // number of bins in histograms
  if (!strcmp(t,s3)) {NUMA=int(v);goto dalej;}   // mass number of nucleus A
  if (!strcmp(t,s4)) {NUMB=int(v);goto dalej;}   // mass number of nucleus B
  if (!strcmp(t,s5)) {WMIN=int(v);goto dalej;}   // minimum number of wounded nucleons to record event
  if (!strcmp(t,s6)) {BMIN=v;goto dalej;}        // minimum impact parameter
  if (!strcmp(t,s7)) {BMAX=v;goto dalej;}        // maximum impact parameter
  if (!strcmp(t,s8)) {RWSA=v;goto dalej;}        // Woods-Saxon radius on nucleus A
  if (!strcmp(t,s9)) {AWSA=v;goto dalej;}        // Wood-Saxon surface thickness parameter for nucleus A
  if (!strcmp(t,s10)) {RWSB=v;goto dalej;}       // Woods-Saxon radius on nucleus B
  if (!strcmp(t,s11)) {AWSB=v;goto dalej;}       // Wood-Saxon surface thickness parameter for nucleus B
  if (!strcmp(t,s12)) {SNN=v;goto dalej;}        // wounding cross section
  if (!strcmp(t,s13)) {SBIN=v;goto dalej;}       // binary cross section
  if (!strcmp(t,s14)) {ALPHA=v;goto dalej;}      // the mixing parameter
  if (!strcmp(t,s15)) {Uw=v;goto dalej;}         // parameter for the distribution superimposed over the wounded nucleons
  if (!strcmp(t,s16)) {Ubin=v;goto dalej;}       // parameter for the distribution superimposed over the binary collisions
  if (!strcmp(t,s17)) {CD=v;goto dalej;}         // the expulsion distance
  if (!strcmp(t,s18)) {MODEL=int(v);goto dalej;}     // switch for the superiposed distribution
  if (!strcmp(t,s19)) {ISEED=UInt_t(v);goto dalej;}  // seed for the random-number generator
  if (!strcmp(t,s20)) {fBTOT=1;BTOT=v;goto dalej;}   // range for histograms
  if (!strcmp(t,s21)) {W0=int(v);goto dalej;}     // minimum number of wounded nucleons
  if (!strcmp(t,s22)) {W1=int(v);goto dalej;}     // maximum number of wounded nucleons
  if (!strcmp(t,s23)) {RDS0=v;goto dalej;}        // minimum RDS (relative deposited stregth, see the paper)
  if (!strcmp(t,s24)) {RDS1=v;goto dalej;}        // maximum RDS
  if (!strcmp(t,s25)) {SHIFT=int(v);goto dalej;}  // parameter controling the shift of the fixed-axes distributions to c.m. frame 
  if (!strcmp(t,s26)) {RET=int(v);goto dalej;}    // parameter controling nuclear density generation
  if (!strcmp(t,s27)) {DW=v;goto dalej;}          // width of the distribution of displacement of the location of source for wounded nucleons
  if (!strcmp(t,s28)) {DBIN=v;goto dalej;}        // width of the distribution of displacement of the location of source for binary collisions
  if (!strcmp(t,s29)) {WFA=v;goto dalej;}         // the w parameter of the Fermi distribution for nucleus A
  if (!strcmp(t,s30)) {WFB=v;goto dalej;}         // the w parameter of the Fermi distribution for nucleus B
  if (!strcmp(t,s31)) {FULL=int(v);goto dalej;}   // parameter controling generation of the full event tree 
  if (!strcmp(t,s32)) {FBIN=int(v);goto dalej;}   // number of histogram bins in the the phi angle
  if (!strcmp(t,s33)) {DOBIN=int(v);goto dalej;}  // if 1, generate binary collision also for the pure wounded-nucleon model
  if (!strcmp(t,s34)) {GA=v;goto dalej;}          // value of the gaussian wounding profile at the origin
  if (!strcmp(t,s35)) {FILES=int(v);goto dalej;}  // not used
 dalej:;
 };

if(RET==1){cout << "Return to beginnning algorithm no longer supported, use RET=0 in the input" << endl; exit(0);};

#if(_files_)
FILES=1;
#else
FILES=0;
#endif

//! correct wrong input
 if ((FILES==1) && CD*CD > 0.000001){CD=0; cout << "correction: CD=" << CD << endl;};
 if ((MODEL!=0) && (MODEL!=1) && (MODEL!=2)) {MODEL=0; cout << "correction: MODEL=" << MODEL << endl;};
 if (BMIN<0) {BMIN=0; cout << "correction: BMIN=" << BMIN << endl;};
 if (BMAX<BMIN) {BMAX=BMIN; cout << "correction: BMAX=" << BMAX << endl;};
 if ((ALPHA<0)||(ALPHA>1)) {ALPHA=0; cout << "correction: ALPHA=" << ALPHA << endl;};
 if ((SHIFT!=0) && (SHIFT!=1)) {SHIFT=0; cout << "correction: SHIFT=" << SHIFT << endl;};
 if (W0<WMIN) {W0=WMIN; cout << "correction: W0=" << W0 << endl;};
 if (W1<W0) {W1=W0; cout << "correction: W1=" << W1 << endl;};
 if (RDS1<RDS0) {RDS1=RDS0+1; cout << "correction: RDS1=" << RDS1 << endl;};
 if (Uw<0) {Uw=1; cout << "correction: Uw=" << 1 << endl;};
 if (Ubin<0) {Ubin=1; cout << "correction: Ubin=" << 1 << endl;};
 if (CD<0) {CD=-CD; cout << "correction: CD=" << CD << endl;};

//! see the paper for the discussion of parametrizatiations of nuclear distributions
if ((RWSA<=0) && ((CD-0.9)*(CD-0.9)<0.001) && (RET==0)) {
   RWSA=1.1*pow(NUMA,1./3.)-0.656*pow(NUMA,-1./3.); AWSA=0.459;
   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.459fm (approximate formula used, see the paper)" << endl;};
if ((RWSB<=0) && ((CD-0.9)*(CD-0.9)<0.001) && (RET==0)) {
   RWSB=1.1*pow(NUMB,1./3.)-0.656*pow(NUMB,-1./3.); AWSB=0.459;
   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSB=0.459fm (approximate formula used, see the paper)" << endl;};

if ((RWSA<=0) && ((CD-0.8)*(CD-0.8)<0.001) && (RET==0)) {
   RWSA=1.103*pow(NUMA,1./3.)-0.55*pow(NUMA,-1./3.); AWSA=0.455;
   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.455fm (approximate formula used, see the paper)" << endl;};
if ((RWSB<=0) && ((CD-0.8)*(CD-0.8)<0.001) && (RET==0)) {
   RWSB=1.103*pow(NUMB,1./3.)-0.55*pow(NUMB,-1./3.); AWSB=0.455;
   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSB=0.455fm (approximate formula used, see the paper)" << endl;};

if ((RWSA<=0) && ((CD-0.4)*(CD-0.4)<0.001) && (RET==0)) {
   RWSA=1.113*pow(NUMA,1./3.)-0.277*pow(NUMA,-1./3.); AWSA=0.45;
   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.45fm (approximate formula used, see the paper)" << endl;};
if ((RWSB<=0) && ((CD-0.4)*(CD-0.4)<0.001) && (RET==0)) {
   RWSB=1.113*pow(NUMB,1./3.)-0.277*pow(NUMB,-1./3.); AWSB=0.45; 
   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSA=0.45fm (approximate formula used, see the paper)" << endl;};

if ((RWSA<=0) && (CD*CD<0.001) && (RET==0)) {
   RWSA=1.114*pow(NUMA,1./3.)-0.246*pow(NUMA,-1./3.); AWSA=0.45;
   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.45fm (approximate formula used, see the paper)" << endl;};
if ((RWSB<=0) && (CD*CD<0.001) && (RET==0)) {
   RWSB=1.114*pow(NUMB,1./3.)-0.246*pow(NUMB,-1./3.); AWSB=0.45; 
   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSA=0.45fm (approximate formula used, see the paper)" << endl;};

if ((RWSB<=0 || RWSB<=0)) {cout << "Correct radii! (no approximate formula available for the chosen CD)"<< endl << endl; exit(0); };

// if (NUMA!=NUMB) {SHIFT=1; cout << "SHIFT=1 for collisions of different nuclei" << endl;};
// set the range for the histograms
if (fBTOT!=1) BTOT = fmax(RWSA,RWSB)+AWSA+AWSB;
};

//! echo parameters to the output
void echopar(){
cout << endl;
if(ISEED1==0){cout << "random ";} else { cout << "fixed ";};
cout << "seed: " << raa.GetSeed()  << ", number of events: " << EVENTS << endl;
#if(_profile_)
   cout << "generates the nuclear density profile for nucleus A" << endl;
#endif
cout << NUMA <<"+" << NUMB;
if((NUMA>2) && (FILES!=1)){cout << ", RA="  << RWSA << "fm, aA=" << AWSA << "fm";};
if((NUMB>2) && (FILES!=1)){cout << ", RB="  << RWSB << "fm, aB=" << AWSB << "fm";}
if(FILES!=1){cout << ", d=" << CD  << "fm";};
if((WFA*WFA>0.00000001) && (FILES!=1)) {cout << ", wA="<< WFA;}
if((WFB*WFB>0.00000001) && (FILES!=1)) {cout << ", wB="<< WFB;}
cout << endl; 
if(ALPHA==0){cout << "wounded nucleon model: sig_w=" << SNN << "mb" << endl;};
if(ALPHA==0 && DOBIN==1){cout << "   (binary collisions counted)" << endl;};
if(ALPHA==0 && DOBIN!=1){cout << "   (binary collisions not counted)" << endl;};
if(ALPHA>0 && SBIN>=SNN){cout << "mixed model: sig_w=" << SNN << "mb, sig_bin=" << SBIN << "mb, alpha=" << ALPHA << endl;};
if(ALPHA>0 && SBIN<SNN){cout << "mixed model with hotspots: sig_w=" << SNN << "mb, sig_bin=" << SBIN << "mb, alpha=" << ALPHA << endl;};
if(MODEL==1){cout << "overlayed Poisson distribution with parameters " << Uw << " (wounded) and " << Ubin << " (binary)" << endl;};
if(MODEL==2){cout << "overlayed Gamma distribution with parameters " << Uw << " (wounded) and " << Ubin << " (binary)" << endl;};
#if(_gauss_!=1)
   cout << "hard sphere NN collision profile" << endl;
#else
   cout << "Gaussian NN collision profile, Ga=" << GA << endl;
#endif
cout << "window: b_min=" << BMIN << "fm, b_max=" << BMAX << "fm";
if(W1!=1000 || W0!=2){cout << ", Nw_min=" << W0 << ", Nw_max=" << W1;};
if(RDS1!=100000 || RDS0!=0){cout << ", RDS_min=" << RDS0 << ", RDS_max=" << RDS1;};
cout << endl;
// if(SHIFT==0){cout << "(fixed-axes coordinates not shifted to the c.m. frame of the fireball)" << endl;};
if(SHIFT==1){cout << "(fixed-axes coordinates shifted to the c.m. frame of the fireball)" << endl;};
if(CD>0.0){
	if (RET==1){
	   cout << "return-to-beginning algorithm (slow, recommended to use RET=0)" << endl;} 
	else {cout << "fix-last algorithm" << endl;};
}; 
if(DW>0.0 || DBIN >0.0) {cout << "source dispersion parameter: wounded=" << DW << "fm, binary=" << DBIN << "fm";}; 
cout << endl; 
if(FULL){cout << "full event tree generated (THIS GENERATES A LOT OF DATA, set FULL=0 to switch off)" << endl;};
};



//! reset the counters used to store physical quantities in the event
void reset_counters(
                   
                   ){
estd.reset(); epart.reset(); estd3.reset(); epart3.reset(); estd4.reset(); epart4.reset(); estd5.reset(); epart5.reset(); 
estd6.reset(); epart6.reset(); nwounded.reset(); nbinary.reset(); nhot.reset(); nweight.reset(); 
                     };

//! print epilog to the output
void epilog(

           ){
// output  basic results to the console
      cout << "Some quantities for the specified b, N_w, and RDS window " << endl 
           << "(+/- gives the e-by-e standard deviation):" << endl;
      if(BMIN*BMIN<0.0001) {cout << "A+B cross section = " << sitot << "mb";};
      if((NUMA>2) && (BMIN*BMIN<0.0001)) {cout << ", equiv. hard-sphere radius = " << sirad << "fm";};
      cout << endl << "efficiency (accepted/all) = " << 100.*EVENTS/evall << "%" << endl;
      cout << "N_w = " << nwounded.mean()  << "+/-" << sqrt(nwounded.var());
      if(ALPHA>0 || DOBIN==1){cout << ", N_bin = " << nbinary.mean()  << "+/-" << sqrt(nbinary.var());};
      if(nhot.mean()>0.0){cout << ", N_hotspot = " << nhot.mean()  << "+/-" << sqrt(nhot.var());};
      cout << endl; 
      cout << "relative deposited strength (RDS) = " << nweight.mean()  << "+/-" << sqrt(nweight.var()) << endl; 
      cout << "eps_fix_2  (standard)   = " << estd.mean() << "+/-" << sqrt(estd.var()) << endl;
      cout << "eps_var_2 (participant) = " << epart.mean() << "+/-" << sqrt(epart.var()) << endl;
              };



