/******************************************************************************
 *                                                                            *
 *           GLISSANDO - GLauber Initial State Simulation AND mOre...         *
 *                       ver. 1.95, 6 August 2010                             *
 *                                                                            *
 * Authors: Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              *
 *          Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              *
 *          Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                      *
 *                                                                            *
 * For the detailed description of the program and further references         *
 * to the description of the model, please, refer to                          *
 *                                                                            *
 * Computer Physics Communications 180(2009)69, arXiv:0710.5731 [nucl-th]     *
 * accessibile from: http://arxiv.org.abs/0710.5731                           *
 *                                                                            *
 * Implementation of nuclear correlations as in Phys. Rev. C81(2010)064909    *
 *                                                                            *
 * Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      *
 *                                                                            *
 * This code can be freely used and redistributed. However, if you decide to  *
 * make modifications, the authors would appreciate notification.             *
 * Any publication or display of results obtained using this code must        *
 * include a reference to our published paper.                                *
 *****************************************************************************/

#include <math.h>
#include <time.h>
#include <string.h>
#include <iostream>
#include <fstream>

#include <TH1D.h>
#include <TH2D.h>
#include <TFile.h>
#include <TTree.h>
#include <TRandom3.h>
#include "functions.h"

//! parameters and their default values

int EVENTS = 50000,       // number of generated events 
    NBIN   = 40,          // number of bins for histogramming in rho or x and y
    FBIN   = 72,          // number of bins for histogramming in the phi angle
    NUMA   = 197,         // mass number of nucleus A
    NUMB   = 197,         // mass number of nucleus B
    WMIN   = 2,           // minimum number of wounded nucleons to record event
    MODEL  = 0,           // 0 - constant weight, 1 - Poisson, 2 - Gamma
    DOBIN  = 0,           // 1 - count binary collisions even in the pure wounded-nucleon model
    W0     = 2,           // minimum allowed number of wounded nucleons
    W1     = 1000,        // maximum allowed number of wounded nucleons
    SHIFT  = 0,           // 1 - shift the coordinates of the fireball to c.m. in the fixed-axes case, 0 - do not
    RET    = 0,           // 0 - fix-last algorithm, 1 - return-to-beginning algorithm for the nuclear density
    FULL   = 0,           // 1 - generate the full event tree, 0 - do not 
    FILES  = 0;           // 1 - read distributions from files, 1 - do not         

UInt_t ISEED  = 0;       // seed for the random number generator. 
                         // If seed is 0 (default value) a random seed is generated

 float BMIN = 0.,       // minimum impact parameter
       BMAX = 25.,      // maximum impact parameter
       RDS0  = 0,       // minimum allowed RDS
       RDS1  = 100000,  // maximum allowed RDS

/* see the paper for a discussion of the parameters of nuclear distributions */
       RWSA = 6.43,     // Woods-Saxon radius -  gold
       AWSA = 0.45,     // Woods-Saxon radius  - gold
       RWSB = 6.43,     // reduced Woods-Saxon - gold
       AWSB = 0.45,     // Woods-Saxon width  -  gold
       WFA = 0.,        // the w parameter in the Fermi distribution for nucleus A
       WFB = 0.,        // the w parameter in the Fermi distribution for nucleus B

       BTOT = fmax(RWSA,RWSB)+AWSA+AWSB,      // range parameter for histogramming

       SNN   = 42.,           // NN "wounding" cross section in millibarns
       SBIN  = 42.,           // NN binary cross section in millibarns
       ALPHA = 0,             // 0 - wounded, 1 - binary, 0.145 - PHOBOS
       Uw    = 2.,            // Poisson or Gamma parameter for wounded
       Ubin  = 2.,            // Poisson or Gamma parameter for binary
       PI    = 4.*atan(1.),   // pi
       CD    = 0.4,           // closest allowed distance between centers of the nucleons in the nucleus in fm (simulation of repulsion)
       DW    = 0.0,           // dispersion of the location of the source for wounded nucleons (in fm)
       DBIN  = 0.0,           // dispersion of the location of the source for binary collisions (in fm)
       GA    = 0.92;          // Gaussian wounding profile at the origin



//! normalized random number using the built-in Root generator
float los() {return raa.Uniform(0,1);}; 
//! discrete random number 0 or 1
float losi(){if(los() > 0.5){return 1.0;} else {return 0.0;};};

//! generators for the distribution for nuclei A and B

//! Woods-Saxon
#if(_deuteron_!=1)
float rlosA(){int x=0;while(x==0){float rr=3.*RWSA*los();float b=RWSA*RWSA*los();
	       if(b<rr*rr*(1.+WFA*rr*rr/RWSA/RWSA)/(exp((rr-RWSA)/AWSA)+1)){x=1;return rr;};};}; // Woods-Saxon/Fermi
#endif

#if(_deuteron_)
//!generator of the distance between nucleons in the deutron according to the Hulthen distribution
float rlosA(){double const ah=.228;double const bh=1.118;int x=0;while(x==0){float rr=15.*los();
float b=.25*los(); // 0.25 is an upper bound of the probability distribution
 if(b< 2*ah*bh*(ah+bh)*(exp(-2.*ah*rr)+exp(-2.*bh*rr)-2.*exp(-(ah+bh)*rr))/(ah-bh)/(ah-bh)){x=1;return rr;};};}; //Hulthen
#endif

float rlosB(){int x=0;while(x==0){float rr=3.*RWSB*los();float b=RWSB*RWSB*los();
	       if(b<rr*rr*(1.+WFB*rr*rr/RWSA/RWSA)/(exp((rr-RWSB)/AWSB)+1)){x=1;return rr;};};}; //Woods-Saxon/Fermi


//! generator for the Poisson distribution
int ipois(float u){
float Z = exp(-u);
float P=1.;
int i = -1;
do {
i++;
P=P*los();
} while (P>Z);
return i; 
}//ipois

//!generator for the Gamma distribution
//f(x) = 1/gamma(a) * x**(a-1)*exp(-x), x>0
float gamgen(float a){
int k = int(a);
float b = a - k;
float x1 = 0.0;
float u=0.0;
float v=0.0;
float x2=0.0;
float s;

if(k==0) goto blok;
s = 1.0 ;
for (int i=1;i<=k;i++)
s=s*los();
x1 = -log(s);

if (b>1e-6) goto blok;
return x1;
 
blok:x2=-log(los());

bloka:u=pow(los(),1.0/b);
v=u+pow(los(),1.0/(1.0-b));
if (v>1.0) goto bloka;
return (x1+x2*u/v);
}//gamgen


//negative binomial distribution
//int negbin(float m,float v) {
//float p=1.0-(m/v);
//float r=m*(1.0-p)/p;
//float x=gamgen(r)*p/(1.0-p);
//int n=ipois(x);
//return n;
//}//negbin


//! distribution overlayed over the distribution of sources
float dist(int m, float u){
 switch(m) {
 case 0: { return 1; break; }
 case 1: { return ipois(u)/u;  break; } 
 case 2: { return gamgen(u)/u; break; }
}
}

//! arctan with the correct branch
// the angle is computed relative to the y-axis and is in the range [-pi,pi]
float atanm(float x, float y){float phi=atan(x/y);if(y<0){phi=phi+PI;};if(phi>PI){phi=phi-2.*PI;}; return phi;};

//! generator of (x,y,z) coordinates of a point in nucleus A according to the Woods-Saxon distribution
void spgenA(float *x, float *y, float *z)
{float cth, sth, phi, r;
  phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosA(); *x=r*cos(phi)*sth; *y=r*sin(phi)*sth; *z=r*cth;};

//! generator of (x,y,z) coordinates of a point in nucleus B according to the Woods-Saxon distribution
void spgenB(float *x, float *y, float *z)
{float cth, sth, phi, r;
  phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosB(); *x=r*cos(phi)*sth; *y=r*sin(phi)*sth; *z=r*cth;};

//! random shift of the source according to the gaussian distribution
float disp(float w){int x=0; while(x==0){float rr=(6.*los()-3.)*w;float b=los();
	       if(b<exp(-rr*rr/(2*w*w))) {x=1;return rr;};};};
