/******************************************************************************
 *                                                                            *
 *           GLISSANDO - GLauber Initial State Simulation AND mOre...         *
 *                           ver. 1.6                                         *
 *                                                                            *
 * Authors: Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              *
 *          Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              *
 *          Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                      *
 *                                                                            *
 * For the detailed description of the program and further references         *
 * to the description of the model, please, refer to arXiv:0710.5731 [nucl-th]*
 * accessibile from: http://arxiv.org.abs/0710.5731                           *
 *                                                                            *
 * Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      *
 *                                                                            *
 * This code can be freely used and redistributed. However, if you decide to  *
 * make modifications to the code, the authors would appreciate notification. *
 * Any publication or display of results obtained using this code must        *
 * include a reference to arXiv:0710.5731 [nucl-th] or the published          *
 * version of it, when available.                                             *
 *                                                                            *
 *****************************************************************************/

#include <math.h>
#include <cstring>
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <sstream>

#include <TH1F.h>
#include <TH1D.h>
#include <TH2F.h>
#include <TFile.h>
#include <TTree.h>

using namespace std;

char hname[25];

// 1D Lagrange interpolation. 
// Calculation of value of Lagrange polynomial f(x) at given x
double w1D(int n, double dx, double x[4],double y[4]){
int i,j;
double dy,m,l;

dy=0;
for (i=0;i<n; i++) {
 l=1; m=1;
  for (j=0;j<n;j++) 
    if (i!=j) {
     m=m*(x[i]-x[j]);
     l=l*(dx-x[j]);
     }
     dy+=y[i]*l/m;
 }
 return dy;
}

// 2D Lagrange interpolation.
// Calculation of value of Lagrange polynomial (first order)
double w2D(double x,double y,double x0,double y0,double u0,double x1,double y1,double u1,double x2,double y2,double u2){

double m = (x1*y2-x2*y1)-(x0*y2-x2*y0)+(x0*y1-x1*y0);
double w0=u0*((x1*y2-x2*y1+(y1-y2)*x+(x2-x1)*y)/m);
double w1=u1*((x2*y0-x0*y2+(y2-y0)*x+(x0-x2)*y)/m);
double w2=u2*((x0*y1-x1*y0+(y0-y1)*x+(x1-x0)*y)/m);

return w0+w1+w2;
}


// Two nodes. Linear regression method.
// Calculation of f(x)=a+x*b
double lin(double dx,double x1,double y1,double x2,double y2){
double al=2.0*(x1*y1+x2*y2)-((x1+x2)*(y1+y2));
double am=2.0*(x1*x1+x2*x2)-((x1+x2)*(x1+x2));
double a=al/am;
double b=(y1+y2-a*(x1+x2))/2.0;
return a*dx+b;
}

// 1D interpolation
// Determination of the node values from the given histogram.
double inter1D(double dx,TFile & f,char *histname) {

int i,j,ibin;
double x[500],y[500];
double xw[4],yw[4];

for (i=0;i<500;i++){
x[i]=0;
y[i]=0;
}

for (i=0;i<4;i++){
xw[i]=0;
yw[i]=0;
}

TH1F *h = (TH1F*)f.Get(histname); 
TAxis *xaxis = h->GetXaxis();

int nr = xaxis->GetNbins();

for (i=1;i<=nr;i++) {
double s = xaxis->GetBinCenter(i);
x[i-1]=s;
double z = h->GetBinContent(i);
y[i-1]=z;
}

if ((dx<0)||(dx>x[nr-2])){
return -1.;
}

if (dx<=x[1]){ 
double x1=x[0];
double y1=y[0];
double x2=x[1];
double y2=y[1];
return lin(dx,x1,y1,x2,y2);
}

for (i=1;i<=nr-1;i++)
if ((dx>x[i]) && (dx<=x[i+1])){
  ibin=i;
  break;
}

 j=0;
for (i=ibin-1;i<=ibin+2;i++) {
 xw[j]=x[i];
 yw[j]=y[i];
 j++;
 }
return w1D(4,dx,xw,yw);
}

// 2D interpolation.
// Determination of the node values from the given histogram.
double inter2D(double dx, double dy, TFile & f,char *histname) {

double x[500],y[500],z[500][500];

int i,j,ibin,jbin;

for (i=0;i<500;i++){
x[i]=0;
y[i]=0;
}

for (i=0;i<500;i++){
for (j=0;j<500;j++){
z[i][j]=0;
}
}

TH1F *h = (TH1F*)f.Get(histname); 

TAxis *xaxis = h->GetXaxis();
TAxis *yaxis = h->GetYaxis();

int nrx = xaxis->GetNbins();
int nry = yaxis->GetNbins();

for (i=1;i<=nrx;i++) {
double sx = xaxis->GetBinCenter(i);
x[i-1]=sx;
}
for (i=1;i<=nry;i++){
double sy = yaxis->GetBinCenter(i);
y[i-1]=sy;
}
for (i=1;i<=nrx;i++){
for (j=1;j<=nry;j++){
double cont = h->GetBinContent(i,j);
z[i-1][j-1]=cont;
}
}

if ((dx<x[0])||(dx>x[nrx-1])){
return -1.;
}
if ((dy<y[0])||(dy>y[nrx-1])){
return -1.;
}

for (i=0;i<=nrx-1;i++){
for (j=0;j<nry-1;j++){
if ((dx>=x[i]) && (dx<x[i+1])){
if ((dy>=y[j]) && (dy<y[j+1])){
  ibin=i;
  jbin=j;
  break;
}
}
}
}

double x1=x[ibin];
double y1=y[jbin];
double x2=x[ibin+1];
double y2=y[jbin+1];

double fun=lin(dx,x1,y1,x2,y2);

if (fun>=dy){ 
//cout << " bottom " << endl;
return w2D(dx,dy,x1,y1,z[ibin][jbin],x2,y1,z[ibin+1][jbin],x2,y2,z[ibin+1][jbin+1]);
}
else{
//cout << " top " << endl;
return w2D(dx,dy,x1,y1,z[ibin][jbin],x2,y2,z[ibin+1][jbin+1],x1,y2,z[ibin][jbin+1]);
}

}

// 1D interpolation part of code
void d1(TFile & f){
double x;
system("clear");
cout << "1D Interpolation" <<endl;
cout << endl;
cout << "Give name of 1D histogram" <<endl;
cin>> hname;
cout << "Give value of rho: ";
cin >> x;
double w = inter1D(x,f,hname);
if (w<=-1.) {
 cout <<"Out of range"<<endl;
 } else {
cout << "Interpolated value at " << x << " equals " << w <<endl;
}
}

// 2D nterpolation part of code
void d2(TFile & f){
double x,y;
system("clear");
cout << "2D Interpolation" <<endl;
cout << endl;
cout << "Give name of 2D histogram" <<endl;
cin>> hname;
cout << "Give value of rho_x: ";
 cin >> x;
cout << "and"<<endl;
cout << " rho_y: ";
cin >> y;
double w = inter2D(x,y,f,hname);
if (w<=-1.) {
 cout <<"Out of range"<<endl;
 } else {
cout << "Interpolated value at " << x <<", "<< y << " equals " <<w <<endl;
}
}

// Starting menu
void start(){
cout << "INTERPOLATION"<< endl;
cout << "ver 1.0" << endl;
cout << endl;

cout << "1 - One Dimensional Interpolation" << endl;
cout << "2 - Two Dimensional Interpolation" << endl;
cout << endl;
cout << "Type 1 or 2 (any other - Exit)"<< endl;
}

// Determination of value of mean weight 
// from the npa branch stored in GLISSANDO Root file
double mean_weight(TFile & f){

float npa;

TTree *ev = (TTree*)f.Get("events");
ev->SetBranchAddress("npa",&npa);

double max=0;
int nentries = (int)ev->GetEntries();
for (int i=0; i<nentries; i++) {
ev->GetEntry(i);
if (npa>max) max=npa;
}

TH1F *hnpa = new TH1F("hnpa","weight distribution",(int)max,0,max);

for (int i=0; i<nentries; i++) {
ev->GetEntry(i);
hnpa->Fill(npa);
}
double sw=hnpa->GetMean();
return sw;
}

// Calculation of value of integral: 2 Pi rho f0(rho) drho 
// from the given (hopefully c0hp) histogram stored in GLISSANDO Root file.
// Rectangle method.
double integral1D(double nbc,TFile & f, char *histname){

TH1F *h = (TH1F*)f.Get(histname);
TAxis *xaxis = h->GetXaxis();
int nr = xaxis->GetNbins();
double width=xaxis->GetBinWidth(1);
double start=xaxis->GetBinCenter(1);
double end=xaxis->GetBinCenter(nr-2);

double max=width*nr;
double w=max/nbc;
double xcalk=start;
double c=0;
while (xcalk<end){
 c=c+(xcalk*w*inter1D(xcalk,f,histname));
 xcalk+=w;
 }
 c=2.0*3.1415926*c;

return c;
}

// Calculation of value of integral: f(rho_x,rho_y) drho_x drho_y
// from the given (hopefully xyhist) histogram stord in GLISSANDO Root file
// Rectanle method
double integral2D(double nbc, TFile & f, char *histname){

TH1F *h = (TH1F*)f.Get(histname);
TAxis *xaxis = h->GetXaxis();
TAxis *yaxis = h->GetYaxis();
int nrx = xaxis->GetNbins();
int nry = yaxis->GetNbins();

double widthx=xaxis->GetBinWidth(1);
double widthy=yaxis->GetBinWidth(1);

double startx=xaxis->GetBinCenter(2);
double endx=xaxis->GetBinCenter(nrx-1);
double starty=yaxis->GetBinCenter(2);
double endy=yaxis->GetBinCenter(nry-1);

double maxx=widthx*nrx;
double maxy=widthy*nry;

double wx=maxx/nbc;
double wy=maxy/nbc;

double xcalk=startx;
double ycalk=starty;

double c1=0;
while (ycalk<endy){
 xcalk=startx;
 while (xcalk<endx) {
 c1=c1+(wx*wy*inter2D(xcalk,ycalk,f,histname));
 xcalk+=wx;
 }
 ycalk+=wy;
 }

return c1;
}

int main(int argc, char **argv){

double nbc=100;
double nbc2D=50;

// Default file name
TString inpfile("glissando.root");
if (argc>1) inpfile = argv[1];
TFile f(inpfile);

char t, more='y';

cout << "Test of normalization"<<endl;
cout <<"Mean RDS= "<<mean_weight(f)<<endl;
cout <<"Integral over 2 Pi rho f0(rho) drho= "<<integral1D(nbc,f,"c0hp")<<endl;
cout <<"Integral over (xyhist) f(rho_x,rho_y) drho_x drho_y= "<<integral2D(nbc2D,f,"xyhist")<<endl;

start();
cin>>t;
if (t=='1'){
while ((more=='y')||(more=='Y')) {
d1(f);
cout << "Once more?(y/n)"<<endl;
cin>>more;
}
} else 
if (t=='2'){
while ((more=='y')||(more=='Y')){
d2(f);
cout << "Once more?(y/n)"<<endl;
cin>>more;
}
}
else {
cout  <<endl;
cout <<"You put number different than \"1\" or \"2\" - exitting"<<endl;
}
f.Close("R");
return 0;
}
