/******************************************************************************
 *                                                                            *
 *           GLISSANDO - GLauber Initial State Simulation AND mOre...         *
 *                           ver. 1.61                                        *
 *                                                                            *
 * Authors: Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              *
 *          Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              *
 *          Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                      *
 *                                                                            *
 * For the detailed description of the program and further references         *
 * to the description of the model, please, refer to arXiv:0710.5731 [nucl-th]*
 * accessibile from: http://arxiv.org.abs/0710.5731                           *
 *                                                                            *
 * Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      *
 *                                                                            *
 * This code can be freely used and redistributed. However, if you decide to  *
 * make modifications to the code, the authors would appreciate notification. *
 * Any publication or display of results obtained using this code must        *
 * include a reference to arXiv:0710.5731 [nucl-th] or the published          *
 * version of it, when available.                                             *
 *                                                                            *
 *****************************************************************************/

#include <TRandom3.h>

extern TRandom3 raa;

extern int EVENTS,          // number of generated events 
           NBIN,            // number of bins for histogramming in rho or x and y
           FBIN,            // number of bins for histogramming in the phi angle
           NUMA,            // mass number of nucleus A
           NUMB,            // mass number of nucleus B
           WMIN,            // minimum number of wounded nucleons to record event
           MODEL,           // switch for the superimposed distribution: 0 - uniform, 1 - Poisson, 2 - Gamma
           DOBIN,           // 1 - count binary collisions even in the pure wounded-nucleon model
           W0,              // minimum allowed number of wounded nucleons
           W1,              // maximum allowed number of wounded nucleons
           SHIFT,           // 1 - shift the coordinates of the fireball to c.m. in the fixed-axes case, 0 - do not
           RET,             // 0 - fix-last algorithm, 1 - return-to-beginning algorithm for the nuclear density
           FULL;            // 1 - generate the full event tree, 0 - do not


extern UInt_t ISEED;        // seed for the ROOT random number generator, if 0 - random seed generated

extern float BMIN,             // minimal impact parameter
             BMAX,             // maximal impact parameter
             RDS0,             // minimum allowed RDS
             RDS1,             // maximum allowed RDS

             BTOT,             // range parameter for histogramming
             RWSA,             // Woods-Saxon radius - standard case gold
             AWSA,             // Woods-Saxon width  - standard case gold
             RWSB,             // Woods-Saxon radius - standard case gold
             AWSB,             // Woods-Saxon width  - standard case gold
             WFA,              // the w parameter in the Fermi distribution for nucleus A
             WFB,              // the w parameter in the Fermi distribution for nucleus B

             BTOT,             // size parameter for histogramming

             SNN,              // NN "wounding" cross section in millibarns
             SBIN,             // NN binary cross section in millibarns
             ALPHA,            // 0 - wounded, 1 - binary, 0.145 - mixed (PHOBOS)
             Uw,               // Poisson or Gamma parameters for wounded
             Ubin,             // and binary
             PI,               // pi
             CD,               // closest allowed distance between nucleons in the nucleus in fm (simulation of repulsion)
             DW,               // dispersion of the location of the source for wounded nucleons (in fm)
             DBIN,             // dispersion of the location of the source for binary collisions (in fm)
             GA;               // Gaussian wounding profile at the origin


//! normalized random number using the built-in Root generator
float los() ; 

//! discrete random number 0 or 1
float losi() ;

//! generator for the Woods-Saxon distribution - nucleus A
float rlosA() ;

//! generator for the Woods-Saxon distribution - nucleus B
float rlosB() ;

//! generator for the Poisson distribution
int ipois(float u) ;

//!generator for the Gamma distribution
// f(x) = 1/gamma(a) * x**(a-1)*exp(-x), x>0
float gamgen(float a) ;

//negative binomial distribution
//int negbin(float m,float v) ;

//! distribution overlayed over the number of sources
float dist(int m, float u) ;

//! arctan with correct branch - phi relative to y axis in the range [-pi,pi]
float atanm(float x, float y) ;

//! generator of (x,y,z) coordinates of a point in nuclei A and B with a Woods-Saxon density profile
void spgenA(float *x, float *y, float *z) ;
void spgenB(float *x, float *y, float *z) ;

//! functions for random shifting the source location
float disp(float x);

//! simple counting class
class counter {
private:
    int count;
    long double value;
public:
    void reset(){count=0;value=0;};
    void add(long double s){count++; value=value+s;};
    int getN(){return count;};
    long double get(){return value;};
    long double mean(){return value/count;};};

//! counting class with variance
class counter2 {
private:
  int count;
  long double  value, value2;
public:
    void reset(){count=0;value=0;value2=0;};
    void add(long double s){count++; value=value+s; value2=value2+s*s;};
    int getN(){return count;};
    long double get(){return value;};
    long double get2(){return value2;};
    long double mean(){return value/count;};
    long double var(){return value2/(count-1)-value/count*value/(count-1);}
    long double vara(){return value2/count-value/count*value/count;};};

//! weighted counting class
class counterw {
private:
  long double count, value;
public:
    void reset(){count=0;value=0;};
    void add(long double s, long double w){count=count+w; value=value+s*w;};
    long double getw(){return count;};
    long double get(){return value;};
    long double mean(){return value/count;};};

//! weighted counting class with variance
class counterw2 {
private:
  long double count, value, value2;
public:
    void reset(){count=0;value=0;value2=0;};
    void add(long double s, long double w){count=count+w; value=value+s*w; value2=value2+s*s*w;};
    long double getw(){return count;};
    long double get(){return value;};
    long double get2(){return value2;};
    long double mean(){return value/count;};
    long double vara(){return value2/count-value/count*value/count;};};
