/******************************************************************************
 *                                                                            *
 *           GLISSANDO - GLauber Initial State Simulation AND mOre...         *
 *                           ver. 1.5                                         *
 *                                                                            *
 * Authors: Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              *
 *          Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              *
 *          Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                      *
 *                                                                            *
 * For the detailed description of the program and further references         *
 * to the description of the model, please, refer to arXiv:0710.5731 [nucl-th]*
 * accessibile from: http://arxiv.org.abs/0710.5731                           *
 *                                                                            *
 * Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      *
 *                                                                            *
 * This code can be freely used and redistributed. However, if you decide to  *
 * make modifications to the code, the authors would appreciate notification. *
 * Any publication or display of results obtained using this code must        *
 * include a reference to arXiv:0710.5731 [nucl-th] or the published          *
 * version of it, when available.                                             *
 *                                                                            *
 *****************************************************************************/

//! the units for all dimentionful quantities are fm

float ver=1.5; //! version of the code

#include <math.h>
#include <time.h>
#include <string.h>
#include <iostream>
#include <fstream>

#include <TH1D.h>
#include <TH2D.h>
#include <TFile.h>
#include <TTree.h>
#include <TRandom3.h>
#include "functions.h"

TRandom3 raa; //! ROOT random number generator 

using namespace std;

int main(int argc, char **argv) {

/****************************************************************************************************************/
//! declarations, initialization, input, etc.
/****************************************************************************************************************/

//! header
cout << endl << "*******************************************************" << endl;
cout << "GLISSANDO ver. " << ver << " ( http://arxiv.org.abs/0710.5731 )" << endl; //! title and reference
cout << "                tested with ROOT ver. 5.08--5.17" << endl;
cout << "********************************************************" << endl;

time_t rawtime; 
struct tm * timeinfo;
time ( &rawtime ); 
timeinfo = localtime ( &rawtime ); 
int ti=int(rawtime);
cout << "Start: " << asctime (timeinfo); //! time stamp
cout << "--------------------------------------" << endl;

#if(_proton_!=1 && _deuteron_!=1) 
cout << "  nucleus - nucleus collisions" << endl; 
#endif
#if(_proton_) 
cout << "   proton - nucleus collisions" << endl; 
#endif
#if(_deuteron_) 
cout << "  deuteron - nucleus collisions" << endl; 
#endif
cout << "--------------------------------------" << endl;
#if(_profile_) 
cout << "DEBUG: generation of the nuclear profile" << endl; 
cout << "--------------------------------------" << endl;
#endif
#if(_weight_) 
cout << "DEBUG: generation of the superimposed distribution" << endl; 
cout << "--------------------------------------" << endl;
#endif

//! names of input parameters
char t[20];
char s1[]={"EVENTS"},
     s2[]={"NBIN"},
     s3[]={"NUMA"},
     s4[]={"NUMB"},
     s5[]={"WMIN"},
     s6[]={"BMIN"},
     s7[]={"BMAX"},
     s8[]={"RWSA"},
     s9[]={"AWSA"},
     s10[]={"RWSB"},
     s11[]={"AWSB"},
     s12[]={"SNN"},
     s13[]={"SBIN"},
     s14[]={"ALPHA"},
     s15[]={"Uw"},
     s16[]={"Ubin"},
     s17[]={"CD"},
     s18[]={"MODEL"},
     s19[]={"ISEED"},
     s20[]={"BTOT"},
     s21[]={"W0"},
     s22[]={"W1"},
     s23[]={"RDS0"},
     s24[]={"RDS1"},
     s25[]={"SHIFT"},
     s26[]={"RET"},
     s27[]={"DW"},
     s28[]={"DBIN"},
     s29[]={"WF"},
     s30[]={"FULL"},
     s31[]={"FBIN"},
     s32[]={"DOBIN"};

int GAUSS; //! shape of the wounding profile
	#if(_gauss_!=1)
             GAUSS=0;
	#endif
	#if(_gauss_)
	     GAUSS=1;
	#endif

//! the input file can be passed as the first command-line argument
TString inpfile("input.dat");
if (argc>1) inpfile = argv[1];
FILE *in; 
in = fopen(inpfile,"rt");
cout << "parameters reset in input file " << inpfile << " :" << endl;

//! scan the input file for the parameters reset from the default values
int fBTOT=0; double v; 
while (!feof(in)){
  fscanf(in,"%s %lf",&t,&v);
  if ((!feof(in)) && (t[0]!='#') && (t[0]!='I')) cout << t << "\t"<< v <<endl;
  if ((!feof(in)) && (t[0]!='#') && (t[0]=='I')) cout << t << "\t"<< UInt_t(v) <<endl;
  if (!strcmp(t,s1)) {EVENTS=int(v);goto dalej;} // number of events
  if (!strcmp(t,s2)) {NBIN=int(v);goto dalej;}   // number of bins in histograms
  if (!strcmp(t,s3)) {NUMA=int(v);goto dalej;}   // mass number of nucleus A
  if (!strcmp(t,s4)) {NUMB=int(v);goto dalej;}   // mass number of nucleus B
  if (!strcmp(t,s5)) {WMIN=int(v);goto dalej;}   // minimum number of wounded nucleons to record event
  if (!strcmp(t,s6)) {BMIN=v;goto dalej;}        // minimum impact parameter
  if (!strcmp(t,s7)) {BMAX=v;goto dalej;}        // maximum impact parameter
  if (!strcmp(t,s8)) {RWSA=v;goto dalej;}        // Woods-Saxon radius on nucleus A
  if (!strcmp(t,s9)) {AWSA=v;goto dalej;}        // Wood-Saxon surface thickness parameter for nucleus A
  if (!strcmp(t,s10)) {RWSB=v;goto dalej;}       // Woods-Saxon radius on nucleus B
  if (!strcmp(t,s11)) {AWSB=v;goto dalej;}       // Wood-Saxon surface thickness parameter for nucleus B
  if (!strcmp(t,s12)) {SNN=v;goto dalej;}        // wounding cross section
  if (!strcmp(t,s13)) {SBIN=v;goto dalej;}       // binary cross section
  if (!strcmp(t,s14)) {ALPHA=v;goto dalej;}      // the mixing parameter
  if (!strcmp(t,s15)) {Uw=v;goto dalej;}         // parameter for the distribution superimposed over the wounded nucleons
  if (!strcmp(t,s16)) {Ubin=v;goto dalej;}       // parameter for the distribution superimposed over the binary collisions
  if (!strcmp(t,s17)) {CD=v;goto dalej;}         // the expulsion distance
  if (!strcmp(t,s18)) {MODEL=int(v);goto dalej;}     // switch for the superiposed distribution
  if (!strcmp(t,s19)) {ISEED=UInt_t(v);goto dalej;}  // seed for the random-number generator
  if (!strcmp(t,s20)) {fBTOT=1;BTOT=v;goto dalej;}   // range for histograms
  if (!strcmp(t,s21)) {W0=int(v);goto dalej;}     // minimum number of wounded nucleons
  if (!strcmp(t,s22)) {W1=int(v);goto dalej;}     // maximum number of wounded nucleons
  if (!strcmp(t,s23)) {RDS0=v;goto dalej;}        // minimum RDS (relative deposited stregth, see the paper)
  if (!strcmp(t,s24)) {RDS1=v;goto dalej;}        // maximum RDS
  if (!strcmp(t,s25)) {SHIFT=int(v);goto dalej;}  // parameter controling the shift of the fixed-axes distributions to c.m. frame 
  if (!strcmp(t,s26)) {RET=int(v);goto dalej;}    // parameter controling nuclear density generation
  if (!strcmp(t,s27)) {DW=v;goto dalej;}          // width of the distribution of displacement of the location of source for wounded nucleons
  if (!strcmp(t,s28)) {DBIN=v;goto dalej;}        // width of the distribution of displacement of the location of source for binary collisions
  if (!strcmp(t,s29)) {WF=v;goto dalej;}          // the w parameter of the Fermi distribution
  if (!strcmp(t,s30)) {FULL=int(v);goto dalej;}   // parameter controling generation of the full event tree 
  if (!strcmp(t,s31)) {FBIN=int(v);goto dalej;}   // number of histogram bins in the the phi angle
  if (!strcmp(t,s32)) {DOBIN=int(v);goto dalej;}  // if 1, generate binary collision also for the pure wounded-nucleon model
 dalej:;
}

//! correct wrong input
 if ((MODEL!=0) && (MODEL!=1) && (MODEL!=2)) {MODEL=0; cout << "correction: MODEL=" << MODEL << endl;};
 if (BMIN<0) {BMIN=0; cout << "correction: BMIN=" << BMIN << endl;};
 if (BMAX<BMIN) {BMAX=BMIN; cout << "correction: BMAX=" << BMAX << endl;};
 if ((ALPHA<0)||(ALPHA>1)) {ALPHA=0; cout << "correction: ALPHA=" << ALPHA << endl;};
 if ((SHIFT!=0) && (SHIFT!=1)) {SHIFT=0; cout << "correction: SHIFT=" << SHIFT << endl;};
 if (W0<WMIN) {W0=WMIN; cout << "correction: W0=" << W0 << endl;};
 if (W1<W0) {W1=W0; cout << "correction: W1=" << W1 << endl;};
 if (RDS1<RDS0) {RDS1=RDS0+1; cout << "correction: RDS1=" << RDS1 << endl;};
 if (Uw<0) {Uw=1; cout << "correction: Uw=" << 1 << endl;};
 if (Ubin<0) {Ubin=1; cout << "correction: Ubin=" << 1 << endl;};
 if (CD<0) {CD=-CD; cout << "correction: CD=" << CD << endl;};
//! see the paper for the discussion of parametrizatiations of nuclear distributions
if ((RWSA<=0) && ((CD-0.8)*(CD-0.8)<0.001) && (RET==0)) {
   RWSA=1.103*pow(NUMA,1./3.)-0.55*pow(NUMA,-1./3.); AWSA=0.455;
   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.455fm (see the paper)" << endl;};
if ((RWSB<=0) && ((CD-0.8)*(CD-0.8)<0.001) && (RET==0)) {
   RWSB=1.103*pow(NUMB,1./3.)-0.55*pow(NUMB,-1./3.); AWSB=0.455;
   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSB=0.455fm (see the paper)" << endl;};
if ((RWSA<=0) && ((CD-0.4)*(CD-0.4)<0.001) && (RET==0)) {
   RWSA=1.113*pow(NUMA,1./3.)-0.277*pow(NUMA,-1./3.); AWSA=0.45;
   cout << "Woods-Saxon parameers: RWSA=" << RWSA << "fm, AWSA=0.45fm (see the paper)" << endl;};
if ((RWSB<=0) && ((CD-0.4)*(CD-0.4)<0.001) && (RET==0)) {
   RWSB=1.113*pow(NUMB,1./3.)-0.277*pow(NUMB,-1./3.); AWSB=0.45; 
   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSA=0.45fm (see the paper)" << endl;};
if ((RWSA<=0) && (CD*CD<0.001) && (RET==0)) {
   RWSA=1.114*pow(NUMA,1./3.)-0.246*pow(NUMA,-1./3.); AWSA=0.45;
   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.45fm (see the paper)" << endl;};
if ((RWSB<=0) && (CD*CD<0.001) && (RET==0)) {
   RWSB=1.114*pow(NUMB,1./3.)-0.246*pow(NUMB,-1./3.); AWSB=0.45; 
   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSA=0.45fm (see the paper)" << endl;};
if ((RWSB<=0 || RWSB<=0)) {cout << "Correct radii!"<< endl; return 0;};

//! if (NUMA!=NUMB) {SHIFT=1; cout << "SHIFT=1 for collisions of different nuclei" << endl;};
//! set the range for the histograms
if (fBTOT!=1) BTOT = fmax(RWSA,RWSB)+AWSA+AWSB;

//! fix the mass number for the proton and deuteron cases
#if(_proton_) 
NUMA=1; 
if(NUMA!=1){cout << "reset: NUMA=1 (proton)" << endl;};
#endif
#if(_deuteron_) 
if(NUMA!=2){NUMA=2; cout << "reset: NUMA=2 (deuteron)" << endl;};
#endif

raa.SetSeed(ISEED); //! seed the Root random generator
UInt_t ISEED1 = ISEED;
UInt_t ISEED = raa.GetSeed();

//! definition of various counters
counter2  estd,   //! epsilon standard (fixed-axes) = <r^2 cos(2 phi)>/<r^2>
epart,            //! epsilon participant (variable-axes) = <r^2 cos(2 phi*)>/<r^2>
estd3,            //! <r^2 cos(3 phi)>/<r^2>
epart3,           //! <r^2 cos(3 phi*)>/<r^2>
estd4, epart4, estd5, epart5, estd6, epart6,    //etc.
nwounded,         //! wounded nucleons
nbinary,          //! binary collisions
nhot,             //! hot-spots
nweight,          //! source RDS
xxc,              //! x center-of-mass coordinate
yyc;              //! y center-of-mass coordinate

//! r^2=x^2+y^2
counterw  c2c0,   //! r^2
c2c2,             //! r^2 cos(2 phi)
c2s2,             //! r^2 sin(2 phi)
c2c3,             //! r^2 cos(3 phi)
c2s3, c2c4, c2s4, c2c5, c2s5, c2c6, c2s6,         // etc.
//! rs^2 = xs^2 +ys^2, xs and ys and in the frame of the center-of-mass of the fireball
c2c0r,            //! rs^2 for shifted coordinates
c2c2r,            //! rs^2 cos(2 (phi-phi*))
c2s2r,            //! rs^2 sin(2 (phi-phi*))
c2c3r,c2s3r,c2c4r,c2s4r,c2c5r,c2s5r,c2c6r,c2s6r;  //etc.

counterw2 xc, yc, zc;  //! x, y, and z

//! resetting the counters
estd.reset(); epart.reset(); estd3.reset(); epart3.reset(); estd4.reset(); epart4.reset(); estd5.reset(); epart5.reset(); estd6.reset(); epart6.reset(); nwounded.reset(); nbinary.reset(); nhot.reset(); nweight.reset(); xxc.reset(); yyc.reset();

int i, j, k,  //! loop counters
evall=0,      //! all attempted A+B collisions (for evaluation of the total cross section)
fA,           //! counts nucleons in the nucleus A or B
good,         //! =1 if location of the nucleon is good (not closer than the expulsion distance from others), =0 otherwise
nbin,         //! binary collisions
w,            //! flag for a wounded nucleon / binary collision
nwA,          //! wounded nucleons in A
nwB,          //! wounded nucleons in B
nhotspot;     //! hot-spots

float x, y, z, //! coordinates of nucleons
d,             //! the wounding distance
dbin,          //! the binary-collision distance 
rd2;           //! distance squared between the centers of nucleons

float nA[208][3];  //! coordinates of nucleons in A
float nB[208][3];  //! coordinates of nucleons in B
float wx[43680];   //! x coordinates of sources (size = 208*208+2*208 - binary+wounded, good up to 208 Pb + 208 Pb colisions)
float wy[43680];   //! y coordinates of sources
float wa[43680];   //! RDS of sources
int wc;            //! number of sources (wounded + binary) in the given event
int nzw;           //! number of sources in the given event with nonzero RDS
int kk;            //! number of the event

//! structure for output of the full event
typedef struct {Float_t X,Y,W; UInt_t KK;} SOURCE;
static SOURCE tSource;

Float_t b, //! impact parameter 
sitot,     //! the total A+B cross section
rwA,       //! wounded nucleons in A
rwB,       //! wounded nucleons in B
rwAB,      //! all wounded nucleons
rbin,      //! binary collisions
rhotspot,  //! hot-spots
rpa,       //! source RDS
r,         //! radius
r2,        //! radius squared 
mm,        //! distance between nucleons squared 
//
// Not all moment listed below are computed. If they are of interest to the user, he should uncomment 
// approprate lines in following parts of the code.
//
es,        //! epsilon standard (fixed-axes), <r^2 cos(2 phi)>/<r^2>
ess,       //! <r^2 sin(2 phi)>/<r^2>
ep,        //! epsilon participant (variable-axes) = <r^2 cos(2 phi*)>/<r^2>
eps,       //! <r^2 sin(2 phi*)>/<r^2> 
es3,       //! <r^2 cos(3 phi)>/<r^2>
es3s,      //! <r^2 sin(3 phi)>/<r^2>
ep3,       //! <r^2 cos(3 phi*)>/<r^2>
ep3s,      //! <r^2 sin(3 phi*)>/<r^2>
es4, es4s, ep4, ep4s, es5, es5s, ep5, ep5s, es6, es6s, ep6, ep6s, // etc.
//
ww,        //! source RDS
phi,       //! polar angle relative to the y-axes
phirot,    //! roration angle phi*
xx,        //! x center-of-mass coordinate
yy,        //! y center-of-mass coordinate
xs,        //! x shifted by xx
ys,        //! y shifted by yy
phis,      //! polar angle for the shifted coordinates
r2s,       //! xs^2+ys^2
wd,        //! superimposed weight, generated by the superposition distribution 
xeps, xseps, xepp, xsepp;


//! the Root output file with the results can be passed as the second command-line argument
TFile *fout;
TString outfile("glissando.root");
if (argc>2) outfile = argv[2];
cout << endl << "generates output file " << outfile;
fout = new TFile(outfile,"RECREATE"); fout->cd();

//! tree for storing the parameters of the run
TTree *param;
param = new TTree("param","param tree");
        param->Branch("EVENTS",&EVENTS,"EVENTS/I");
        param->Branch("NBIN",&NBIN,"NBIN/I");
        param->Branch("FBIN",&FBIN,"FBIN/I");
        param->Branch("NUMA",&NUMA,"NUMA/I");
        param->Branch("NUMB",&NUMB,"NUMB/I");
        param->Branch("WMIN",&WMIN,"WMIN/I");
        param->Branch("MODEL",&MODEL,"MODEL/I");
        param->Branch("W0",&W0,"W0/I");
        param->Branch("W1",&W1,"W1/I");
        param->Branch("RDS0",&RDS0,"RDS0/F");
        param->Branch("RDS1",&RDS1,"RDS1/F");
        param->Branch("ISEED",&ISEED,"ISEED/i");
        param->Branch("BMIN",&BMIN,"BMIN/F");
        param->Branch("BMAX",&BMAX,"BMAX/F");
        param->Branch("BTOT",&BTOT,"BTOT/F");
        param->Branch("RWSA",&RWSA,"RWSA/F");
        param->Branch("AWSA",&AWSA,"AWSA/F");
        param->Branch("RWSB",&RWSB,"RWSB/F");
        param->Branch("AWSB",&AWSB,"AWSB/F");
        param->Branch("WF",&WF,"WF/F");
        param->Branch("SNN",&SNN,"SNN/F");
        param->Branch("SBIN",&SBIN,"SBIN/F");
        param->Branch("ALPHA",&ALPHA,"ALPHA/F");
        param->Branch("DOBIN",&DOBIN,"DOBIN/I");
        param->Branch("Uw",&Uw,"Uw/F");
        param->Branch("Ubin",&Ubin,"Ubin/F");
        param->Branch("CD",&CD,"CD/F");
        param->Branch("SHIFT",&SHIFT,"SHIFT/I");
        param->Branch("RET",&RET,"RET/I");      
        param->Branch("DW",&DW,"DW/F");
        param->Branch("DBIN",&DBIN,"DBIN/F");
        param->Branch("GAUSS",&GAUSS,"GAUSS/I");
        param->Branch("ver",&ver,"ver/F");

        param->Fill(); //! fill the parameter tree

TTree *phys; //! for storing the A+B cross section and other physical results
phys = new TTree("phys","physical results");
        phys->Branch("sitot",&sitot,"sitot/F");
        phys->Branch("eps_fixed",&xeps,"seps/F");
        phys->Branch("eps_variable",&xepp,"sepp/F");
        phys->Branch("sigma_eps_fixed",&xseps,"xseps/F");
        phys->Branch("sigma_eps_variable",&xsepp,"xsepp/F");


TTree *tree;  //! stores the basic results
TTree *tree1; //! stores the nuclear density profile of nucleus A and the superimposed weights

TTree *full_event; //! stores the full info on the event (positions and RDS of the sources)
full_event = new TTree("full_event","full event");
	full_event -> Branch("full_source",&tSource,"X:Y:W:KK");

        tree =  new TTree("events","event tree");
        tree1 = new TTree("density","nuclear density tree");
 
        tree->Branch("nwA",&rwA,"nwA");    // wounded nucleons in A
        tree->Branch("nwB",&rwB,"nwB");    // wounded nucleons in B
        tree->Branch("nwAB",&rwAB,"nwAB"); // all wounded nucleons
        tree->Branch("nbin",&rbin,"nbin"); // binary collisions
        tree->Branch("npa",&rpa,"npa");    // source RDS
        tree->Branch("b",&b,"b");          // impact parameter

//! fixed-axes and variable-axes cos and sin moments (some commented out)
        tree->Branch("es",&es,"es");       // < r^2 cos(2 phi) >
        tree->Branch("ep",&ep,"ep");       // < r^2 cos(2 (phi-phi*)) >
//        tree->Branch("es3",&es3,"es3");  // < r^2 cos(3 phi) >
//        tree->Branch("ep3",&ep3,"ep3");  // < r^2 cos(3 (phi-phi*)) >
        tree->Branch("es4",&es4,"es4");    // < r^2 cos(4 phi) >
        tree->Branch("ep4",&ep4,"ep4");    // < r^2 cos(4 (phi-phi*)) >
//        tree->Branch("es5",&es5,"es5");  // < r^2 cos(5 phi) >
//        tree->Branch("ep5",&ep5,"ep5");  // < r^2 cos(5 (phi-phi*)) >
        tree->Branch("es6",&es6,"es6");    // < r^2 cos(6 phi) >
        tree->Branch("ep6",&ep6,"ep6");    // < r^2 cos(6 (phi-phi*)) >

//! fixed-axes and variable-axes sin moments
//        tree->Branch("ess",&ess,"ess");    // < r^2 sin(2 phi) >
//        tree->Branch("eps",&eps,"eps");    // < r^2 sin(2 (phi-phi*)) >
        tree->Branch("es3s",&es3s,"es3s");   // < r^2 sin(3 phi) >
        tree->Branch("ep3s",&ep3s,"ep3s");   // < r^2 sin(3 (phi-phi*)) >
//        tree->Branch("es4s",&es4s,"es4s"); // < r^2 sin(4 phi) >
//        tree->Branch("ep4s",&ep4s,"ep4s"); // < r^2 sin(4 (phi-phi*)) >
//        tree->Branch("es5s",&es5s,"es5s"); // < r^2 sin(5 phi) >
//        tree->Branch("ep5s",&ep5s,"ep5s"); // < r^2 sin(5 (phi-phi*)) >
//        tree->Branch("es6s",&es6s,"es6s"); // < r^2 sin(6 phi) >
//        tree->Branch("ep6s",&ep6s,"ep6s"); // < r^2 sin(6 (phi-phi*)) >

        tree->Branch("phir",&phirot,"phir"); //! the rotationa angle phi*
        tree->Branch("xx",&xx,"x");          //! x c.m. coordinate
        tree->Branch("yy",&yy,"y");          //! y c.m. coordinate

        #if(_profile_)
           tree1->Branch("r",&r,"r"); //! radius in the nuclear profile on nucleus A
        #endif

        #if(_weight_)
           tree1->Branch("wd",&wd,"wd"); //! weight generated by the superposition distribution
        #endif

//! histograms for the 2D profiles
	TH2F *xyhist   = new TH2F("xyhist",  "fixed-axes density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
	TH2F *xyhistr  = new TH2F("xyhistr", "variable-axes density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);

//! histograms for the 1D profiles
//! fixed-axes cos
	TH2F *c0hist  = new TH2F("c0hist", "f_{0}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c2hist  = new TH2F("c2hist", "f_{2}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c3hist  = new TH2F("c3hist", "f_{3}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c4hist  = new TH2F("c4hist", "f_{4}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c5hist  = new TH2F("c5hist", "f_{5}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c6hist  = new TH2F("c6hist", "f_{6}", NBIN, 0., BTOT,FBIN,-PI,PI);
//! variable-axes cos
	TH2F *c0rhist  = new TH2F("c0rhist", "f^{*}_{0}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c2rhist  = new TH2F("c2rhist", "f^{*}_{2}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c3rhist  = new TH2F("c3rhist", "f^{*}_{3}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c4rhist  = new TH2F("c4rhist", "f^{*}_{4}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c5rhist  = new TH2F("c5rhist", "f^{*}_{5}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *c6rhist  = new TH2F("c6rhist", "f^{*}_{6}", NBIN, 0., BTOT,FBIN,-PI,PI);
//! fixed-axes and variable-axes sin
	TH2F *s3hist   = new TH2F("s3hist",      "g_{3}", NBIN, 0., BTOT,FBIN,-PI,PI);
	TH2F *s3rhist  = new TH2F("s3rhist", "g^{*}_{3}", NBIN, 0., BTOT,FBIN,-PI,PI);


//! histograms for the dependence of certain quantities on the number of wounded nucleons
	TH1F *nx = new TH1F("nx", "<x> vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	TH1F *nx2= new TH1F("nx2", "var(x) vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 
	TH1F *ny = new TH1F("ny", "<y> vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	TH1F *ny2= new TH1F("ny2", "var(y) vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 

	TH1F *neps = new TH1F("neps", "#epsilon vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	TH1F *neps2= new TH1F("neps2", "var(#epsilon)/#epsilon^{2} vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 
	TH1F *nepsp = new TH1F("nepsp", "#epsilon^{*} vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	TH1F *nepsp2= new TH1F("nepsp2", "var(#epsilon*)/#epsilon*^{2} vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 
	TH1F *nuni = new TH1F("nuni", "event multiplicity vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5);

	TH1F *nepsb = new TH1F("nepsb", "#epsilon vs. b", 200, BMIN, BMAX);
	TH1F *neps2b= new TH1F("neps2b", "var(#epsilon)/#epsilon^{2} vs. b", 200, BMIN, BMAX); 
	TH1F *nepspb = new TH1F("nepspb", "#epsilon* vs. b", 200, BMIN, BMAX);
	TH1F *nepsp2b= new TH1F("nepsp2b", "var(#epsilon*)/#epsilon*^{2} vs. b", 200, BMIN, BMAX); 
	TH1F *nunib = new TH1F("nunib", "event multiplicity vs. b", 200, BMIN, BMAX);

//! histograms for fluctuations of number of wounded nucleons and RDS
	TH1F *nwei = new TH1F("nwei", "RDS vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);
	TH1F *nwei2= new TH1F("nwei2", "var(RDS)/RDS vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5); 
	TH1F *ntarg = new TH1F("ntarg", "N_{w}^{B} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);
	TH1F *ntarg2= new TH1F("ntarg2", "var(N_{w}^{B})/N_{w}^{B} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5); 
	TH1F *nbinar = new TH1F("nbinar", "N_{bin} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);
	TH1F *nbinar2= new TH1F("nbinar2", "var(N_{bin})/N_{bin} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5); 
	TH1F *nunp = new TH1F("nunp", "event multiplicity vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);


float CD2=CD*CD;
d=sqrt(SNN/PI/10.);      //! the wounding distance, 10 converts to fm^2 from mb
float d2=d*2;
dbin=d;                  //! the binary collision distance (= the wounding distance)
float dbin2=dbin*dbin;
float mbin=SBIN/SNN;     //! ratio for the hot-spot model

int kks=int(fmax(1,fmin(1000,EVENTS/10))); //! for output of the progress of the run

//! echo basic parameters to the console
cout << endl;
if(ISEED1==0){cout << "random ";} else { cout << "fixed ";};
cout << "seed: " << raa.GetSeed()  << ", number of events: " << EVENTS << endl;
#if(_profile_)
   cout << "generates the nuclear density profile for nucleus A" << endl;
#endif
cout << NUMA <<"+" << NUMB;
if(NUMA>2){cout << ", RA="  << RWSA << "fm, aA="<< AWSA << "fm";};
cout << ", RB=" << RWSB << "fm, aB=" << AWSB << "fm, d=" << CD  << "fm";
if(WF*WF>0.00000001) {cout << ", w="<< WF;}
cout << endl; 
if(ALPHA==0){cout << "wounded nucleon model: sig_w=" << SNN << "mb" << endl;};
if(ALPHA==0 && DOBIN==1){cout << "   (binary collisions counted)" << endl;};
if(ALPHA==0 && DOBIN!=1){cout << "   (binary collisions not counted)" << endl;};
if(ALPHA>0 && SBIN>=SNN){cout << "mixed model: sig_w=" << SNN << "mb, sig_bin=" << SBIN << "mb, alpha=" << ALPHA << endl;};
if(ALPHA>0 && SBIN<SNN){cout << "mixed model with hotspots: sig_w=" << SNN << "mb, sig_bin=" << SBIN << "mb, alpha=" << ALPHA << endl;};
if(MODEL==1){cout << "overlayed Poisson distribution with parameters " << Uw << " (wounded) and " << Ubin << " (binary)" << endl;};
if(MODEL==2){cout << "overlayed Gamma distribution with parameters " << Uw << " (wounded) and " << Ubin << " (binary)" << endl;};
#if(_gauss_!=1)
   cout << "hard sphere NN collision profile" << endl;
#endif
#if(_gauss_)
   cout << "Gaussian NN collision profile" << endl;
#endif
cout << "window: b_min=" << BMIN << "fm, b_max=" << BMAX << "fm";
if(W1!=1000 || W0!=2){cout << ", Nw_min=" << W0 << ", Nw_max=" << W1;};
if(RDS1!=100000 || RDS0!=0){cout << ", RDS_min=" << RDS0 << ", RDS_max=" << RDS1;};
cout << endl;
// if(SHIFT==0){cout << "(fixed-axes coordinates not shifted to the c.m. frame of the fireball)" << endl;};
if(SHIFT==1){cout << "(fixed-axes coordinates shifted to the c.m. frame of the fireball)" << endl;};
if(CD>0.0){
	if (RET==1){
	   cout << "return-to-beginning algorithm (slow, recommended to use RET=0)" << endl;} 
	else {cout << "fix-last algorithm" << endl;};
}; 
if(DW>0.0 || DBIN >0.0) {cout << "source dispersion parameter: wounded=" << DW << "fm, binary=" << DBIN << "fm";}; 
cout << endl; 
if(FULL){cout << "full event tree generated (THIS GENERATES A LOT OF DATA, set FULL=0 to switch off)" << endl;};

/****************************************************************************************************************/
//! simulation
/****************************************************************************************************************/

//! loop over events
kk=0; //! counter of events where a collision occurs
for(k=1;kk<EVENTS;k++){

evall++; //! attempted A+B collision (all cases, those where an NN collision occured or not)

//! generation of b with the distribution proportional to b^2 on the range (BMAX, BMIN)
b=sqrt((BMAX*BMAX-BMIN*BMIN)*los()+BMIN*BMIN);

nbin=0; nwA=0; nwB=0; nhotspot=0;  //! reset the counters of binary collisions, wounded nucleons, and hot-spots

xc.reset();yc.reset();zc.reset(); //! reset the center-of-mass counter of nucleus A

#if(_proton_!=1 && _deuteron_!=1) //! nucleus A
//! generation of positions of nucleons in A with the closest distance exclusion, distance squared mm cannot be less than CD*CD
if(RET==1){ //! return-to-beginning algorithm 
	fA=1;
	while(fA<NUMA){
		fA=1;
		xc.reset();
		yc.reset();
		#if(_profile_)
		   zc.reset();
		#endif
		spgenA(&x,&y,&z); //! generate the first nucleon
		nA[0][0]=x;
		nA[0][1]=y;
		nA[0][2]=z;
		xc.add(x,1);
		yc.add(y,1);
		#if(_profile_)
		   zc.add(z,1);
		#endif
		good=1;
		while(good){
			spgenA(&x,&y,&z); //! generate subsequent nucleons
			for(j=1;(j<fA) && good;j++){
				mm=(x-nA[j-1][0])*(x-nA[j-1][0])+(y-nA[j-1][1])*(y-nA[j-1][1])+(z-nA[j-1][2])*(z-nA[j-1][2]);
  				if(mm<CD2){good=0;} //! if closer than the expulsion distance, break
			};
			fA++;
			nA[fA-1][0]=x;
			nA[fA-1][1]=y;
			xc.add(x,1);
			yc.add(y,1);
			nA[fA-1][2]=z;
			#if(_profile_)
			   zc.add(z,1);
			#endif
			};
		};
	}
else { //! fix-last algorithm 
	fA=1;
	spgenA(&x,&y,&z); //! generate the first nucleon
	nA[0][0]=x;
	nA[0][1]=y;
	nA[0][2]=z;
	xc.add(x,1);
	yc.add(y,1);
	zc.add(z,1); 
	while(fA<NUMA){
		spgenA(&x,&y,&z); //! generate subsequent nucleons
		good=1;
		for(j=1;j<fA;j++){
			mm=(x-nA[j-1][0])*(x-nA[j-1][0])+(y-nA[j-1][1])*(y-nA[j-1][1])+(z-nA[j-1][2])*(z-nA[j-1][2]);
 			if(mm<CD2){good=0;};
		};
		if(good==1){
			fA++;
			nA[fA-1][0]=x;
			nA[fA-1][1]=y;
			xc.add(x,1);
			yc.add(y,1);
			nA[fA-1][2]=z;
			#if(_profile_)
			   zc.add(z,1);
			#endif
		};
	};
};

#if(_profile_) //! store the nuclear density profile
for(j=1;j<=fA;j++){
	x=nA[j-1][0]-xc.mean();
	y=nA[j-1][1]-yc.mean();
	z=nA[j-1][2]-zc.mean();
	r=sqrt(x*x+y*y+z*z); 
	tree1->Fill();};
#endif

#endif //! nucleus A


#if(_deuteron_) //! deuteron
//! place the first nucleon at (0,0,0) and the second at random according to the Hulten distribution
x=0;
y=0;
z=0;
nA[0][0]=x;
nA[0][1]=y;
nA[0][2]=z;
xc.add(x,1);
yc.add(y,1); 
spgenA(&x,&y,&z);
nA[1][0]=x;
nA[1][1]=y;
xc.add(x,1);
yc.add(y,1);
nA[1][2]=z;
#if(_profile_) //! store the Hulthen distribution
   r=sqrt(x*x+y*y+z*z);
   tree1->Fill();
#endif

#endif //! deuteron

#if(_proton_) //! proton
//! for the proton just place the nucleon at 0
x=0;
y=0;
z=0;
nA[0][0]=x;
nA[0][1]=y;
nA[0][2]=z;
xc.add(x,1);
yc.add(y,1); 
#endif //! proton

//! shift the coordinates of the nucleons in nucleus A such that the center of mass is at the point (b*NUMB/(NUMA+NUMB),0)
for(j=1;j<=NUMA;j++){
	nA[j-1][0]=nA[j-1][0]-xc.mean()+b*NUMB/(NUMA+NUMB);
	nA[j-1][1]=nA[j-1][1]-yc.mean();};


//! generation of positions of nucleons in B with the closest distance exclusion, distance squared mm cannot be less than CD*CD
//! (same as for the nucleus A above)
xc.reset(); //! reset the center-of-mass counter 
yc.reset(); 
if(RET==1){ //! return-to-beginnng algorithm 
	fA=1;
	while(fA<NUMB){
		fA=1;
		xc.reset();
		yc.reset();
		spgenB(&x,&y,&z);
		nB[0][0]=x;
		nB[0][1]=y;
		nB[0][2]=z;
		xc.add(x,1);
		yc.add(y,1);
		good=1;
		while(good){
			spgenB(&x,&y,&z);
			for(j=1;(j<fA) && good;j++){
				mm=(x-nB[j-1][0])*(x-nB[j-1][0])+(y-nB[j-1][1])*(y-nB[j-1][1])+(z-nB[j-1][2])*(z-nB[j-1][2]);
  				if(mm<CD2){good=0;}
			};
			fA++;
			nB[fA-1][0]=x;
			nB[fA-1][1]=y;
			xc.add(x,1);
			yc.add(y,1);
			nB[fA-1][2]=z;
		};
	};
}
else { //! fix-last algorithm 
	fA=1;
	spgenB(&x,&y,&z);
	nB[0][0]=x;
	nB[0][1]=y;
	nB[0][2]=z;
	xc.add(x,1);
	yc.add(y,1);  
	while(fA<NUMB){
		spgenB(&x,&y,&z);
		good=1;
		for(j=1;j<fA;j++){
			mm=(x-nB[j-1][0])*(x-nB[j-1][0])+(y-nB[j-1][1])*(y-nB[j-1][1])+(z-nB[j-1][2])*(z-nB[j-1][2]);
			if(mm<CD2){good=0;};
		};
		if(good==1){
			fA++;
			nB[fA-1][0]=x;
			nB[fA-1][1]=y;
			nB[fA-1][2]=z;
			xc.add(x,1);
			yc.add(y,1);
		};
	};
};

//! shift the coordinates of the nucleons in nucleus B such that the center of mass is at the point (-b*NUMA/(NUMA+NUMB),0)
for(j=1;j<=NUMB;j++){
	nB[j-1][0]=nB[j-1][0]-xc.mean()-b*NUMA/(NUMA+NUMB);
	nB[j-1][1]=nB[j-1][1]-yc.mean();
};


xc.reset(); yc.reset(); //! reset the center-of-mass counter to be used for the fireball

wc=0;   //! reset number of wounded nucleons
rpa=0.; //! reset the source RDS
nzw=0;  //! reset the number of sources with nonzero RDS

//! count wounded nucleons in A, generate the RDS
for(i=1;i<=NUMA;i++){
	w=0; //! set the "wounded" flag to 0
	for(j=1;j<=NUMB;j++){
	if(w==0){ //! if not wounded, do
		rd2=(nA[i-1][0]-nB[j-1][0])*(nA[i-1][0]-nB[j-1][0])+(nA[i-1][1]-nB[j-1][1])*(nA[i-1][1]-nB[j-1][1]);
		#if(_gauss_!=1) 
		   if(rd2 < d2) //! hard-sphere wounding profile
		#endif
		#if(_gauss_)     
		   if(los() < exp(-rd2/(d2))) //! Gaussian wounding profile
		#endif
	       	    {   w=1; //! set the "wounded" flag to 1
			nwA++;
			wc++;
 			wx[wc-1]=nA[i-1][0];
			if(DW>0){
				wx[wc-1]+=disp(DW); //! displace randomly 
			};
 			wy[wc-1]=nA[i-1][1];
			if(DW>0){
				wy[wc-1]+=disp(DW); //! displace randomly
			};
 			wa[wc-1]=(1-ALPHA)/2.0*dist(MODEL,Uw); //! wounded nucleon RDS
			#if(_weight_)
			   wd=wa[wc-1]/(1-ALPHA)*2.0; //! the weight of the superimposed distribution
			   tree1->Fill();
			#endif
           		rpa=rpa+wa[wc-1];
	        	if(wa[wc-1]>0.0){
				nzw++; //! count wounded if RDS > 0
			}; 
                	xc.add(wx[wc-1],wa[wc-1]); 
			yc.add(wy[wc-1],wa[wc-1]);
		     }
		}
	}
};

//! count wounded nucleons in B, generate the RDS
//! (same as for nucleus A above)
for(j=1;j<=NUMB;j++){
	w=0;
	for(i=1;i<=NUMA;i++){
		if(w==0){
        		rd2=(nA[i-1][0]-nB[j-1][0])*(nA[i-1][0]-nB[j-1][0])+(nA[i-1][1]-nB[j-1][1])*(nA[i-1][1]-nB[j-1][1]);
			#if(_gauss_!=1)
            		   if(rd2 < d2)
			#endif
			#if(_gauss_)
	    		   if(los() < exp(-rd2/(d2)))
			#endif
			     {	w=1;
				nwB++;
				wc++;
			 	wx[wc-1]=nB[j-1][0];
				if(DW>0){
					wx[wc-1]+=disp(DW);
				};
				wy[wc-1]=nB[j-1][1];
				if(DW>0){
					wy[wc-1]+=disp(DW);
				};
				wa[wc-1]=(1-ALPHA)/2.0*dist(MODEL,Uw);
				rpa=rpa+wa[wc-1];  
               			if(wa[wc-1]>0.0){
					nzw++;
				}; 
				xc.add(wx[wc-1],wa[wc-1]); 
				yc.add(wy[wc-1],wa[wc-1]);
			     }
		}
	}
};

//! count binary collisions, generate the RDS (only when ALPHA > 0 or DOBIN=1)
if(ALPHA>0 || DOBIN==1){
	for(i=1;i<=NUMA;i++){
		for(j=1;j<=NUMB;j++){
             		rd2=(nA[i-1][0]-nB[j-1][0])*(nA[i-1][0]-nB[j-1][0])+(nA[i-1][1]-nB[j-1][1])*(nA[i-1][1]-nB[j-1][1]);
			#if(_gauss_!=1)
        		   if(rd2 < dbin2)
			#endif
			#if(_gauss_)
			   if(los() < exp(-rd2/(dbin2)))
			#endif
			   {	nbin++;
				wx[wc+nbin-1]=(nA[i-1][0]+nB[j-1][0])/2;
				if(DBIN>0){
					wx[wc+nbin-1]+=disp(DBIN);
				};
                        	wy[wc+nbin-1]=(nA[i-1][1]+nB[j-1][1])/2;
				if(DBIN>0){
					wy[wc+nbin-1]+=disp(DBIN);
				};
		  		if(los()>mbin){
					wa[wc+nbin-1]=0;
				} 
				else {
					wa[wc+nbin-1]=ALPHA/mbin*dist(MODEL,Ubin); 
					nhotspot++; 
					rpa=rpa+wa[wc+nbin-1];  
					if(wa[wc+nbin-1]>0.0){
						nzw++;
					}; 
					xc.add(wx[wc+nbin-1],wa[wc+nbin-1]);
					yc.add(wy[wc+nbin-1],wa[wc+nbin-1]);
				};
			   };
		};
	};
};

//! Include the event only when the number of wounded nucleons is at least WMIN (=2 by default)
//! and, additionally, lies between W0 and W1 and the relative deposited strength (RDS) lies between RDS0 and RDS1.
//! We also request that the number of sources with non-zero RDS is >= than WMIN (relevant
//! for the Poisson superposition, where RDS can be zero)

if((wc>=WMIN) && (wc >= W0) && (wc <= W1) && (rpa >= RDS0) && (rpa <= RDS1)&& (nzw>=WMIN)){

	kk++; //! count the events in the specified window

	//! xx and yy are the coordinates of the center of mass of the fireball
	xx=xc.mean(); 
	xxc.add(xx); 
	yy=yc.mean(); 
	yyc.add(yy);

	//! compute the fixed-axes moments needed for determination of the rotation angle
        c2c2.reset(); 
	c2s2.reset(); 
	for(j=1;j<=wc+nbin;j++){
		x=wx[j-1]-xx;
		y=wy[j-1]-yy; //shifted coordinates
		phi=atanm(x,y); // use the corrected arctan
		r2=x*x+y*y; 
		ww=wa[j-1];
       		if(r2>0.0){
			c2c2.add(r2*cos(2*phi),ww);
			c2s2.add(r2*sin(2*phi),ww);
		}
       		else {
			c2c2.add(0.0,ww);
			c2s2.add(0.0,ww);
		};
	} //for

	//! determination of the rotation angle which maximizes the second harmonic of the distribution
	phirot=atan(c2s2.mean()/c2c2.mean())/2;
        if(c2c2.mean()/cos(2*phirot)<0){phirot=phirot-PI/2;};
        if(phirot<-PI/2){phirot=phirot+2*PI/2;};

	//! compute the fixed-axes and variable axes moments
        c2c0.reset(); c2c2.reset(); c2c3.reset(); c2c4.reset(); c2c5.reset(); c2c6.reset(); 
        c2s2.reset(); c2s3.reset(); c2s4.reset(); c2s5.reset(); c2s6.reset();
	c2c0r.reset(); c2c2r.reset(); c2c3r.reset(); c2c4r.reset(); c2c5r.reset(); c2c6r.reset();
	c2s2r.reset(); c2s3r.reset(); c2c4r.reset(); c2s5r.reset(); c2s6r.reset();

	//! if SHIFT=1 shift the coordinates x and y to the center-of-mass of the fireball, if SHIFT=0, do not
	//! for asymmetric collisions use SHIFT=1 in the input
       	for(j=1;j<=wc+nbin;j++){
		x=wx[j-1]-SHIFT*xx;
		y=wy[j-1]-SHIFT*yy;
		phi=atanm(x,y); // use the corrected arctan
       		r2=x*x+y*y;
		//! shifted coordinates xs and ys
       		xs=wx[j-1]-xc.mean();
		ys=wy[j-1]-yc.mean();
		phis=atanm(xs,ys); // use the corrected arctan
       		r2s=xs*xs+ys*ys;
        	ww=wa[j-1]; //! the RDS

		//! write the full info on the source: x and y coordinates, RDS, and event number 
		if(FULL){
			tSource.X=wx[j-1]; 
			tSource.Y=wy[j-1]; 
			tSource.W=ww; 
			tSource.KK=kk; 
			full_event->Fill();
		};

		//! compute the fixed-axes moments (use r^2)
		if(r2>0.0){ 
          		c2c0.add(r2,ww);
          		c2c2.add(r2*cos(2*phi),ww);
			//  c2s2.add(r2*sin(2*phi),ww);
			//  c2c3.add(r2*cos(3*phi),ww);
          		c2s3.add(r2*sin(3*phi),ww);
          		c2c4.add(r2*cos(4*phi),ww);
			//  c2s4.add(r2*sin(4*phi),ww);
			//  c2c5.add(r2*cos(5*phi),ww);
			//  c2s5.add(r2*sin(5*phi),ww);
          		c2c6.add(r2*cos(6*phi),ww);
			//  c2s6.add(r2*sin(6*phi),ww);
		}
     		else {
          		c2c0.add(0.0,ww);
          		c2c2.add(0.0,ww);
			//  c2s2.add(0.0,ww);
			//  c2c3.add(0.0,ww);
          		c2s3.add(0.0,ww);
          		c2c4.add(0.0,ww);
			//  c2s4.add(0.0,ww);
			//  c2c5.add(0.0,ww);
			//  c2s5.add(0.0,ww);
          		c2c6.add(0.0,ww);
			//  c2s6.add(0.0,ww);
		};

		//! compute the variable-axes moments (use rs^2)
		if(r2s>0){          
          		c2c0r.add(r2s,ww);
          		c2c2r.add(r2s*cos(2*(phis-phirot)),ww);
			//  c2c3r.add(r2s*cos(3*(phis-phirot)),ww);
          		c2c4r.add(r2s*cos(4*(phis-phirot)),ww);
			//  c2c5r.add(r2s*cos(5*(phis-phirot)),ww);
          		c2c6r.add(r2s*cos(6*(phis-phirot)),ww);
			//  c2s2r.add(r2s*sin(2*(phis-phirot)),ww);
          		c2s3r.add(r2s*sin(3*(phis-phirot)),ww);
			//  c2s4r.add(r2s*sin(4*(phis-phirot)),ww);
			//  c2s5r.add(r2s*sin(5*(phis-phirot)),ww);
			//  c2s6r.add(r2s*sin(6*(phis-phirot)),ww);
		}
  		else {
          		c2c0r.add(0.0,ww);
          		c2c2r.add(0.0,ww);
			//  c2c3r.add(0.0,ww);
          		c2c4r.add(0.0,ww);
			//  c2c5r.add(0.0,ww);
          		c2c6r.add(0.0,ww);
			//  c2s2r.add(0.0,ww);
          		c2s3r.add(0.0,ww);
			//  c2s4r.add(0.0,ww);
			//  c2s5r.add(0.0,ww);
			//  c2s6r.add(0.0,ww);
		};
 
                //! weight for the one-dim. histograms
	  	float wei=ww/(2*PI*sqrt(r2))/(BTOT/NBIN)/EVENTS;   //! density
	  	float weis=ww/(2*PI*sqrt(r2s))/(BTOT/NBIN)/EVENTS; //! density
	  	//float wei=ww/(BTOT/NBIN)/EVENTS;                 //! radial density

		//! weight for the 2-dim. histograms
          	float wei2=ww/(4*BTOT*BTOT/NBIN/NBIN)/EVENTS;

		//! fill the 2D profile histograms
          	xyhist-> Fill(x,y,wei2); //fixed-axes
          	xyhistr->Fill(xs*cos(phirot)-ys*sin(phirot),ys*cos(phirot)+xs*sin(phirot),wei2); //variable-axes

		//! fill the 1D profile histograms
		//! fixed-axes
          	c0hist->Fill(sqrt(r2),phi,wei);
          	c2hist->Fill(sqrt(r2),phi,wei*cos(2*phi));
		//  c3hist->Fill(sqrt(r2),phi,wei*cos(3*phi));
          	s3hist->Fill(sqrt(r2),phi,wei*sin(3*phi));
          	c4hist->Fill(sqrt(r2),phi,wei*cos(4*phi));
		//  c5hist->Fill(sqrt(r2),phi,wei*cos(5*phi));
          	c6hist->Fill(sqrt(r2),phi,wei*cos(6*phi));

		//! variable axes
		Float_t phirr=phis-phirot;
		if(phirr<-PI) {phirr=phirr+2*PI;};
		if(phirr>PI) {phirr=phirr-2*PI;};

          	c0rhist->Fill(sqrt(r2s),phirr,weis);
          	c2rhist->Fill(sqrt(r2s),phirr,weis*cos(2*(phis-phirot)));
		//  c3rhist->Fill(sqrt(r2s),phirr,weis*cos(3*(phis-phirot)));
          	s3rhist->Fill(sqrt(r2s),phirr,weis*sin(3*(phis-phirot)));
          	c4rhist->Fill(sqrt(r2s),phirr,weis*cos(4*(phis-phirot)));
		//  c5rhist->Fill(sqrt(r2s),phirr,weis*cos(5*(phis-phirot)));
          	c6rhist->Fill(sqrt(r2s),phirr,weis*cos(6*(phis-phirot)));
	} //for

        rwA=nwA;rwB=nwB;rwAB=nwA+nwB;rbin=nbin;rhotspot=nhotspot;
        nwounded.add(rwAB);nbinary.add(rbin);nhot.add(rhotspot);nweight.add(rpa);
   
	//! fixed-axes (es_i) and variable-axes (ep_i) moments
        es=c2c2.mean()/c2c0.mean();  
	ep=c2c2r.mean()/c2c0r.mean();
	//  es3=c2c3.mean()/c2c0.mean();ep3=c2c3r.mean()/c2c0r.mean();
        es4=c2c4.mean()/c2c0.mean();
	ep4=c2c4r.mean()/c2c0r.mean();  
	//  es5=c2c5.mean()/c2c0.mean();ep5=c2c5r.mean()/c2c0r.mean();  
        es6=c2c6.mean()/c2c0.mean();
	ep6=c2c6r.mean()/c2c0r.mean();

	//! fixed-axes (es_is) and variable-axes (ep_is) sine moments - these moments average to zero but fluctuate
	//  ess=c2s2.mean()/c2c0.mean();  eps=c2s2r.mean()/c2c0r.mean();
        es3s=c2s3.mean()/c2c0.mean();
	ep3s=c2s3r.mean()/c2c0r.mean();
	// es4s=c2s4.mean()/c2c0.mean();ep4s=c2s4r.mean()/c2c0r.mean();  
	// es5s=c2s5.mean()/c2c0.mean();ep5s=c2s5r.mean()/c2c0r.mean();  
	// es6s=c2s6.mean()/c2c0.mean();ep6s=c2s6r.mean()/c2c0r.mean();

        estd.add(es); 
	epart.add(ep); 
	// estd3.add(es3);epart3.add(ep3);
        estd4.add(es4);
	epart4.add(ep4);
	// estd5.add(es5);epart5.add(ep5);
        estd6.add(es6);
	epart6.add(ep6);

//! fill the data
        nx -> Fill(rwAB,xx);
        nx2 -> Fill(rwAB,xx*xx);
        ny -> Fill(rwAB,yy);
        ny2 -> Fill(rwAB,yy*yy);

        neps -> Fill(rwAB,es);
        neps2 -> Fill(rwAB,es*es);
        nepsp -> Fill(rwAB,ep);
        nepsp2 -> Fill(rwAB,ep*ep);
        nuni -> Fill(rwAB,1);

        nepsb -> Fill(b,es);
        neps2b -> Fill(b,es*es);
        nepspb -> Fill(b,ep);
        nepsp2b -> Fill(b,ep*ep);
        nunib -> Fill(b,1);

//! for multiplicity fluctuations

        nwei->Fill(rwA,rpa);
        nwei2->Fill(rwA,rpa*rpa);
	ntarg->Fill(rwA,rwB);
	ntarg2->Fill(rwA,rwB*rwB);
	nbinar->Fill(rwA,rbin);
	nbinar2->Fill(rwA,rbin*rbin);
	nunp->Fill(rwA,1);

	tree->Fill();

}; // if(wc>WMIN ... )

//! write the current event count
if((kk % kks)==0){cout << "\revent: " << kk << "   ("<< int(float(kk)/float(EVENTS)*100.) << "%)              "; cout.flush();};

}; // end of loop over events 


cout << "\revent: " << kk << "     ("<< 100 << "%)              "; cout.flush();
cout << endl << endl;

/****************************************************************************************************************/
//! output of results
/****************************************************************************************************************/

//! the total cross section
sitot=PI*BMAX*BMAX*EVENTS/evall*10; // 10 converts from fm^2 to mb
float sirad=sqrt(sitot/PI/10.)/2.; //! equivalent hard-sphere radius in fm    

//! transformation of histograms and writing to the Root output file

//! project out the marginal distribution in the radial variable (the radial Fourier profiles)
TH1D *c0hp = c0hist->ProjectionX("c0hp");
TH1D *c2hp = c2hist->ProjectionX("c2hp");
// TH1D *c3hp = c3hist->ProjectionX("c3hp");
TH1D *s3hp = s3hist->ProjectionX("s3hp");
TH1D *c4hp = c4hist->ProjectionX("c4hp");        
// TH1D *c5hp = c5hist->ProjectionX("c5hp");        
TH1D *c6hp = c6hist->ProjectionX("c6hp");

TH1D *c0rhp = c0rhist->ProjectionX("c0rhp");
TH1D *c2rhp = c2rhist->ProjectionX("c2rhp");
TH1D *s3rhp = s3rhist->ProjectionX("s3rhp");
// TH1D *c3rhp = c3rhist->ProjectionX("c3rhp");        
TH1D *c4rhp = c4rhist->ProjectionX("c4rhp");
// TH1D *c5rhp = c5rhist->ProjectionX("c5rhp");        
TH1D *c6rhp = c6rhist->ProjectionX("c6rhp");

#if(_profile_ || _weight_)
   tree1->Write();
#endif
c0hp->Write(); 
c2hp->Write(); 
//c3hp->Write(); 
s3hp->Write();
c4hp->Write(); 
//c5hp->Write(); 
c6hp->Write(); 
c0rhp->Write();
c2rhp->Write(); 
//c3rhp->Write();
s3rhp->Write(); 
c4rhp->Write(); 
//c5rhp->Write();
c6rhp->Write(); 

xyhist->Write(); 
xyhistr->Write();

c0hist->Write(); 
c0rhist->Write();

//! generate variances of Delta x, Delta y, and histograms var(eps)/eps^2, var(eps*)/eps*^2
	nx->Divide(nx,nuni);
        nx->Write();
        nx2->Divide(nx2,nuni);
        nx->Multiply(nx,nx);
        nx2->Add(nx,-1);

        ny->Divide(ny,nuni);
        ny->Write();
        ny2->Divide(ny2,nuni);
        ny->Multiply(ny,ny);
        ny2->Add(ny,-1);

        neps->Divide(neps,nuni);
        neps->Write();
        neps2->Divide(neps2,nuni);
        neps->Multiply(neps,neps);
        neps2->Add(neps,-1);
        neps2->Divide(neps2,neps);

        nepsp->Divide(nepsp,nuni);
        nepsp->Write();
        nepsp2->Divide(nepsp2,nuni);
        nepsp->Multiply(nepsp,nepsp);
        nepsp2->Add(nepsp,-1);
        nepsp2->Divide(nepsp2,nepsp);
        	
        nepsb->Divide(nepsb,nunib);
        nepsb->Write();
        neps2b->Divide(neps2b,nunib);
        nepsb->Multiply(nepsb,nepsb);
        neps2b->Add(nepsb,-1);
        neps2b->Divide(neps2b,nepsb);

        nepspb->Divide(nepspb,nunib);
        nepspb->Write();
        nepsp2b->Divide(nepsp2b,nunib);
        nepspb->Multiply(nepspb,nepspb);
        nepsp2b->Add(nepspb,-1);
        nepsp2b->Divide(nepsp2b,nepspb);

        nx2->Write();
        ny2->Write();
        neps2->Write();
        nepsp2->Write();
        nuni->Write();
        neps2b->Write();
        nepsp2b->Write();
        nunib->Write();

	//! for fluctuations
	nwei->Divide(nwei,nunp);
	ntarg->Divide(ntarg,nunp);
	nbinar->Divide(nbinar,nunp);
        nwei->Write();
	ntarg->Write();
	nbinar->Write();
	   
	nwei2->Divide(nwei2,nunp);
        nwei2->Divide(nwei2,nwei);
	nwei2->Add(nwei,-1);
  	nwei2->Write();
	   
	ntarg2->Divide(ntarg2,nunp);
        ntarg2->Divide(ntarg2,ntarg);
	ntarg2->Add(ntarg,-1);
  	ntarg2->Write();
	   
	nbinar2->Divide(nbinar2,nunp);
        nbinar2->Divide(nbinar2,nbinar);
	nbinar2->Add(nbinar,-1);
  	nbinar2->Write();
	   
	nunp->Write();

//! fill the tree with the physical info
  	xeps=estd.mean();
      	xseps=sqrt(estd.var());
      	xepp=epart.mean();
      	xsepp=sqrt(epart.var());
      	phys->Fill();

//! write the Root trees
      	fout->cd();
      	param->Write();
      	tree->Write();
      	phys->Write(); 
      	if(FULL){full_event->Write();}; //! write the full event info

//! output  basic results to the console
      cout << "Quantities for the specified b, N_w, and RDS window:" << endl;
      if(BMIN*BMIN<0.0001) {cout << "A+B cross section = " << sitot << "mb";};
      if((NUMA>2) && (BMIN*BMIN<0.0001)) {cout << ", equiv. hard-sphere radius = " << sirad << "fm";};
      cout << endl << "efficiency (accepted/all) = " << 100.*EVENTS/evall << "%" << endl;
      cout << "N_w = " << nwounded.mean()  << "+/-" << sqrt(nwounded.var());
      if(ALPHA>0 || DOBIN==1){cout << ", N_bin = " << nbinary.mean()  << "+/-" << sqrt(nbinary.var());};
      if(nhot.mean()>0.0){cout << ", N_hotspot = " << nhot.mean()  << "+/-" << sqrt(nhot.var());};
      cout << endl; 
      cout << "relative deposited strength (RDS) = " << nweight.mean()  << "+/-" << sqrt(nweight.var()) << endl; 
      cout << "eps_fix_2  (std.) = " << estd.mean() << "+/-" << sqrt(estd.var()) << endl;
      cout << "eps_var_2 (part.) = " << epart.mean() << "+/-" << sqrt(epart.var()) << endl;
      cout << "eps_fix_4 = " << estd4.mean() << "+/-" << sqrt(estd4.var()) << endl;      
      cout << "eps_var_4 = " << epart4.mean() << "+/-" << sqrt(epart4.var()) << endl;
      cout << "x_cm = " << xxc.mean() << "+/-" << sqrt(xxc.var()) << ",  y_cm = " << yyc.mean() << "+/-" << sqrt(yyc.var()) << endl;

//! footer
cout << endl; time ( &rawtime ); timeinfo = localtime ( &rawtime ); 
ti=int(rawtime) - ti; int tig=ti/3600; ti=ti-tig*3600; int tim=ti/60; ti=ti-tim*60;
cout << "Finish: " << asctime (timeinfo) << "(" << tig << "h:" << tim <<"m:" << ti << "s)" << endl;
cout << "**************************************" << endl << endl;
}
