/** \file glissando3.cxx
 * The main file of GLISSANDO 3
 * 
 */


/*! \mainpage
                                                                                         
            GLISSANDO 3 - GLauber Initial State Simulation AND mOre...       
                          ver. 3.44, 11 June 2020
                                                                              
 Homepage: http://www.ujk.edu.pl/strony/mryb/GLISSANDO/index.html

 Authors: 
           - Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              
           - Maciej Rybczynski   (Maciej.Rybczynski@ujk.edu.pl)            
           - Grzegorz Stefanek   (Grzegorz.Stefanek@ujk.edu.pl)   
           - Piotr Bozek         (Piotr.Bozek@fis.agh.edu.pl)                    

  For ver. 3, see
  Computer Physics Communications 245 (2019) 106850arXiv:1901.04484 [nucl-th]
          
  For ver. 2, see
  Computer Physics Communications 185 (2014) 1759, arXiv:1310.5475 [nucl-th] 
                                                                                                                                        
  For ver. 1, see
  Computer Physics Communications 180(2009)    69, arXiv:0710.5731 [nucl-th]     
                                                                                                                                                         
  
  GLISSANDO is a Glauber Monte-Carlo generator for early-stages of relativistic
  heavy-ion collisions, written in c++ and interfaced to ROOT.    
                                             
  A reference manual, generated by doxygen, is supplied at the Homepage.                  
                                                                             
  The code can be freely used and redistributed. However, if you decide to  
  make modifications, the authors would appreciate notification for the record.             
  In any publication or display of results obtained using GLISSANDO, please,       
  include a reference to our papers

Computer Physics Communications 245 (2019) 106850arXiv:1901.04484 [nucl-th],
Computer Physics Communications 180 (2009)   69, arXiv:0710.5731 [nucl-th], and
Computer Physics Communications 185 (2014) 1759, arXiv:1310.5475 [nucl-th]
                           
*/

#include <math.h>
#include <time.h>
#include <string.h>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <TH1D.h>
#include <TH2D.h>
#include <TH3D.h>
#include <TFile.h>
#include <TTree.h>
#include <TRandom3.h>


/***********************
    current version 
***********************/
#define _VER_ 3.44

Float_t ver=_VER_;  //!< current version of the code

/***************************
Root random number generator  
****************************/
TRandom3 raa;       //!< ROOT random number generator 

/*******************
   GLISSANDO 3 icludes
********************/
#include "counter.h"
#include "functions.h"
#include "distrib.h"
#include "collision.h"

using namespace std;

/***********************************************************************************************/
//! the main function of GLISSANDO 3
/*!
The main function of GLISSANDO 3 contains the basic structure of the Glauber Monte Carlo simulation, i.e., 
declarations and definitions of basic objects of the nucleus and collision classes, the main loop over events, 
evaluation of basic quantities, etc. It is meant to be taylored by the user to meet his needs. 
For speed of the execution, some switches of the code are controlled by the preprocessor variables 
(_nnwp_, _files_, etc.).
*/

Int_t main(
        Int_t argc, //!< number of command line parameters
//! used for passing input and output file names
/*! first argument: <input.dat>
    second argument: <output.root>
    third argument: <nucl_A.dat>  (present only when _files_=1)
    fourth argument: <nucl_B.dat> (esent pronly when _files_=1) */
        char* argv[] //!< command line parameters (file names)
        ) {

// value passed to functions.h
ARGC=argc;

//! (the units for all dimensionful quantities in GLISSANDO are powers of fm)

Int_t ts; // time
//! start time
ts=time_start(); 

//! print basic info 
helper(argc, argv[1]);

//! print header
header();

//! set the input file
// the input file can be passed as the first command-line argument
TString inpfile("input/input.dat"); 
if (argc>1) inpfile = argv[1];

if (fstream(inpfile)) {} 
    else {cout << "input file " << inpfile << " does not exist or failed to open" << endl; exit(0);};

//! process input parameters
readpar(inpfile);

//! set the ROOT output file
// the ROOT output file with the results can be passed as the second command-line argument
TFile *fout;
TString outfile("output/glissando.root"); 
if (argc>2) outfile = argv[2];
fout = new TFile(outfile,"RECREATE"); 
// fout->cd();

if (fout->IsOpen()){cout << endl << "generates Root output file " << outfile << endl;}
    else {cout << endl << "output file " << outfile << " could not be created" << endl; exit(0);};

#if(_evout_)
//! output file for hydro
ofstream eveout(outfile+".points");
cout << endl << "generates events output file " << outfile << ".points (large output!)" << endl;
#endif

//! seed the ROOT random-number generator
raa.SetSeed(ISEED); 
ISEED1 = ISEED;
ISEED = raa.GetSeed();

//! reset counters used for some basic physical quantities
reset_counters(); 

//! declare and initialize trees and histograms for storage of data
tr_his_c tr_his; // basic storage object for ROOT trees and histograms
tr_his.init();   // initialize

//! set the minimum wounding and binary-collision distances (hard-sphere profile) or the Gaussian wounding parameters (Gaussian profile)
d=sqrt(SNN/PI/10.);         // the minimum wounding distance, 10 converts to fm^2 from mb
Float_t d2=d*d;             // square of the minimum wounding distance
// dbin=sqrt(SBIN/PI/10.);  // the minimum binary collision distance 
dbin=d;                     // the minimum binary collision distance (note that in our model choice it is equal to the wounding distance)
Float_t dbin2=dbin*dbin;    // square of the minimum binary collision distance 
Float_t mbin=SBIN/SNN;      // ratio for the hot-spot model (if mbin<1, only a fraction of mbin binary collisions is included, 
                            // but with the RDS enhanced by the factor 1/mbin)
Float_t ds=sqrt(2.0)*DS;    // source smearing parameter (fixed value) rescaled conventionally by sqrt(2)

Int_t kks=Int_t(fmax(1,fmin(1000,EVENTS/10))); // for output of the progress of the run made once a while

//! echo basic parameters to the console
echopar();

/****************************************************************************************************************/
//! #if(_files_) then initialize the nucleon distributions from external tables from other sources
//! the third command line argument is the file for nucleus A
//! and the fourth (optional) for nucleus B - if absent, B is generated randomly in Glissando3.
//! Format of reading is adjusted to the format of files, which is different for A=3 (helium, tritium) and A>3.
/****************************************************************************************************************/

#if(_files_) // initialize tables for the nucleon distributions

FILE *nucleusA, *nucleusB;

// the input files with nuclear configurations can be passed as the third and fourth command-line arguments

if (argc<4){cout << "Provide in the command line the input files with nuclear distributions! " << endl << endl; exit(0);}

TString nAfile("nucl/pb208.dat");
if (argc>3) nAfile = argv[3];
nucleusA = fopen(nAfile,"rt");

TString nBfile("nucl/pb208.dat");
if (argc>4) nBfile = argv[4]; else nBfile = argv[3];
nucleusB = fopen(nBfile,"rt");

// count number of configurations stored in files
cout << "counting the length of nuclear distribution files " << endl;

// the formats for the files for NUMA or NUMB = 208, 40 and 16 are different for NUMA or NUMB = 3
 
Int_t cfc=0,bfc;
while ((bfc=fgetc(nucleusA))!=EOF) cfc+=(bfc==10)?1:0;fseek(nucleusA,0,SEEK_SET);
Int_t snuA=cfc;
if((NUMA!=3) && (snuA % NUMA != 0)){cout << "wrong number of lines in " << nAfile << ", check! - exiting" << endl << endl; exit(0);}
nucleusA = fopen(nAfile,"rt");

cfc=0; 
while ((bfc=fgetc(nucleusB))!=EOF) cfc+=(bfc==10)?1:0;fseek(nucleusB,0,SEEK_SET);
Int_t snuB=cfc;
if((NUMB!=3) && (argc>4) && (snuB % NUMB != 0)){cout << "wrong number of lines in " << nBfile << ", check! - exiting" << endl << endl; exit(0);}
nucleusB = fopen(nBfile,"rt");

Float_t x, y, z;
Float_t x1, x2, x3, y1, y2, y3, z1, z2, z3, ax1, ax2, ax3;
Int_t countA=0, countB=0, charge;

Float_t* posAx=NULL; Float_t* posBx=NULL; Float_t* posAy=NULL; Float_t* posBy=NULL; Float_t* posAz=NULL; Float_t* posBz=NULL;

if(NUMA==3){

posAx=new Float_t[3*snuA];
posAy=new Float_t[3*snuA]; 
posAz=new Float_t[3*snuA];

cout  << "reading " << snuA << " configurations for nucleus A from " << nAfile << endl;
for(Int_t i=0;i<snuA;i++){
        fscanf(nucleusA,"%f %f %f %f %f %f %f %f %f %f %f %f %d",&x1,&y1,&z1,&x2,&y2,&z2,&x3,&y3,&z3,&ax1,&ax2,&ax3,&charge);
		posAx[3*i]=x1;
	    posAy[3*i]=y1;
	    posAz[3*i]=z1;
		posAx[3*i+1]=x2;
	    posAy[3*i+1]=y2;
	    posAz[3*i+1]=z2;
		posAx[3*i+2]=x3;
	    posAy[3*i+2]=y3;
	    posAz[3*i+2]=z3;
    };
};

if(NUMA==12){

posAx=new Float_t[12*snuA];
posAy=new Float_t[12*snuA]; 
posAz=new Float_t[12*snuA];

cout  << "reading " << snuA << " configurations for nucleus A from " << nAfile << endl;
for(Int_t i=0;i<snuA;i++){
                fscanf(nucleusA,"%f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f",
				&ax1,&ax2, 
				&posAx[12*i], &posAy[12*i], &posAz[12*i],
				&posAx[12*i+1], &posAy[12*i+1], &posAz[12*i+1],
				&posAx[12*i+2], &posAy[12*i+2], &posAz[12*i+2],
				&posAx[12*i+3], &posAy[12*i+3], &posAz[12*i+3],
				&posAx[12*i+4], &posAy[12*i+4], &posAz[12*i+4],
				&posAx[12*i+5], &posAy[12*i+5], &posAz[12*i+5],
				&posAx[12*i+6], &posAy[12*i+6], &posAz[12*i+6],
				&posAx[12*i+7], &posAy[12*i+7], &posAz[12*i+7],
				&posAx[12*i+8], &posAy[12*i+8], &posAz[12*i+8],
				&posAx[12*i+9], &posAy[12*i+9], &posAz[12*i+9],
				&posAx[12*i+10], &posAy[12*i+10], &posAz[12*i+10],
				&posAx[12*i+11], &posAy[12*i+11], &posAz[12*i+11]);
    };
};

if(NUMA==16){

posAx=new Float_t[16*snuA];
posAy=new Float_t[16*snuA]; 
posAz=new Float_t[16*snuA];

cout  << "reading " << snuA << " configurations for nucleus A from " << nAfile << endl;
for(Int_t i=0;i<snuA;i++){
                fscanf(nucleusA,"%f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f",
				&posAx[16*i], &posAy[16*i], &posAz[16*i],
				&posAx[16*i+1], &posAy[16*i+1], &posAz[16*i+1],
				&posAx[16*i+2], &posAy[16*i+2], &posAz[16*i+2],
				&posAx[16*i+3], &posAy[16*i+3], &posAz[16*i+3],
				&posAx[16*i+4], &posAy[16*i+4], &posAz[16*i+4],
				&posAx[16*i+5], &posAy[16*i+5], &posAz[16*i+5],
				&posAx[16*i+6], &posAy[16*i+6], &posAz[16*i+6],
				&posAx[16*i+7], &posAy[16*i+7], &posAz[16*i+7],
				&posAx[16*i+8], &posAy[16*i+8], &posAz[16*i+8],
				&posAx[16*i+9], &posAy[16*i+9], &posAz[16*i+9],
				&posAx[16*i+10], &posAy[16*i+10], &posAz[16*i+10],
				&posAx[16*i+11], &posAy[16*i+11], &posAz[16*i+11],
				&posAx[16*i+12], &posAy[16*i+12], &posAz[16*i+12],
				&posAx[16*i+13], &posAy[16*i+13], &posAz[16*i+13],
				&posAx[16*i+14], &posAy[16*i+14], &posAz[16*i+14],
				&posAx[16*i+15], &posAy[16*i+15], &posAz[16*i+15]);
    };
};

if(NUMA!=3 && NUMA!=12 && NUMA!=16){

posAx=new Float_t[snuA]; 
posAy=new Float_t[snuA]; 
posAz=new Float_t[snuA]; 

cout  << "reading " << snuA/NUMA << " configurations for nucleus A from " << nAfile << endl;
for(Int_t i=0;i<snuA;i++){
        fscanf(nucleusA,"%f %f %f %d",&x,&y,&z,&charge);
		posAx[i]=x;
	    posAy[i]=y;
	    posAz[i]=z;};
};

if (NUMB==3 && argc>4) {

posBx=new Float_t[3*snuB];
posBy=new Float_t[3*snuB]; 
posBz=new Float_t[3*snuB];

cout  << "reading " << snuB << " configurations for nucleus B from " << nBfile << endl << endl;
for(Int_t i=0;i<snuB;i++){
        fscanf(nucleusB,"%f %f %f %f %f %f %f %f %f %f %f %f %d",&x1,&y1,&z1,&x2,&y2,&z2,&x3,&y3,&z3,&ax1,&ax2,&ax3,&charge);
		posBx[3*i]=x1;
	    posBy[3*i]=y1;
	    posBz[3*i]=z1;
		posBx[3*i+1]=x2;
	    posBy[3*i+1]=y2;
	    posBz[3*i+1]=z2;
		posBx[3*i+2]=x3;
	    posBy[3*i+2]=y3;
	    posBz[3*i+2]=z3;
    };
};

if (NUMB==12 && argc>4) {

posBx=new Float_t[12*snuB];
posBy=new Float_t[12*snuB]; 
posBz=new Float_t[12*snuB];

cout  << "reading " << snuB << " configurations for nucleus B from " << nBfile << endl << endl;
for(Int_t i=0;i<snuB;i++){
                fscanf(nucleusB,"%f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f",
				&ax1,&ax2, 
				&posBx[12*i], &posBy[12*i], &posBz[12*i],
				&posBx[12*i+1], &posBy[12*i+1], &posBz[12*i+1],
				&posBx[12*i+2], &posBy[12*i+2], &posBz[12*i+2],
				&posBx[12*i+3], &posBy[12*i+3], &posBz[12*i+3],
				&posBx[12*i+4], &posBy[12*i+4], &posBz[12*i+4],
				&posBx[12*i+5], &posBy[12*i+5], &posBz[12*i+5],
				&posBx[12*i+6], &posBy[12*i+6], &posBz[12*i+6],
				&posBx[12*i+7], &posBy[12*i+7], &posBz[12*i+7],
				&posBx[12*i+8], &posBy[12*i+8], &posBz[12*i+8],
				&posBx[12*i+9], &posBy[12*i+9], &posBz[12*i+9],
				&posBx[12*i+10], &posBy[12*i+10], &posBz[12*i+10],
				&posBx[12*i+11], &posBy[12*i+11], &posBz[12*i+11]);
    };
};

if(NUMB==16 && argc>4){

posBx=new Float_t[16*snuB];
posBy=new Float_t[16*snuB]; 
posBz=new Float_t[16*snuB];

cout  << "reading " << snuB << " configurations for nucleus B from " << nBfile << endl;
for(Int_t i=0;i<snuB;i++){
                fscanf(nucleusB,"%f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f %f",
				&posBx[16*i], &posBy[16*i], &posBz[16*i],
				&posBx[16*i+1], &posBy[16*i+1], &posBz[16*i+1],
				&posBx[16*i+2], &posBy[16*i+2], &posBz[16*i+2],
				&posBx[16*i+3], &posBy[16*i+3], &posBz[16*i+3],
				&posBx[16*i+4], &posBy[16*i+4], &posBz[16*i+4],
				&posBx[16*i+5], &posBy[16*i+5], &posBz[16*i+5],
				&posBx[16*i+6], &posBy[16*i+6], &posBz[16*i+6],
				&posBx[16*i+7], &posBy[16*i+7], &posBz[16*i+7],
				&posBx[16*i+8], &posBy[16*i+8], &posBz[16*i+8],
				&posBx[16*i+9], &posBy[16*i+9], &posBz[16*i+9],
				&posBx[16*i+10], &posBy[16*i+10], &posBz[16*i+10],
				&posBx[16*i+11], &posBy[16*i+11], &posBz[16*i+11],
				&posBx[16*i+12], &posBy[16*i+12], &posBz[16*i+12],
				&posBx[16*i+13], &posBy[16*i+13], &posBz[16*i+13],
				&posBx[16*i+14], &posBy[16*i+14], &posBz[16*i+14],
				&posBx[16*i+15], &posBy[16*i+15], &posBz[16*i+15]);
    };
};


if (NUMB!=3 && NUMB!=12 && NUMB!=16 && argc>4) {

posBx=new Float_t[snuB]; 
posBy=new Float_t[snuB]; 
posBz=new Float_t[snuB]; 

cout  << "reading " << snuB/NUMB << " configurations for nucleus B from " << nBfile << endl << endl;
for(Int_t i=0;i<snuB;i++){
        fscanf(nucleusB,"%f %f %f %d",&x,&y,&z,&charge);
		posBx[i]=x;
	    posBy[i]=y;
	    posBz[i]=z;};
};

#endif

/****************************************************************************************************************
  simulation
****************************************************************************************************************/

/****************************************************************************************************************
  set nuclei A and B
****************************************************************************************************************/

//! declare nuclei A nad B
nucleus partonA(NCS*NUMA), partonB(NCS*NUMB); // wounded quarks 
nucleus nucA(NUMA), nucB(NUMB); // colliding nuclei

//! declare the collision
collision collAB(NCS*NUMA,NCS*NUMB); //collision declaration for partons, with NCS==1 works for the nucleons as well

//! --- start the main loop over events
kk=0; // counter of events when a collision occurs
for(Int_t k=1;kk<EVENTS;k++){ 

evall++; // attempted A+B collision (all cases, those where an NN collision occurs or not)

//! generate the distributions of nucleons in nuclei A and B
#if(_files_)
nucA.set_file(posAx, posAy, posAz, snuA, NUMA); 
if (argc>4)  
nucB.set_file(posBx, posBy, posBz, snuB, NUMB);
else 
if(NUMB>16 && BETA2B!=0) nucB.set_random_B_def(CD); 	   
      else if(NUMB>16) nucB.set_random_B(CD); 	  
#if(_clusters_)
      else if (NUMB==16) nucB.set_oxygen_cluster(SCALEB, CD, SIGMAB);
      else if (NUMB==12) nucB.set_carbon_cluster(SCALEB, CD, SIGMAB);
      else if (NUMB==9)  nucB.set_berillium_9_cluster(SCALEB, CD, SIGMAB, SIGMABISB);
      else if (NUMB==8)  nucB.set_berillium_8_cluster(SCALEB, CD, SIGMAB);
      else if (NUMB==7)  nucB.set_berillium_7_cluster(SCALEB, CD, SIGMAB, SIGMABISB);
      else if (NUMB==4)  nucB.set_alpha_cluster(CD, SIGMAB,SCALEB,SIGMABISB);
      else if (NUMB==3)  nucB.set_tritium(SCALEB);
#endif
      else if ((NUMB<17)&&(NUMB>2)) nucB.set_random_B_hos(CD);	  
      else if( NUMB==2)  nucB.set_deuteron(); 
      else nucB.set_proton();
#else

if(NUMA>16 && BETA2A!=0) nucA.set_random_A_def(CD); 
      else if(NUMA>16) nucA.set_random_A(CD); 
	 	  
#if(_clusters_)
      else if (NUMA==16) nucA.set_oxygen_cluster(SCALEA, CD, SIGMAA);
      else if (NUMA==12) nucA.set_carbon_cluster(SCALEA, CD, SIGMAA);
      else if (NUMA==9)  nucA.set_berillium_9_cluster(SCALEA, CD, SIGMAA, SIGMABISA);
      else if (NUMA==8)  nucA.set_berillium_8_cluster(SCALEA, CD, SIGMAA);
      else if (NUMA==7)  nucA.set_berillium_7_cluster(SCALEA, CD, SIGMAA, SIGMABISA);
      else if (NUMA==4)  nucA.set_alpha_cluster(CD, SIGMAA,SCALEA,SIGMABISA);
      else if (NUMA==3)  nucA.set_tritium(SCALEA); 
#endif
      else if ((NUMA<17)&&(NUMA>2)) nucA.set_random_A_hos(CD); 
      else if (NUMA==2)  nucA.set_deuteron();
      else nucA.set_proton();


if(NUMB>16 && BETA2B!=0) nucB.set_random_B_def(CD); 	   
      else if(NUMB>16) nucB.set_random_B(CD);
  
#if(_clusters_)
      else if (NUMB==16) nucB.set_oxygen_cluster(SCALEB, CD, SIGMAB);
      else if (NUMB==12) nucB.set_carbon_cluster(SCALEB, CD, SIGMAB);
      else if (NUMB==9)  nucB.set_berillium_9_cluster(SCALEB, CD, SIGMAB, SIGMABISB);
      else if (NUMB==8)  nucB.set_berillium_8_cluster(SCALEB, CD, SIGMAB);
      else if (NUMB==7)  nucB.set_berillium_7_cluster(SCALEB, CD, SIGMAB, SIGMABISB);
      else if (NUMB==4)  nucB.set_alpha_cluster(CD, SIGMAB,SCALEB,SIGMABISB);
      else if (NUMB==3)  nucB.set_tritium(SCALEB);
#endif
      else if ((NUMB<17)&&(NUMB>2)) nucB.set_random_B_hos(CD); 	  
      else if (NUMB==2)  nucB.set_deuteron(); 
      else nucB.set_proton();
#endif

#if(_uncluster_)
      nucA.uncluster();
      nucB.uncluster();
#endif


//! shift nuclei to the center-of-mass frame
nucA.shift_cmx(); nucA.shift_cmy(); nucA.shift_cmz(); 
nucB.shift_cmx(); nucB.shift_cmy(); nucB.shift_cmz();

/*********************************************************************
The distributions read from files or the deformed one must be rotated.
For the external distributions, three Euler rotations are carried out. 

Rotations are done only in the appropriate angle is set to -1.
*********************************************************************/

#if(_files_==1)
//! rotate randomly in the azimuth
   Float_t phnucl0=2*PI*los(), phnucl01=2*PI*los();
   if(ROTA_PHI==-1.0){nucA.rotate(phnucl0);}; 			// if A is external
   if(ROTB_PHI==-1.0 && argc>4){nucB.rotate(phnucl01);};	// if B is also external
#endif

/*********************************************************************
The clustered distributions od 12C and O16 three Euler rotations
must be carried out. 

Rotations are done only in the appropriate angle is set to -1.
*********************************************************************/

#if(_clusters_==1)
//! rotate randomly in the azimuth
if(NUMA==12 || NUMA==16){
   Float_t phnucl0=2*PI*los();
   if(ROTA_PHI==-1.0){nucA.rotate(phnucl0);};}	
if(NUMB==12 || NUMB==16){
   Float_t phnucl0=2*PI*los();
   if(ROTB_PHI==-1.0){nucB.rotate(phnucl0);};}			
#endif

/**************************************************************************
Since the deformed nuclei generated in Glissando are axially symmetric, two 
rotations are sufficient. 

Rotations are done only in the appropriate angle is set to -1.
***************************************************************************/

//! rotate the nuclei by theta (zx plane) and phi (xy plane) angles 
//! The proper order of rotations is important

if(ROTA_THETA==-1.0)
{
   Float_t ethnucl=0.0;
   ethnucl=2*los()-1;
   nucA.rotate_polar(ethnucl); 
}
   else
   {Float_t ethnucl=0.0;
    ethnucl=cos(PI*(ROTA_THETA)/180.0);
    nucA.rotate_polar(ethnucl);
   }

if(ROTA_PHI==-1.0)
{
   Float_t phnucl=0.0;
   phnucl=2*PI*los();
   nucA.rotate(phnucl);
}
   else
   {Float_t phnucl=0.0;
    phnucl=2*PI*(ROTA_PHI)/360.0;
    nucA.rotate(phnucl);
   }

if(ROTB_THETA==-1.0)
{   
   Float_t ethnucl=0.0;                              
   ethnucl=2*los()-1;
   nucB.rotate_polar(ethnucl);
}
   else
   {Float_t ethnucl=0.0;
    ethnucl=cos(PI*(ROTB_THETA)/180.0);
    nucB.rotate_polar(ethnucl);
   }

if(ROTB_PHI==-1.0)
{
   Float_t phnucl=0.0;              
   phnucl=2*PI*los();
   nucB.rotate(phnucl);
}
   else
   {Float_t phnucl=0.0;
    phnucl=2*PI*(ROTB_PHI)/360.0;
    nucB.rotate(phnucl);
   }


/****************************************************************************************************************
  generate partons
****************************************************************************************************************/

#if(_partons_)
//! parton positions generated on nuclei
partonA.set_parton_A(nucA,NCS);
partonB.set_parton_B(nucB,NCS);
#endif

/****************************************************************************************************************
 optionally, generate nuclear or partonic profiles
****************************************************************************************************************/


#if(_profile_)
Float_t r_nucl;  /*!< the radial position of the nucleon or parton in the nucleus */

#if(_partons_==0)
for(Int_t j=0;j<nucA.n;j++){
#else
for(Int_t j=0;j<partonA.n;j++){
#endif

//! calculation of the radial distribution of nucleons or partons in the nucleus A
#if(_partons_==0)
r_nucl=sqrt(nucA.x[j]*nucA.x[j]+nucA.y[j]*nucA.y[j]+nucA.z[j]*nucA.z[j]);
#else
r_nucl=sqrt(partonA.x[j]*partonA.x[j]+partonA.y[j]*partonA.y[j]+partonA.z[j]*partonA.z[j]);
#endif
tr_his.radA->Fill(r_nucl,1.);

}; //for

#if(_partons_==0)
for(Int_t j=0;j<nucB.n;j++){
#else
for(Int_t j=0;j<partonB.n;j++){
#endif

//! calculation of the radial distribution of nucleons or partons in the nucleus B
#if(_partons_==0)
r_nucl=sqrt(nucB.x[j]*nucB.x[j]+nucB.y[j]*nucB.y[j]+nucB.z[j]*nucB.z[j]);
#else
r_nucl=sqrt(partonB.x[j]*partonB.x[j]+partonB.y[j]*partonB.y[j]+partonB.z[j]*partonB.z[j]);
#endif 
tr_his.radB->Fill(r_nucl,1.);

}; //for


// evaluation of the deformed profile and the 2-particle correlations						 
#if(_partons_==0)

for(Int_t j=0;j<nucA.n;j++){

//! calculation of the radial position of the nucleon in the nucleus A
 r_nucl=sqrt(nucA.x[j]*nucA.x[j]+nucA.y[j]*nucA.y[j]+nucA.z[j]*nucA.z[j]);  
 tr_his.radA->Fill(sqrt(nucA.x[j]*nucA.x[j]+nucA.y[j]*nucA.y[j]+nucA.z[j]*nucA.z[j])); 

//#if(_clusters_) 
 tr_his.radA2xy->Fill(nucA.x[j],nucA.y[j]); // all nucleons in radA2xy
 tr_his.radA2xz->Fill(nucA.x[j],nucA.z[j]); // all nucleons in radA2xz
 tr_his.radA2yz->Fill(nucA.y[j],nucA.z[j]); // all nucleons in radA2yz
 tr_his.radA3->Fill(nucA.x[j],nucA.y[j],nucA.z[j]);
//#endif
 
//! calculation of one-body density in x-y coordinate system in nucleus A
 tr_his.xyhist_nuclA->Fill(nucA.x[j]*1.0,nucA.y[j]*1.0);         
//! calculation of one-body density in r-cos(theta) coordinate system in nucleus A
 tr_his.rcostheta_nuclA->Fill(r_nucl*1.0,nucA.z[j]*1.0/r_nucl); 

    
          for(Int_t i=0;i<j;i++){
                          tr_his.rrelA->Fill(sqrt((nucA.x[j]-nucA.x[i])*(nucA.x[j]-nucA.x[i])+
                                                  (nucA.y[j]-nucA.y[i])*(nucA.y[j]-nucA.y[i])+
                                                  (nucA.z[j]-nucA.z[i])*(nucA.z[j]-nucA.z[i])));  
                      if(NUMA==NUMB){              
                         tr_his.rrel_u->Fill(sqrt((nucB.x[j]-nucA.x[i])*(nucB.x[j]-nucA.x[i])+
                                                  (nucB.y[j]-nucA.y[i])*(nucB.y[j]-nucA.y[i])+
                                                  (nucB.z[j]-nucA.z[i])*(nucB.z[j]-nucA.z[i])));  
                                     };
                               };
                         };
for(Int_t j=0;j<nucB.n;j++){

    
          for(Int_t i=0;i<j;i++){
                           tr_his.rrelB->Fill(sqrt((nucB.x[j]-nucB.x[i])*(nucB.x[j]-nucB.x[i])+
                                                  (nucB.y[j]-nucB.y[i])*(nucB.y[j]-nucB.y[i])+
                                                  (nucB.z[j]-nucB.z[i])*(nucB.z[j]-nucB.z[i])));  
                              };
                         };						 
#endif
#endif

/****************************************************************************************************************
  generate the impact parameter and shift the distributions
****************************************************************************************************************/

//! generate the impact parameter b with the distribution proportional to b^2 in the range (BMAX, BMIN)
b=sqrt((BMAX*BMAX-BMIN*BMIN)*los()+BMIN*BMIN);

#if(_partons_==0)
//! shift the coordinates of the nucleons in nucleus A such that the center of mass is at the point (b*NUMB/(NUMA+NUMB),0)
nucA.shift_x(b*NUMB/(NUMA+NUMB));
//! shift the coordinates of the nucleons in nucleus B such that the center of mass is at the point (-b*NUMA/(NUMA+NUMB),0)
nucB.shift_x(-b*NUMA/(NUMA+NUMB));
#else
//! shift the coordinates of the partons in nucleus A such that the center of mass is at the point (b*NUMB/(NUMA+NUMB),0)
partonA.shift_x(b*NUMB/(NUMA+NUMB));
//! shift the coordinates of the partons in nucleus B such that the center of mass is at the point (-b*NUMA/(NUMA+NUMB),0)
partonB.shift_x(-b*NUMA/(NUMA+NUMB));
#endif

/****************************************************************************************************************
  collide
****************************************************************************************************************/

//! collide the nuclei, create the sources (wounded objects, binary collisions) and RDS
#if(_partons_)
collAB.gen_RDS(partonA,partonB,d2,dbin2,mbin);
#else
collAB.gen_RDS(nucA,nucB,d2,dbin2,mbin);
#endif

// Include the event only when the number of wounded objects is at least WMIN (=2 by default)
// and, additionally, lies between W0 and W1 and the relative deposited strength (RDS) lies between RDS0 and RDS1.
// We also may request that the number of sources with non-zero RDS >= WMIN (relevant
// for the Poisson superposition, where RDS can be zero)


// allow events with zero RDS multiplicity
if((collAB.nwAB >= WMIN) && (collAB.nwAB >= W0) && (collAB.nwAB <= W1) && (collAB.rpa >= RDS0) && (collAB.rpa <= RDS1) /* && (collAB.nzw > WMIN) */ ){

kk++; // count the event in the specified acceptance window

// for asymmetric collisions necessarily use SHIFT=1 in the input
if(SHIFT==1){collAB.shift_cmx_w_c(); collAB.shift_cmy_w_c();};

#if(_partons_)
// find the number of wounded nucleons (based of wounded partons) in A nad B
wfqA=0; wfqB=0;
for(Int_t i=0;i<NUMA;i++){Int_t co=0; for(Int_t j=0;j<NCS;j++){co+=collAB.wwA[NCS*i+j];}; if(co>0){wfqA++;}};
for(Int_t i=0;i<NUMB;i++){Int_t co=0; for(Int_t j=0;j<NCS;j++){co+=collAB.wwB[NCS*i+j];}; if(co>0){wfqB++;}};
wfq=wfqA+wfqB;
#endif

//! #if(_weight_) generate the histograms for the NN collision profiles
#if(_weight_)
for(Int_t j=0;j<collAB.n;j++){if(collAB.c[j]!=0){tr_his.weih->Fill(collAB.w[j]*2/(1-ALPHA),1);}
                            else {tr_his.weih_bin->Fill(collAB.w[j]/ALPHA,1);};};
#endif

//! generate various 2-dim histograms with the distributions of sources
collAB.fill_xy(tr_his.xyhist, 1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS); // all sources
collAB.fill_xy(tr_his.xyhist_mantle, 1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS, -1000, -1);  // nucleus B
collAB.fill_xy(tr_his.xyhist_core, 1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS, 1, 1000); // nucleus A


/****************************************************************************************************************
 output for hydro
****************************************************************************************************************/

#if(_evout_)
//! write out info on the event
collAB.writerds(eveout);
#endif

/****************************************************************************************************************
 output results
****************************************************************************************************************/


//! generate the participant-plane Fourier moments (up to 6-th moment)

// rotate to the frame maximizing the m=1 moment a la Derek Teaney, i.e. with the power r^3
Float_t pht1=collAB.phrot(1,3); 
phirot1=pht1; 
//ep1s=sqrt(fmax(0.,collAB.qx(1,3,ds)*collAB.qx(1,3,ds)+collAB.qy(1,3,ds)*collAB.qy(1,3,ds)));

ep1s=sqrt(pow(collAB.qx(1,3,ds),2)+pow(collAB.qy(1,3,ds),2));

// cout << endl;
// cout << collAB.qx_old(1,3,ds) << " " << collAB.qx(1,3,ds) << " " << ep1s << endl;


// rotate to the frame maximizing the m=2 moment 
roo=2;
if(RO==0){roo=2;};
ppp=PP; if(PP==-1) ppp=2;
Float_t pht2=collAB.phrot(roo,ppp); 
phirot=pht2; 
collAB.rotate(pht2);
collAB.fill_xy(tr_his.xyhistr,       1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS);
collAB.fill_xy(tr_his.xyhist_mantle, 1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS, 1, 1);     // corona sources
collAB.fill_xy(tr_his.xyhist_mantle, 1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS, -1, -1);   // corona sources
collAB.fill_xy(tr_his.xyhist_core,   1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS, 2, 1000);  // core sources
collAB.fill_xy(tr_his.xyhist_core,   1./(4*BTOT*BTOT/NBIN/NBIN)/EVENTS, -1000,-2); // core sources
collAB.rotate(-pht2);
//eps=sqrt(fmax(0.,collAB.qx(2,ppp,ds)*collAB.qx(2,ppp,ds)+collAB.qy(2,ppp,ds)*collAB.qy(2,ppp,ds)));

eps=sqrt(pow(collAB.qx(2,ppp,ds),2)+pow(collAB.qy(2,ppp,ds),2));

//cout << collAB.qx_old(2,ppp,ds) << " " << collAB.qx(2,ppp,ds) << " " << eps << endl;


// rotate to the frame maximizing the m=3 moment 
roo=2;
if(RO==0){roo=3;};
ppp=PP; if(PP==-1) ppp=3;
Float_t pht3=collAB.phrot(roo,ppp);  
phirot3=pht3;
//ep3s=sqrt(fmax(0.,collAB.qx(3,ppp,ds)*collAB.qx(3,ppp,ds)+collAB.qy(3,ppp,ds)*collAB.qy(3,ppp,ds)));

ep3s=sqrt(pow(collAB.qx(3,ppp,ds),2)+pow(collAB.qy(3,ppp,ds),2));

//cout << collAB.qx(3,ppp,ds) << " " << collAB.qx(3,ppp,ds) << " " << ep3s << endl;


// rotate to the frame maximizing the m=4 moment 
roo=2;
if(RO==0){roo=4;};
ppp=PP; if(PP==-1) ppp=4;
Float_t pht4=collAB.phrot(roo,ppp);  
phirot4=pht4;
//ep4s=sqrt(fmax(0.,collAB.qx(4,ppp,ds)*collAB.qx(4,ppp,ds)+collAB.qy(4,ppp,ds)*collAB.qy(4,ppp,ds)));

ep4s=sqrt(pow(collAB.qx(4,ppp,ds),2)+pow(collAB.qy(4,ppp,ds),2));

//cout << collAB.qx_old(4,ppp,ds) << " " << collAB.qx(4,ppp,ds) << " " << ep4s << endl;

// rotate to the frame maximizing the m=5 moment 
roo=2;
if(RO==0){roo=5;};
ppp=PP; if(PP==-1) ppp=5;
Float_t pht5=collAB.phrot(roo,ppp); 
phirot5=pht5;
//ep5s=sqrt(fmax(0.,collAB.qx(5,ppp,ds)*collAB.qx(5,ppp,ds)+collAB.qy(5,ppp,ds)*collAB.qy(5,ppp,ds)));

ep5s=sqrt(pow(collAB.qx(5,ppp,ds),2)+pow(collAB.qy(5,ppp,ds),2));

//cout << collAB.qx_old(5,ppp,ds) << " " << collAB.qx(5,ppp,ds) << " " << ep5s << endl;


// rotate to the frame maximizing the m=6 moment 
roo=2;
if(RO==0){roo=6;};
ppp=PP; if(PP==-1) ppp=6;
Float_t pht6=collAB.phrot(roo,ppp);  
phirot6=pht6;
//ep6s=sqrt(fmax(0.,collAB.qx(6,ppp,ds)*collAB.qx(6,ppp,ds)+collAB.qy(6,ppp,ds)*collAB.qy(6,ppp,ds)));

ep6s=sqrt(pow(collAB.qx(6,ppp,ds),2)+pow(collAB.qy(6,ppp,ds),2));

//cout << collAB.qx_old(6,ppp,ds) << " " << collAB.qx(6,ppp,ds) << " " << ep6s << endl;



//! get some basic properties of the event
rwA=collAB.nwA; rwB=collAB.nwB; rwAB=collAB.nwAB; rbin=collAB.nbin; rhotspot=collAB.nhotspot; rpa=collAB.rpa;
//sizeav=collAB.size(); 
sizeav=sqrt(collAB.msrad_t_w(ds));
xx=collAB.cmx_w();  yy=collAB.cmy_w(); 

nwounded.add(rwAB); nbinary.add(rbin); nhot.add(rhotspot); 
nweight.add(rpa); epart1.add(ep1s); epart.add(eps); epart3.add(ep3s); epart4.add(ep4s);

//! fill the data in trees and histograms

tr_his.fill_res(); tr_his.fill_tr();

//! if(FULL) write the full event info to the file (added for comparability reasons, takes a lot of space)
if(FULL){for(Int_t j=0;j<collAB.n;j++){
			tSource.X=collAB.x[j]; 
			tSource.Y=collAB.y[j]; 
			tSource.W=collAB.w[j]; 
			tSource.KK=kk; 
			tr_his.full_event->Fill();};};

}; // if((collAB.nwAB >= WMIN)...

// write the current event count
if((kk % kks)==0){cout << "\revent: " << kk << "   ("<< Int_t(Float_t(kk)/Float_t(EVENTS)*100.) << "%)              "; cout.flush();};

}; // end of loop over events 

//! --- end of main loop over events

cout << "\revent: " << kk << "     ("<< 100 << "%)              "; cout.flush();
cout << endl << endl;

/****************************************************************************************************************/
//! output some final results to the console
/****************************************************************************************************************/

//! the total cross section and the equivalent hard-sphere radius 

// the total cross section and the equivalent hard-sphere radius
sitot=PI*BMAX*BMAX*EVENTS/evall*10; // obtained A+B inealstic cross section, 10 converts from fm^2 to mb
sirad=sqrt(sitot/PI/10.)/2.; // equivalent hard-sphere interaction radius in fm    

/****************************************************************************************************************
  output histograms
****************************************************************************************************************/

//! project out the marginal distribution in the radial variable (generate the radial Fourier profiles)
// tr_his.proj();
tr_his.write_d();

//! generate and write some histograms with physical quantities
tr_his.gen();

// compute some physical info
xepp=epart.mean(); xsepp=sqrt(epart.var());

// write out the histograms 
        tr_his.fill();
        tr_his.write();

#if(_profile_)
         tr_his.write_r();

Float_t rAm=tr_his.radA -> GetMean();
Float_t rAs=tr_his.radA -> GetRMS();
if(sqrt(rAs*rAs+rAm*rAm)>0){cout << "rms radius of nucleus A: " << sqrt(rAs*rAs+rAm*rAm) << "fm" << endl;}

Float_t rBm=tr_his.radB -> GetMean();
Float_t rBs=tr_his.radB -> GetRMS();
if(sqrt(rBs*rBs+rBm*rBm)>0){cout << "rms radius of nucleus B: " << sqrt(rBs*rBs+rBm*rBm) << "fm" << endl;};
#endif

//! #if(_weight_) normalize to the wounding and the binary cross sections and write
 
#if(_weight_)
        tr_his.weih->Scale(1./(tr_his.weih->Integral()));
        tr_his.weih_bin->Scale(1./(tr_his.weih_bin->Integral()));

        tr_his.write_w();

// normalize to the wounding cross section  
// (BMAX*BMAX/100. is the bin width in the histograms)
        Float_t intn=collAB.w_distr->Integral();
        collAB.w_distr->Scale(1./intn/PI*SNN/10/(BMAX*BMAX/100.));
        collAB.w_distr->Write();

// normalize to the binary cross section         
        intn=collAB.w_distr_bin->Integral();
        collAB.w_distr_bin->Scale(1./intn/PI*SBIN/10/(BMAX*BMAX/100.));
        collAB.w_distr_bin->Write();
#endif

/****************************************************************************************************************
  closing
****************************************************************************************************************/

//! closing ROOT file
fout->Close();
#if(_evout_)
eveout.close();
#endif

//! write exit info
epilog();

//! stop time and print stamp
time_stop(ts);

}
