/** \file functions.h
 * Part of GLISSANDO 3
 * 
 */


#ifndef _GL_FUNCTION1
#define _GL_FUNCTION1


#include <math.h>
#include <time.h>
#include <string.h>
#include <iostream>
#include <fstream>
#include <cstdlib>

#include <TH1D.h>
#include <TH2D.h>
#include <TFile.h>
#include <TTree.h>
#include <TRandom3.h>

#include "counter.h"

using namespace std; 


//! structure for output of the full event - transverse coordinates, weight, number of the event
typedef struct {
                Float_t X, //!< x coordinate
                        Y, //!< y coordinate
                        W; //!< weight
                UInt_t KK; //!< number of the event
                } SOURCE; 

static SOURCE tSource;      //!< structure used when FULL=1 

/**********************************
parameters and their default values
**********************************/
Int_t      PARTONS, CLUSTERS, ARGC;

Int_t      EVENTS=1000,    //!< number of generated events 
           NBIN=40,         //!< number of bins for histogramming in x, y, and r 
           FBIN=72,         //!< number of bins for histogramming in the azimuthal angle
           NUMA=208,        //!< mass number of nucleus A
           NUMB=208,        //!< mass number of nucleus B 
           NCS=3,           //!< number of partons in the nucleon
           WMIN=2,          //!< minimum number of wounded nucleons to record the event 
           MODEL=0,         //!< switch for the superimposed multiplicity distribution: 0 - scale, 1 - Poisson, 2 - Gamma, 3 - Negative Binomial
           DOBIN=0,         //!< 1 - count binary collisions even in the pure wounded-nucleon model. 0 - do not
           W0=2,            //!< minimum allowed number of wounded objects in the acceptance window
           W1=100000,       //!< maximum allowed number of wounded objects in the acceptance window
           SHIFT=1,         //!< 1 - shift the coordinates of the fireball to c.m., 0 - do not
           RET=0,           //!< 0 - fix-last algorithm, 1 - return-to-beginning algorithm for the generation of the nuclear distribution (not used)
           FULL=0,          //!< 1 - generate the full event tree (large output file), 0 - do not
           NNWP=0,          //!< 0 - hard-sphere wounding profile, 1 - Gaussian wounding profile, 2 - gamma wounding profile
           PP=-1,           //!< power of the transverse radius in the Fourier moments 
           RO=0,            //!< rank of the rotation axes (0 - rotation rank = rank of the Fourier moment)
           FILES;           //!< 1 - read from files
           
UInt_t     ISEED,           //!< read seed for the ROOT random number generator, if 0 - random seed generated
           ISEED1;          //!< copy of ISEED

Float_t      BMIN=0.,              //!< minimum value of the impact parameter in the acceptance window
             BMAX=25.,             //!< maximum value of the impact parameter in the acceptance window
             RDS0=0.,              //!< minimum value of the relative deposited strength (RDS) in the acceptance window 
             RDS1=100000,          //!< maximum value of the relative deposited strength (RDS) in the acceptance window 
             RWSA=-1.,             //!< Woods-Saxon radius for nucleus A
             AWSA=0.459,           //!< Woods-Saxon width for nucleus A
	     BETA2A=-1,            //!< Deformation parameter beta2 of deformed Woods-Saxon distribution for nucleus A 
             BETA4A=-1,            //!< Deformation parameter beta4 of deformed Woods-Saxon distribution for nucleus A  
             ROTA_THETA=-1.0,      //!< Parameter controlling the rotation of nucleus A in XZ plane (polar angle THETA, -1 means random rotation)
             ROTA_PHI=-1.0,        //!< Parameter controlling the rotation of nucleus A in XY plane (azimuthal angle PHI, -1 means random rotation)
             RWSB=-1.,             //!< Woods-Saxon radius for nucleus B
             AWSB=0.459,           //!< Woods-Saxon width for nucleus B
	     BETA2B=-1,            //!< Deformation parameter beta2 of deformed Woods-Saxon distribution for nucleus B 
             BETA4B=-1,            //!< Deformation parameter beta4 of deformed Woods-Saxon distribution for nucleus B 
             ROTB_THETA=-1.0,      //!< Parameter controlling the rotation of nucleus B in XZ plane (polar angle THETA, -1 means random rotation)
             ROTB_PHI=-1.0,        //!< Parameter controlling the rotation of nucleus B in XY plane (azimuthal angle PHI, -1 means random rotation)

             BTOT = fmax(RWSA,RWSB)+AWSA+AWSB,    //!< maximum coordinate value for some histograms

             WFA=0.,               //!< the w parameter for the Fermi distribution for nucleus A
             WFB=0.,               //!< the w parameter for the Fermi distribution for nucleus B
             SNN=-1,               //!< NN "wounding" cross section in milibarns
             SBIN=-1,              //!< NN binary cross section in milibarns
             ALPHA=0.12,           //!< the mixing parameter: 0 - wounded, 1 - binary, 0.145 - mixed (PHOBOS)
             Uw=1.,                //!< Poisson or Gamma parameters for superimposed distribution, wounded nucleons
             Ubin=1.,              //!< Poisson or Gamma parameters for superimposed distribution, binary collisions
             Vw=2.,                //!< Negative binomial variance, wounded nucleons
             Vbin=2.,              //!< Negative binomial variance, binary collisions
             PI=4.*atan(1.),       //!< the number pi
             CD=0.9,               //!< closest allowed distance (expulsion distance) between nucleons in the nucleus in fm (simulation of repulsion)
             DW=0.,                //!< dispersion of the location of the source for wounded nucleons (in fm)
             DBIN=0.,              //!< dispersion of the location of the source for binary collisions (in fm)
             GA=1,                 //!< Gaussian wounding profile parameter (height at the origin)
             RCHA=5.66,            //!< harmonic oscillator shell model density mean squared charge radii of  12C-nucleus 
	     RCHB=5.66,            //!< harmonic oscillator shell model density mean squared charge radii of  12C-nucleus 
             RCHP=0.7714,          //!< harmonic oscillator shell model density mean squared charge radii of  proton
	     OMEGA=-1 ,            //!< relative variance of cross-section fluctuations
             GAMA=-1,              //!< gamma wounding profile parameter (height at the origin)
	     SCALEA=1.,            //!< scale parameter for the size of the nucleus (cluster version)
             SIGMAA=1.,            //!< standard deviation of x,y,z coordinates of nucleons in the alpha cluster 
             SIGMABISA=1.,         //!< standard deviation of x,y,z coordinates of nucleons in the 3He cluster or nucleon no. 9
	     SCALEB=1.,            //!< scale parameter for the size of the nucleus (cluster version)
             SIGMAB=1.,            //!< standard deviation of x,y,z coordinates of nucleons in the alpha cluster 
             SIGMABISB=1.,         //!< standard deviation of x,y,z coordinates of nucleons in the 3He cluster or nucleon no. 9
	     QSCALE=-1,            //!< scale parameter in the parton distribution function, f(r)=r^2*exp(-r/QSCALE), QSCALE=1/(4.27/Sqrt(3/2))
             DS=-1,                //!< source smearing parameter			 
	     ECM=-1.;              //!< center of mass energy of the colliding system [GeV]

//! NN inelastic cross section
/*! NN inelastic cross section from our parametrization of the COMPETE model */
Double_t fsNN(
              Double_t ecm //!< sqrt(s_NN)
             ){
Double_t A1 = -30.14595, A2 = 40.32411, A3 = 53.08089, A4 = 0.10405, A5 = -8.75167;	
return A1 + A2 * pow((ecm + A3),A4) + (A5/ecm);
}

//! parton-parton inelastic cross section
/*! parton-parton inelastic cross section that (together with fqscale) reproduces the NN inelastic cross section and the NN wounding profile */
Double_t fsQQ(
              Double_t n,  //!< number of partons
              Double_t ecm //!< sqrt(s_NN)
             ){
//Double_t A1 = -2.08617, A2 = 3.42394, A3 = 63.08614, A4 = 0.17114, A5 = 1.23047; // old parameterization
Double_t A1=-1.6563649628315888 + 8.727372931232146/n + 0.10574688687619543*n;
Double_t A2=-1.5813401885078082 + 9.033465918497289/n + 0.07705413110190783*n;
Double_t A3=0.22047566579219896 - 0.07059010936761091/n + 0.007651515419909485*n;
Double_t A4=-1.0199338858825862 + 23.098621668552994/n - 0.051885801462319595*n;

//return A1 + A2 * pow((ecm + A3),A4) + (A5/ecm); // old parameterization
return A1 + A2 * pow(ecm,A3) + A4/ecm;
}

//!  parton separation parameter
/*! parton-parton separation that (together with fsQQ) reproduces the NN inelastic cross section and the NN wounding profile */
Double_t fqscale(
              Double_t n,  //!< number of partons
              Double_t ecm //!< sqrt(s_NN)
             ){
//Double_t A1 = 0.30531, A2 = -0.17831, A3 = -0.40881; // old parameterization
Double_t A1=0.373548609686603 - 0.9450568031916784/pow(n,2) + 0.11046962684840048/n;
Double_t A2=-0.20700467242030343 - 0.3948182009375831/pow(n,2) + 0.21810489134825164/n;
Double_t A3=-0.2811570439583786 - 5.209372355251708/pow(n,2) + 1.3551868124547997/n;
return A1 + A2 * pow(ecm,A3);
}

Double_t fgama(Double_t ecm){
Double_t A1 = -3.38182e9, A2 = -1.27309e8, A3 = -3.20981e9, 
A4 = 0.0626076, A5 = 0.856094, A6 = -1.28867e8;
return (A1 + A2 * pow(ecm,A5))/(1. + A3 * pow(ecm,A4) + A6 * pow(ecm,A5));
}

Double_t fomega(Double_t ecm){
Double_t A1 = -3.9675, A2 = 4.40719, A3 = 1.8925, A4 = 0.403262, 
A5 = 0.244449, A6 = -0.269337;
return (A1 + A2 * pow(ecm,A5))/(1. + A3 * pow(ecm,A4) + A6 * pow(ecm,A5));
}

			 
//! process the input file containing parameters
void readpar(
            TString inpfile //!< name of the input file
            ){ 

FILE *in; 
in = fopen(inpfile,"rt");

char t[80];

char s1[]={"EVENTS"},
     s2[]={"NBIN"},
     s3[]={"NUMA"},
     s4[]={"NUMB"},
     s5[]={"WMIN"},
     s6[]={"BMIN"},
     s7[]={"BMAX"},
     s8[]={"RWSA"},
     s9[]={"AWSA"},
     s10[]={"BETA2A"},  
     s11[]={"BETA4A"},
     s12[]={"ROTA_THETA"},
     s13[]={"ROTA_PHI"},
     s14[]={"RWSB"},
     s15[]={"AWSB"},
     s16[]={"BETA2B"},  
     s17[]={"BETA4B"},
     s18[]={"ROTB_THETA"},
     s19[]={"ROTB_PHI"},
     s20[]={"SNN"},
     s21[]={"SBIN"},
     s22[]={"ALPHA"},
     s23[]={"Uw"},
     s24[]={"Ubin"},
     s25[]={"Vw"},
     s26[]={"Vbin"},
     s27[]={"CD"},
     s28[]={"MODEL"},
     s29[]={"ISEED"},
     s30[]={"BTOT"},
     s31[]={"W0"},
     s32[]={"W1"},
     s33[]={"RDS0"},
     s34[]={"RDS1"},
     s35[]={"SHIFT"},
     s36[]={"RET"},
     s37[]={"DW"},
     s38[]={"DBIN"},
     s39[]={"WFA"},
     s40[]={"WFB"},
     s41[]={"FULL"},
     s42[]={"FBIN"},
     s43[]={"DOBIN"},
     s44[]={"GA"},
     s45[]={"PP"},
     s46[]={"RO"},
     s47[]={"RCHA"},  
     s48[]={"RCHB"},  
     s49[]={"RCHP"},
     s50[]={"OMEGA"},
     s51[]={"GAMA"},
     s52[]={"SCALEA"},
     s53[]={"SIGMAA"},
     s54[]={"SIGMABISA"},
     s55[]={"SCALEB"},
     s56[]={"SIGMAB"},
     s57[]={"SIGMABISB"},
     s58[]={"NCS"},
     s59[]={"QSCALE"},
     s60[]={"DS"},
     s61[]={"ECM"};

	#if(_nnwp_==0)
            NNWP=0;
	#elif(_nnwp_==1)
	    NNWP=1;
        #elif(_nnwp_==2)
	    NNWP=2;
	#endif

cout << "parameters reset from default in " << inpfile << ":" << endl;

//! scan the input file for the parameters reset from the default values
Int_t fBTOT=0; double v; 
while (!feof(in)){
  fscanf(in,"%s %lf",t,&v);
  if ((!feof(in)) && (t[0]!='#') && (t[0]!='I')) cout << t << "\t"<< v <<endl;

  if ((!feof(in)) && (t[0]!='#') && (t[0]=='I')) cout << t << "\t"<< UInt_t(v) <<endl;
  if (!strcmp(t,s1)) {EVENTS=Int_t(v);goto dalej;} // number of events
  if (!strcmp(t,s2)) {NBIN=Int_t(v);goto dalej;}   // number of bins in histograms
  if (!strcmp(t,s3)) {NUMA=Int_t(v);goto dalej;}   // mass number of nucleus A
  if (!strcmp(t,s4)) {NUMB=Int_t(v);goto dalej;}   // mass number of nucleus B
  if (!strcmp(t,s5)) {WMIN=Int_t(v);goto dalej;}   // minimum number of wounded nucleons to record event
  if (!strcmp(t,s6)) {BMIN=v;goto dalej;}          // minimum impact parameter
  if (!strcmp(t,s7)) {BMAX=v;goto dalej;}          // maximum impact parameter
  if (!strcmp(t,s8)) {RWSA=v;goto dalej;}          // Woods-Saxon radius on nucleus A
  if (!strcmp(t,s9)) {AWSA=v;goto dalej;}          // Wood-Saxon surface thickness parameter for nucleus A
  if (!strcmp(t,s10)) {BETA2A=v;goto dalej;}       // Wood-Saxon deformation parameter beta2 for nucleus A  
  if (!strcmp(t,s11)) {BETA4A=v;goto dalej;}       // Wood-Saxon deformation parameter beta4 for nucleus A  
  if (!strcmp(t,s12)) {ROTA_THETA=v;goto dalej;}   // Parameter controlling rotation of nucleus A in XZ plane (polar angle THETA)
  if (!strcmp(t,s13)) {ROTA_PHI=v;goto dalej;}     // Parameter controlling rotation of nucleus A in XY plane (azimuthal angle PHI) 
  if (!strcmp(t,s14)) {RWSB=v;goto dalej;}         // Woods-Saxon radius on nucleus B
  if (!strcmp(t,s15)) {AWSB=v;goto dalej;}         // Wood-Saxon surface thickness parameter for nucleus B
  if (!strcmp(t,s16)) {BETA2B=v;goto dalej;}       // Wood-Saxon deformation parameter beta2 for nucleus B  
  if (!strcmp(t,s17)) {BETA4B=v;goto dalej;}       // Wood-Saxon deformation parameter beta4 for nucleus B  
  if (!strcmp(t,s18)) {ROTB_THETA=v;goto dalej;}   // Parameter controlling rotation of nucleus B in XZ plane (polar angle THETA)
  if (!strcmp(t,s19)) {ROTB_PHI=v;goto dalej;}     // Parameter controlling rotation of nucleus B in XY plane (azimuthal angle PHI)
  if (!strcmp(t,s20)) {SNN=v;goto dalej;}          // wounding cross section
  if (!strcmp(t,s21)) {SBIN=v;goto dalej;}         // binary cross section
  if (!strcmp(t,s22)) {ALPHA=v;goto dalej;}        // the mixing parameter
  if (!strcmp(t,s23)) {Uw=v;goto dalej;}           // parameter for the distribution superimposed over the wounded nucleons
  if (!strcmp(t,s24)) {Ubin=v;goto dalej;}         // parameter for the distribution superimposed over the binary collisions
  if (!strcmp(t,s25)) {Vw=v;goto dalej;}             // parameter for the distribution superimposed over the wounded nucleons
  if (!strcmp(t,s26)) {Vbin=v;goto dalej;}           // parameter for the distribution superimposed over the binary collisions
  if (!strcmp(t,s27)) {CD=v;goto dalej;}             // the expulsion distance
  if (!strcmp(t,s28)) {MODEL=Int_t(v);goto dalej;}   // switch for the superimposed distribution
  if (!strcmp(t,s29)) {ISEED=UInt_t(v);goto dalej;}  // seed for the random-number generator
  if (!strcmp(t,s30)) {fBTOT=1;BTOT=v;goto dalej;}   // range for histograms
  if (!strcmp(t,s31)) {W0=Int_t(v);goto dalej;}      // minimum number of wounded nucleons
  if (!strcmp(t,s32)) {W1=Int_t(v);goto dalej;}      // maximum number of wounded nucleons
  if (!strcmp(t,s33)) {RDS0=v;goto dalej;}           // minimum RDS (relative deposited strength, see the paper)
  if (!strcmp(t,s34)) {RDS1=v;goto dalej;}           // maximum RDS
  if (!strcmp(t,s35)) {SHIFT=Int_t(v);goto dalej;}   // parameter controlling the shift to c.m. frame 
  if (!strcmp(t,s36)) {RET=Int_t(v);goto dalej;}     // parameter controlling nuclear density generation
  if (!strcmp(t,s37)) {DW=v;goto dalej;}             // width of the distribution of displacement of the location of source for wounded nucleons
  if (!strcmp(t,s38)) {DBIN=v;goto dalej;}           // width of the distribution of displacement of the location of source for binary collisions
  if (!strcmp(t,s39)) {WFA=v;goto dalej;}            // the w parameter of the Fermi distribution for nucleus A
  if (!strcmp(t,s40)) {WFB=v;goto dalej;}            // the w parameter of the Fermi distribution for nucleus B
  if (!strcmp(t,s41)) {FULL=Int_t(v);goto dalej;}    // parameter controlling generation of the full event tree 
  if (!strcmp(t,s42)) {FBIN=Int_t(v);goto dalej;}    // number of histogram bins in the the phi angle
  if (!strcmp(t,s43)) {DOBIN=Int_t(v);goto dalej;}   // if 1, generate binary collision also for the pure wounded-nucleon model
  if (!strcmp(t,s44)) {GA=v;goto dalej;}             // value of the Gaussian wounding profile at the origin
  if (!strcmp(t,s45)) {PP=Int_t(v);goto dalej;}      // power of the transverse radius in the Fourier moments 
  if (!strcmp(t,s46)) {RO=Int_t(v);goto dalej;}      // rank of the rotation axes (0 - rotation rank = rank of the Fourier moment)
  if (!strcmp(t,s47)) {RCHA=v;goto dalej;}           // harmonic oscillator shell model density mean squared charge radii of  A-nucleus
  if (!strcmp(t,s48)) {RCHB=v;goto dalej;}           // harmonic oscillator shell model density mean squared charge radii of  B-nucleus
  if (!strcmp(t,s49)) {RCHP=v;goto dalej;}           // harmonic oscillator shell model density mean squared charge radii of  proton 
  if (!strcmp(t,s50)) {OMEGA=v;goto dalej;}          // relative variance of cross-section fluctuations
  if (!strcmp(t,s51)) {GAMA=v;goto dalej;}           // gamma wounding profile parameter (height at the origin)
  if (!strcmp(t,s52)) {SCALEA=v;goto dalej;}         // scale parameter for the size of the nucleus (cluster version)
  if (!strcmp(t,s53)) {SIGMAA=v;goto dalej;}         // standard deviation of x,y,z coordinates of nucleons in the cluster 
  if (!strcmp(t,s54)) {SIGMABISA=v;goto dalej;}      // standard deviation of x,y,z coordinates of nucleons in the 3He cluster or nucleon no. 9
  if (!strcmp(t,s55)) {SCALEB=v;goto dalej;}         // scale parameter for the size of the nucleus (cluster version)
  if (!strcmp(t,s56)) {SIGMAB=v;goto dalej;}         // standard deviation of x,y,z coordinates of nucleons in the cluster 
  if (!strcmp(t,s57)) {SIGMABISB=v;goto dalej;}      // standard deviation of x,y,z coordinates of nucleons in the 3He cluster or nucleon no. 9
  if (!strcmp(t,s58)) {NCS=Int_t(v);goto dalej;}     // number of partons in each nucleon
  if (!strcmp(t,s59)) {QSCALE=v;goto dalej;}         // scale parameter in the parton distribution function, f(r)=r^2*exp(-r/QSCALE)
  if (!strcmp(t,s60)) {DS=v;goto dalej;}             // source smearing parameter
  if (!strcmp(t,s61)) {ECM=v;goto dalej;}            // center of mass energy of colliding system [GeV]

 dalej:;
 };

 
if (ECM< 5.) {cout << "Small ECM=sqrt(s) <= 5 GeV!" << endl;};

if(ECM<5 &&(SNN<0 || SBIN<0 || OMEGA<0 || GAMA<0)){cout << "Use ECM > 5 GeV or provide your own reaction parameters in the input file." << endl; exit(0);}; 

#if(_partons_)
 if(NNWP!=1){cout << "Gaussian profile _nnwp_=1 must be used for the parton case in Makefile." << endl; exit(0);};
 //if (NCS!=3 && (SNN==-1 || SBIN==-1 || QSCALE==-1)) {cout << "parton-parton cross-section may be calculated only for NCS=3. Provide your own values of SNN and QSCALE." << endl; exit(0);};
 if (((NCS<3)||(NCS>10)) && (SNN==-1 || SBIN==-1 || QSCALE==-1)) {cout << "parton-parton cross-section may be calculated only for 3<=NCS<10. Provide your own values of SNN and QSCALE." << endl; exit(0);};
 
 
 if(SNN==-1){SNN=fsQQ(NCS,ECM);};
 if(SBIN==-1){SBIN=SNN;};
 if(QSCALE==-1){QSCALE=fqscale(NCS,ECM);};
// cout << "parton-parton parameters for sqrt(s_NN)="<< ECM <<" GeV: inel. cross section SNN="<< SNN <<", separation QSCALE=" << QSCALE << endl;
#else
 NCS=1; // one "parton" per nucleon
 if(SNN==-1){SNN=fsNN(ECM);};
 if(SBIN==-1){SBIN=SNN;};
// cout << "nucleon-nucleon inelastic cross-section for sqrt(s_NN)="<< ECM <<" GeV: SNN= "<< SNN << endl;
 if (NNWP==2){
 if(OMEGA==-1){OMEGA=fomega(ECM);};
 if(GAMA==-1){GAMA=fgama(ECM);};
}
#endif 
	

// set the smearing width
if(DS==-1){
#if(_partons_)
DS=0.2;
#else
DS=0.4;
#endif
};

if(DS>0 && ((PP!=-1) || (RO !=0))){cout << "Smearing only works for the model where power of the radius in the definition of eccentricities is equal to the rank. Examine DS, PP(=-1), and  RO(=0)." << endl; exit(0);};

if(RET==1){cout << "Return to beginning algorithm is obsolete, use RET=0 in the input" << endl; exit(0);};

#if(_files_)
FILES=1;
#else
FILES=0;
#endif

#if(_partons_)
PARTONS=1;
#else
PARTONS=0;
#endif

#if(_clusters_)
CLUSTERS=1;
#else
CLUSTERS=0;
#endif

//! reset number of constituents to 1 in the wounded nucleon case
#if(_partons_==0)
if(NCS!=1){NCS=1;
// cout << "correction: NCS reset to 1" << endl;
};
#endif


//! correct wrong input
// if ((FILES==1) && CD*CD > 0.000001){CD=0; cout << "correction: CD=" << CD << endl;};
 if ((MODEL!=0) && (MODEL!=1) && (MODEL!=2) && (MODEL!=3)) {MODEL=0; cout << "correction: MODEL=" << MODEL << endl;};
 if (BMIN<0) {BMIN=0; cout << "correction: BMIN=" << BMIN << endl;};
 if (BMAX<BMIN) {BMAX=BMIN; cout << "correction: BMAX=" << BMAX << endl;};
 if ((ALPHA<0)||(ALPHA>1)) {ALPHA=0; cout << "correction: ALPHA=" << ALPHA << endl;};
 if ((SHIFT!=0) && (SHIFT!=1)) {SHIFT=0; cout << "correction: SHIFT=" << SHIFT << endl;};
 if (W0<WMIN) {W0=WMIN; cout << "correction: W0=" << W0 << endl;};
 if (W1<W0) {W1=W0; cout << "correction: W1=" << W1 << endl;};
 if (RDS1<RDS0) {RDS1=RDS0+1; cout << "correction: RDS1=" << RDS1 << endl;};
 if (Uw<0) {Uw=1; cout << "correction: Uw=" << Uw << endl;};
 if (Ubin<0) {Ubin=1; cout << "correction: Ubin=" << Ubin << endl;};
 if (Vw<0) {Vw=1; cout << "correction: Vw=" << Vw << endl;};
 if (Vbin<0) {Vbin=1; cout << "correction: Vbin=" << Vbin << endl;};
 if (CD<0) {CD=-CD; cout << "correction: CD=" << CD << endl;};
#if(_partons_==0)
// if ((NUMA<3) && (NUMB<3)) {cout << "GLISSANDO 2 is designed only for p(d)+A and A+A collisions of nuclei heavier than deuteron " << endl; 
// exit(0);};
#endif
 if (((ROTA_THETA<0.0) || (ROTA_THETA>180.0)) && (ROTA_THETA!=-1.0))
    {cout << "ROTA_THETA out of range. It should be in range 0.0 - 180.0 or equal -1.0 (random rotation) "<< endl; 
exit(0); };
 if (((ROTA_PHI<0.0) || (ROTA_PHI>360.0)) && (ROTA_PHI!=-1.0)) 
    {cout << "ROTA_PHI out of range. It should be in range 0.0 - 360.0 or equal -1.0 (random rotation) "<< endl;
exit(0); }; 
 if (((ROTB_THETA<0.0) || (ROTB_THETA>180.0)) && (ROTB_THETA!=-1.0))
    {cout << "ROTB_THETA out of range. It should be in range 0.0 - 180.0 or equal -1.0 (random rotation) "<< endl;
exit(0); };
 if (((ROTB_PHI<0.0) || (ROTB_PHI>360.0)) && (ROTB_PHI!=-1.0))
    {cout << "ROTB_PHI out of range. It should be in range 0.0 - 360.0 or equal -1.0 (random rotation) "<< endl;
exit(0); };
//! see the paper for the discussion of parametrizatiations of the nuclear distributions
if (NUMA>16){
if ((RWSA<=0) && ((CD-0.9)*(CD-0.9)<0.001) && (RET==0)) {
   RWSA=1.1*pow(NUMA,1./3.)-0.656*pow(NUMA,-1./3.); AWSA=0.459;
//   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.459fm (see the paper)" << endl;
};
}
if (NUMB>16){  
if ((RWSB<=0) && ((CD-0.9)*(CD-0.9)<0.001) && (RET==0)) {
   RWSB=1.1*pow(NUMB,1./3.)-0.656*pow(NUMB,-1./3.); AWSB=0.459;
//   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSB=0.459fm (see the paper)" << endl;
};
}

if (NUMA>16){  
if ((RWSA<=0) && (CD*CD<0.001) && (RET==0)) {
   RWSA=1.114*pow(NUMA,1./3.)-0.246*pow(NUMA,-1./3.); AWSA=0.45;
//   cout << "Woods-Saxon parameters: RWSA=" << RWSA << "fm, AWSA=0.45fm (see the paper)" << endl;
};
}
if (NUMB>16){  
if ((RWSB<=0) && (CD*CD<0.001) && (RET==0)) {
   RWSB=1.114*pow(NUMB,1./3.)-0.246*pow(NUMB,-1./3.); AWSB=0.45; 
//   cout << "Woods-Saxon parameters: RWSB=" << RWSB << "fm, AWSA=0.45fm (see the paper)" << endl;
};
}

#if(_files_==0)
if (NUMA>16){  
if (RWSA<=0) {cout << "Correct radii! (no formula available for the chosen CD)"<< endl << endl; exit(0); };
}
if (NUMB>16){  
if (RWSB<=0) {cout << "Correct radii! (no formula available for the chosen CD)"<< endl << endl; exit(0); };
}
#endif

#if(_uncluster_==1)
    cout << "nucleus A unclustered";
#endif
#if(_uncluster_==2)
    cout << "nucleus B unclustered";
#endif
#if(_uncluster_==3)
    cout << "nucleus A unclustered";
    cout << "nucleus B unclustered";
#endif

if ((MODEL==3) && ((Vw<=Uw) || (Vbin<=Ubin))) {cout << "Correct variance of Negative Binomial distribution! (V>U)"<< endl << endl; exit(0); };

if (NUMA!=NUMB && SHIFT==0) {SHIFT=1; cout << "Reset to SHIFT=1 for collisions of different nuclei" << endl;};

#if(_nnwp_==2)
if ((OMEGA<=0.0) || (OMEGA>=1.0)){cout << "Provide ECM or correct OMEGA! OMEGA IN(0,1)"<< endl << endl; exit(0); };
if ((GAMA<=0.0) || (GAMA>1.0)){cout << "Provide ECM or correct GAMA! GAMA IN(0,1]"<< endl << endl; exit(0); };
#endif

//! set the range for the histograms
if (fBTOT!=1) BTOT = fmax(RWSA,RWSB)+AWSA+AWSB;
};

//  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
//! echo parameters to the output
void echopar(){
cout << endl;
//if(ISEED1==0){cout << "random ";} else { cout << "fixed ";};
//cout << "seed: " << raa.GetSeed()  << " << endl; 
cout << "number of events: " << EVENTS << endl << endl;

#if(_profile_)
cout << "generates the density profile for nuclei A and B" << endl;
#endif

cout << NUMA <<"+" << NUMB;
if(ECM>5){cout << " @ sqrt(s_NN)=" << ECM << "GeV";};
cout << endl;

if((NUMA>16) && (FILES!=1)){cout << endl << "Woods-Saxon: RA="  << RWSA << "fm, aA=" << AWSA << "fm";};
if((NUMA>16) &&(WFA*WFA>0.00000001) && (FILES!=1)) {cout << ", wA="<< WFA;}
if((NUMA>16) && (FILES!=1)){cout << ", dA=" << CD  << "fm ";};
if((NUMB>16) && ((FILES!=1)||(FILES==1 && ARGC==4))){cout << endl << "Woods-Saxon: RB="  << RWSB << "fm, aB=" << AWSB << "fm";}
if((NUMB>16) &&(WFB*WFB>0.00000001) && ((FILES!=1)||(FILES==1 && ARGC==4))) {cout << ", wB="<< WFB;}
if((NUMB>16) && ((FILES!=1)||(FILES==1 && ARGC==4))){cout << ", dB=" << CD  << "fm ";};

cout << endl;

//! set the deformation parameters

if(BETA2A==-1){
switch (NUMA){
        case 238:{BETA2A=0.215;break;}
	case 197:{BETA2A=-0.131;break;}
	case 129:{BETA2A=0.143;break;}	
	case 63:{BETA2A=0.162;break;}
	case 28:{BETA2A=-0.478;break;}
        default:{BETA2A=0;break;}};
               };

if(BETA4A==-1){
switch (NUMA){
        case 238:{BETA4A=0.093;break;}
	case 197:{BETA4A=-0.031;break;}
	case 129:{BETA4A=0.001;break;}	
	case 63:{BETA4A=-0.06;break;}
	case 28:{BETA4A=0.25;break;}
        default:{BETA4A=0;break;}};
               };

if(BETA2B==-1){
switch (NUMB){
        case 238:{BETA2B=0.215;break;}
	case 197:{BETA2B=-0.131;break;}
	case 129:{BETA2B=0.143;break;}	
	case 63:{BETA2B=0.162;break;}
	case 28:{BETA2B=-0.478;break;}
        default:{BETA2B=0;break;}};
              };

if(BETA4B==-1){
switch (NUMB){
        case 238:{BETA4B=0.093;break;}
	case 197:{BETA4B=-0.031;break;}
	case 129:{BETA4B=0.001;break;}	
	case 63:{BETA4B=-0.06;break;}
	case 28:{BETA4B=0.25;break;}
        default:{BETA4B=0;break;}};
               };


#if(_clusters_)
switch (NUMA) {
        case 16:{SCALEA=3.;SIGMAA=1.23;break;}
	case 12:{SCALEA=2.837;SIGMAA=1.15;break;}
	case 9:{SCALEA=3.55;SIGMAA=0.91;SIGMABISA=1.08;break;}	
	case 8:{SCALEA=3.61;SIGMAA=1.11;break;}
	case 7:{SCALEA=3.17;SIGMAA=0.97;SIGMABISA=1.14;break;}
	case 4:{SCALEA=1.;SIGMAA=1.;SIGMABISA=1.;break;}}
switch (NUMB) {
        case 16:{SCALEB=3.;SIGMAB=1.23;break;}
	case 12:{SCALEB=2.837;SIGMAB=1.15;break;}
	case 9:{SCALEB=3.55;SIGMAB=0.91;SIGMABISB=1.08;break;}	
	case 8:{SCALEB=3.61;SIGMAB=1.11;break;}
	case 7:{SCALEB=3.17;SIGMAB=0.97;SIGMABISB=1.14;break;}
	case 4:{SCALEB=1.;SIGMAB=1.;SIGMABISB=1.;break;}}
#endif

#if(!_clusters_)

// if(NUMA<17 && NUMA >3){cout << "WARNING: HO parametrization is tuned to CD=0.9" << endl;};
// if(NUMB<17 && NUMB >3){cout << "WARNING: HO parametrization is tuned to CD=0.9" << endl;};

switch (NUMA){
    case 16:{RCHA=6.81;break;}
    case 15:{RCHA=6.32;break;}
    case 14:{RCHA=6.08;break;}
    case 13:{RCHA=5.6;break;}
    case 12:{RCHA=5.66;break;}
    case 11:{RCHA=5.36;break;}
    case 10:{RCHA=5.5;break;}
    case 9:{RCHA=6.0;break;}
    case 8:{RCHA=5.1;break;}
    case 7:{RCHA=6.69;break;}
    case 6:{RCHA=6.4;break;}
    case 4:{RCHA=2.45;break;}}

switch (NUMB){
    case 16:{RCHB=6.81;break;}
    case 15:{RCHB=6.32;break;}
    case 14:{RCHB=6.08;break;}
    case 13:{RCHB=5.6;break;}
    case 12:{RCHB=5.66;break;}
    case 11:{RCHB=5.36;break;}
    case 10:{RCHB=5.5;break;}
    case 9:{RCHB=6.0;break;}
    case 8:{RCHB=5.1;break;}
    case 7:{RCHB=6.69;break;}
    case 6:{RCHB=6.4;break;}
    case 4:{RCHB=2.45;break;}}
#endif

if((BETA2A != 0) || (BETA4A != 0)) cout << "deformation parameters for nucleus A: BETA2=" << BETA2A << ", BETA4=" << BETA4A << endl;
if(((BETA2A != 0) || (BETA4A != 0)) && ((ROTA_THETA!=-1) || (ROTA_PHI!=-1))) 
  cout << "rotation angles for nucleus A: cos(theta)=" << ROTA_THETA << ", phi=" << ROTA_PHI << endl;
if((BETA4B != 0) || (BETA4B != 0)) cout << "deformation parameters for nucleus B: BETA2=" << BETA2B << ", BETA4=" << BETA4B << endl;
if(((BETA2B != 0) || (BETA4B != 0)) && ((ROTB_THETA!=-1) || (ROTB_PHI!=-1))) 
  cout << "rotation angles for nucleus B: cos(theta)=" << ROTB_THETA << ", phi=" << ROTB_PHI << endl;

if(FILES!=1){
if((CLUSTERS==1)&&((NUMA==7)||(NUMA==8)||(NUMA==9)||(NUMA==12)||(NUMA==16))){cout << "clustered distribution for nucleus A" << endl;}
else
// harmonic oscillator shell model
if ((NUMA<17) && (NUMA>3)){cout << "HO: ms radius of nucleus A =" <<RCHA << " fm^2" << ", dA=" << CD  << "fm " <<endl;};};

if((FILES!=1)||(FILES==1 && ARGC==4)){
if((CLUSTERS==1)&&((NUMB==7)||(NUMB==8)||(NUMB==9)||(NUMB==12)||(NUMB==16))){cout << "clustered distribution for nucleus B" << endl;}
else
// harmonic oscillator shell model
if ((NUMB<17) && (NUMB>3)){cout << "HO: ms radius of nucleus B =" <<RCHB << " fm^2" << ", dB=" << CD  << "fm " <<endl;};};


#if(_uncluster_)
   cout << endl << "UNCLUSTERED CASE" << endl;
#endif

cout << endl;
#if(_partons_)
cout << "PARTON MODEL (number of partons=" << NCS << ")" << endl;
#if(_pardis_==0)
cout << "exponential parton distribution, QSCALE=";
#endif
#if(_pardis_==1)
cout << "gaussian parton distribution, QSCALE=";
#endif
cout << QSCALE << "fm" << endl;
#else
cout << "NUCLEON MODEL" << endl;
#endif


if(ALPHA==0){cout << "sig_w=" << SNN << "mb" << endl;};
if(ALPHA==0 && DOBIN==1){cout << "   (binary collisions counted)" << endl;};
// if(ALPHA==0 && DOBIN!=1){cout << "   (binary collisions not counted)" << endl;};
if(ALPHA>0 && SBIN>=SNN){cout << "mixed model: sig_w=" << SNN << "mb, sig_bin=" << SBIN << "mb, alpha=" << ALPHA << endl;};
if(ALPHA>0 && SBIN<SNN){cout << "mixed model with hotspots: sig_w=" << SNN << "mb, sig_bin=" << SBIN << "mb, alpha=" << ALPHA << endl;};

if(MODEL==0){cout << "RDS distribution scale with u=" << Uw << " (wounded)"; if(ALPHA>0 || DOBIN==1){cout << " and u=" << Ubin << " (binary)";}; cout << endl;};
if(MODEL==1){cout << "overlaid Poisson distribution with u=" << Uw << " (wounded)"; if(ALPHA>0 || DOBIN==1){cout << " and u=" << Ubin << " (binary)";}; cout << endl;};
if(MODEL==2){cout << "overlaid Gamma distribution with u=" << Uw << ", v="<< Vw 
                  << ", k=" << Uw*Uw/(Vw-Uw) << " (wounded)"; if(ALPHA>0 || DOBIN==1){cout << " and u=" 
                  << Ubin << ", v="<< Vbin << ", k=" << Ubin*Ubin/(Vbin-Ubin) <<  " (binary)";}; cout << endl;};
if(MODEL==3){cout << "overlaid negative binomial distribution with u=" << Uw << ", v="<< Vw 
                  << ", k=" << Uw*Uw/(Vw-Uw) << " (wounded)"; if(ALPHA>0 || DOBIN==1){cout << " and u=" 
                  << Ubin << ", v="<< Vbin << ", k=" << Ubin*Ubin/(Vbin-Ubin) <<  " (binary)";}; cout << endl;};

#if(_nnwp_==0)
   cout << "hard sphere wounding profile" << endl;
#elif(_nnwp_==1)
   cout << "Gaussian wounding profile, G=" << GA << endl;
#elif(_nnwp_==2)
    cout << "gamma wounding profile, G="<< GAMA <<", omega="<< OMEGA << endl;
#endif


cout << "source smearing width: " << DS << " fm" << endl;

//if(RO==0){cout << "rank of rotation corresponds to the rank of the given Fourier moment" << endl;};
//if(RO>0){cout << "rank of rotation fixed to " << RO << " for all Fourier moments" << endl;};
if(PP==-1) ;
// cout << "power of transverse radius in eccentricities = rank of the Fourier moment" << endl;
else cout << "power of transverse radius in eccentricities =" << PP << endl;
cout << "acceptance window: b_min=" << BMIN << "fm, b_max=" << BMAX << "fm";
if(W1!=1000 || W0!=2){cout << ", Nw_min=" << W0 << ", Nw_max=" << W1;};
if(RDS1!=100000 || RDS0!=0){cout << ", RDS_min=" << RDS0 << ", RDS_max=" << RDS1;};
cout << endl;


if(SHIFT==0){cout << "(fireball not shifted to its center of mass)" << endl;};
// if(SHIFT==1){cout << "(coordinates shifted to the c.m. frame of the fireball)" << endl;};
if(CD>0.0){
	if (RET==1){
	   cout << "return-to-beginning algorithm (slow, recommended to use RET=0)" << endl;} 
//	else {cout << "fix-last algorithm" << endl;};
}; 
if(DW>0.0 || DBIN >0.0) {cout << "source dispersion parameter: wounded=" << DW << "fm, binary=" << DBIN << "fm";}; 
cout << endl; 
if(FULL){cout << "full event tree generated (THIS GENERATES A LOT OF DATA, set FULL=0 to switch off)" << endl;};
};


/*************************************
 declaration of counters and variables 
*************************************/

counter2 

epart1,           //!< counter for participant-plane <r^3 cos(phi)>/<r^3>
epart,            //!< counter for participant-plane <r^n cos(2 phi)>/<r^n>
epart3,           //!< counter for participant-plane <r^n cos(3 phi)>/<r^n>
epart4,           //!< counter for participant-plane <r^n cos(4 phi)>/<r^n>
epart5,           //!< counter for participant-plane <r^n cos(5 phi)>/<r^n>
epart6,           //!< counter for participant-plane <r^n cos(6 phi)>/<r^n>

nwounded,         //!< counter for number of wounded objects
nbinary,          //!< counter for number of binary collisions
nhot,             //!< counter for number of hot-spots
nweight;          //!< counter for relative deposited strength (RDS)

counter_2D 
angles;           //!< counter for forward-backward reaction-plane angle correlations

Int_t evall,      //!< number of all attempted event
       roo,       //!< Fourier rank for the rotation axis
       ppp,       //!< power of the weight in eccentricty definition
        kk;       //!< number of the current event

Float_t 
d,         //!< the wounding distance
dbin,      //!< the binary-collision distance 
b,         //!< impact parameter 
sitot,     //!< the total A+B cross section in the acceptance window
sirad,     //!< equivalent hard-sphere radius for the cross section
rwA,       //!< number of wounded objects in A
rwB,       //!< number of wounded objects in B
rwAB,      //!< number of all wounded objects
rbin,      //!< number of binary collisions
rhotspot,  //!< number of hot-spots
rpa,       //!< relative deposited strength (RDS)
sizeav,	   //!< size

//! participant-plane eccentricities
ep1s,      
eps,       
ep3s,      
ep4s,      
ep5s,     
ep6s,      

//! angles of the principal axis
phirot1,   
phirot,    
phirot3,   
phirot4,  
phirot5,   
phirot6, 

xx,        //!< center-of-mass x coordinate
yy,        //!< center-of-mass y coordinate
xepp,      //!< average ep 
xsepp,     //!< standard deviation of ep
wfqA,      //!< number of wounded nucleons evaluated from the number of wounded partons in nucleus A
wfqB,      //!< number of wounded nucleons evaluated from the number of wounded partons in nucleus B
wfq;       //!< total number of wounded nucleons evaluated from the number of wounded partons

//! reset the counters used to store physical quantities in the event
void reset_counters(                  
                   ){
epart1.reset(); epart.reset(); epart3.reset(); epart4.reset(); epart5.reset(); 
epart6.reset(); nwounded.reset(); nbinary.reset(); nhot.reset(); nweight.reset(); angles.reset();
// nwAwB.reset(); 
                     };

//! class storing the trees and histograms
/*!
Class for storage of ROOT structures (trees, histograms) used for later off-line analysis within ROOT or other codes 
*/
class tr_his_c {
public:

// trees
    TTree *param,      //!< parameters 
          *phys,       //!< A+B cross section and other physical results
          *full_event, //!< full info on the event (positions and RDS of the sources)
          *tree;       //!< basic physical results

// histograms
	TH2D *xyhist,        //!< cartesian fixed-axes distribution
	     *xyhist_nuclA,  //!< cartesian fixed-axes distribution of density in nucleus A
             *xyhist_nuclB,  //!< cartesian fixed-axes distribution of density in nucleus B
             *rcostheta_nuclA,  //!< (r,cos(theta)) distribution of density in nucleus A
             *rcostheta_nuclB,  //!< (r,cos(theta)) distribution of density in nucleus B
             *xyhist_mantle, //!< cartesian fixed-axes mantle distribution
             *xyhist_core,   //!< cartesian fixed-axes core distribution
             *xyhistr;       //!< cartesian participant-plane distribution
 
// histograms for the dependence of certain quantities on the total number of wounded objects
	TH1D *nx,      //!< center-of-mass x coordinate of the source distribution vs. Nw
             *nx2,     //!< square of cm x coordinate, then its variance, vs. Nw 
             *ny,      //!< center-of-mass y coordinate of the source distribution vs. Nw
             *ny2,     //!< square of cm y coordinate, then its variance, vs. Nw
             *nsize,   //!< size vs. Nw
             *nsize2,  //!< square of size, then its variance/size^2, vs. Nw
             *nepsp,   //!< participant-plane eccentricity vs. Nw
             *nepsp2,  //!< square of participant-plane eccentricity, then its variance, vs. Nw
             *nepsp4,  //!< participant-plane fourth moment vs. Nw
             *nuni,    //!< frequency of Nw, i.e. histogram of unity vs. Nw
             *nuniRDS, //!< frequency of RDS, i.e. histogram of unity vs. RDS
             *nunib,   //!< frequency of b, i.e. histogram of unity vs. b
             *nwb,     //!< number of wounded objects in nucleus B vs. total number of wounded nucleons
             *nw2b;    //!< square of the number of wounded objects in nucleus B, then its variance, vs. total number of wounded nucleons
 
// histograms for fluctuations of number of wounded objects and RDS
	TH1D *nwei,   //!< RDS vs. number of wounded objects in nucleus A
             *nwei2,  //!< square of RDS, then its variance, vs. number of wounded objects in nucleus A
             *ntarg,  //!< number of wounded objects in nucles B vs. number of wounded objects in nucleus A
             *ntarg2, //!< square of the number of wounded objects in nucles B, then its variance, vs. number of wounded objects in nucleus A
             *nbinar, //!< number of binary collisions vs. number of wounded objects in nucleus A 
             *nbinar2,//!< square of the number of binary collisions, then its variance, vs. number of wounded objects in nucleus A  
             *nunp;   //!< frequency of the number of wounded objects in nucleus A


// histograms for nuclear profiles, correlations, and the weight distribution
    TH1D  *radA,      //!< one-body radial distribution in the nucleus A 
	  *radB,      //!< one-body radial distribution in the nucleus B 
          *rrelA,     //!< distance between the pair of objects in the nucleus A 
	  *rrelB,     //!< distance between the pair of objects in the nucleus B 
          *rrel_u,   //!< uncorrelated distance between the pair of objects in the nucleus (one nucleon from A, the other one from B) 
          *weih,     //!< the distribution overlaid on the wounded objects
          *weih_bin, //!< the distribution overlaid over binary collisions
          *wpro;     //!< the wounding profile

//#if(_clusters_) 
          TH2D  *radA2xy, *radA2xz, *radA2yz;
          TH3D  *radA3;		  
//#endif

//! initialize the histograms
    void init(){
        param = new TTree("param","param tree"); //!< tree storing parameters
        param->Branch("EVENTS",&EVENTS,"EVENTS/I");
        param->Branch("NCS",&NCS,"NCS/I");
        param->Branch("NBIN",&NBIN,"NBIN/I");
        param->Branch("FBIN",&FBIN,"FBIN/I");
        param->Branch("NUMA",&NUMA,"NUMA/I");
	param->Branch("PARTONS",&PARTONS,"PARTONS/I");
	param->Branch("CLUSTERS",&CLUSTERS,"CLUSTERS/I");		
	param->Branch("BETA2A",&BETA2A,"BETA2A/F");   
        param->Branch("BETA4A",&BETA4A,"BETA4A/F");   
        param->Branch("ROTA_THETA",&ROTA_THETA,"ROTA_THETA/F");  
        param->Branch("ROTA_PHI",&ROTA_PHI,"ROTA_PHI/F");        
        param->Branch("NUMB",&NUMB,"NUMB/I");
	param->Branch("BETA2B",&BETA2B,"BETA2B/F");   
        param->Branch("BETA4B",&BETA4B,"BETA4B/F");   
        param->Branch("ROTB_THETA",&ROTB_THETA,"ROTB_THETA/F");  
        param->Branch("ROTB_PHI",&ROTB_PHI,"ROTB_PHI/F");        
        param->Branch("WMIN",&WMIN,"WMIN/I");
        param->Branch("MODEL",&MODEL,"MODEL/I");
        param->Branch("W0",&W0,"W0/I");
        param->Branch("W1",&W1,"W1/I");
        param->Branch("RDS0",&RDS0,"RDS0/F");
        param->Branch("RDS1",&RDS1,"RDS1/F");
        param->Branch("ISEED",&ISEED,"ISEED/i");
        param->Branch("BMIN",&BMIN,"BMIN/F");
        param->Branch("BMAX",&BMAX,"BMAX/F");
        param->Branch("BTOT",&BTOT,"BTOT/F");
        param->Branch("RWSA",&RWSA,"RWSA/F");
        param->Branch("AWSA",&AWSA,"AWSA/F");
        param->Branch("RWSB",&RWSB,"RWSB/F");
        param->Branch("AWSB",&AWSB,"AWSB/F");
        param->Branch("WFA",&WFA,"WFA/F");
        param->Branch("WFB",&WFB,"WFB/F");
        param->Branch("SNN",&SNN,"SNN/F");
        param->Branch("SBIN",&SBIN,"SBIN/F");
        param->Branch("ALPHA",&ALPHA,"ALPHA/F");
        param->Branch("DOBIN",&DOBIN,"DOBIN/I");
        param->Branch("Uw",&Uw,"Uw/F");
        param->Branch("Ubin",&Ubin,"Ubin/F");
        param->Branch("Vw",&Vw,"Vw/F");
        param->Branch("Vbin",&Vbin,"Vbin/F");
        param->Branch("CD",&CD,"CD/F");
        param->Branch("SHIFT",&SHIFT,"SHIFT/I");
        param->Branch("RET",&RET,"RET/I");      
        param->Branch("DW",&DW,"DW/F");
        param->Branch("DBIN",&DBIN,"DBIN/F");
        param->Branch("NNWP",&NNWP,"NNWP/I");
        param->Branch("GA",&GA,"GA/F");
        param->Branch("FILES",&FILES,"FILES/I");
        param->Branch("PP",&PP,"PP/I");
        param->Branch("RO",&RO,"RO/I");
 	param->Branch("RCHA",&RCHA,"RCHA/F");  
	param->Branch("RCHB",&RCHB,"RCHB/F");  
	param->Branch("RCHP",&RCHP,"RCHP/F");  
        param->Branch("OMEGA",&OMEGA,"OMEGA/F");  
	param->Branch("GAMA",&GAMA,"GAMA/F");
        param->Branch("SCALEA",&SCALEA,"SCALEA/F");  
	param->Branch("SIGMAA",&SIGMAA,"SIGMAA/F");	
	param->Branch("SIGMABISA",&SIGMABISA,"SIGMABISA/F");
        param->Branch("SCALEB",&SCALEB,"SCALEB/F");  
	param->Branch("SIGMAB",&SIGMAB,"SIGMAB/F");	
	param->Branch("SIGMABISB",&SIGMABISB,"SIGMABISB/F");		
	param->Branch("QSCALE",&QSCALE,"QSCALE/F");
	param->Branch("DS",&DS,"DS/F");
	param->Branch("ECM",&ECM,"ECM/F");
    param->Branch("ver",&ver,"ver/F");

        phys = new TTree("phys","physical results"); //!< tree storing some physical quantities 
        phys->Branch("sitot",&sitot,"sitot/F");
        phys->Branch("eps",&xepp,"sepp/F");
        phys->Branch("sigma_eps",&xsepp,"xsepp/F");

        full_event = new TTree("full_event","full event"); //!< tree storing full info on the events
        full_event -> Branch("full_source",&tSource,"X:Y:W:KK");
     
        tree =  new TTree("events","event tree");         //!< tree storing basic information on events
        tree->Branch("nwA",&rwA,"nwA");    // wounded objects in A
        tree->Branch("nwB",&rwB,"nwB");    // wounded objects in B
        tree->Branch("nwAB",&rwAB,"nwAB"); // all wounded objects
        tree->Branch("nbin",&rbin,"nbin"); // binary collisions
        tree->Branch("npa",&rpa,"npa");    // source RDS
        tree->Branch("b",&b,"b");          // impact parameter

// participant-plane cos and sin moments
	tree->Branch("size",&sizeav,"size"); // <r>
        tree->Branch("ep1s",&ep1s,"ep1s");      // < r^3 cos((phi-phi*)) >
        tree->Branch("eps", &eps, "eps");       // < r^n cos(2 (phi-phi*)) >
        tree->Branch("ep3s",&ep3s,"ep3s");      // < r^n cos(3 (phi-phi*)) >
        tree->Branch("ep4s",&ep4s,"ep4s");      // < r^n cos(4 (phi-phi*)) >
        tree->Branch("ep5s",&ep5s,"ep5s");      // < r^n cos(5 (phi-phi*)) >
        tree->Branch("ep6s",&ep6s,"ep6s");      // < r^n cos(6 (phi-phi*)) >

// other quantities
        tree->Branch("phir1",&phirot1,"phir1");    // the rotation angle phi1*
        tree->Branch("phir", &phirot, "phir");     // the rotation angle phi*
        tree->Branch("phir3",&phirot3,"phir3");    // the rotation angle phi3*
        tree->Branch("phir4",&phirot4,"phir4");    // the rotation angle phi4*
        tree->Branch("phir5",&phirot5,"phir5");    // the rotation angle phi5*
        tree->Branch("phir6",&phirot6,"phir6");    // the rotation angle phi6*

        tree->Branch("xx",&xx,"x");             // x c.m. coordinate
        tree->Branch("yy",&yy,"y");             // y c.m. coordinate

        tree->Branch("wfqA",&wfqA,"wfqA");     
        tree->Branch("wfqB",&wfqB,"wfqB");     
        tree->Branch("wfq",&wfq,"wfq");     

        radA =  new TH1D("radA", "one-body distribution, nucleus A", NBIN, 0.000001, BTOT);  
	radB =  new TH1D("radB", "one-body distribution, nucleus B", NBIN, 0.000001, BTOT);  
        rrelA = new TH1D("rrelA", "relative distance - correlated, nucleus A", 2*NBIN, 0.000001, 2.*BTOT);
	rrelB = new TH1D("rrelB", "relative distance - correlated, nucleus B", 2*NBIN, 0.000001, 2.*BTOT);
        rrel_u = new TH1D("rrel_u", "relative distance - uncorrelated", 2*NBIN, 0.000001, 2.*BTOT);

//#if(_clusters_) 
	radA2xy =  new TH2D("radA2xy", "clusters in xy plane", NBIN, -BTOT, BTOT, NBIN, -BTOT, BTOT);  
	radA2xz =  new TH2D("radA2xz", "clusters in xz plane", NBIN, -BTOT, BTOT, NBIN, -BTOT, BTOT);  
	radA2yz =  new TH2D("radA2yz", "clusters in yz plane", NBIN, -BTOT, BTOT, NBIN, -BTOT, BTOT);  
	radA3 =  new TH3D("radA3", "clusters 3D", 30, -BTOT, BTOT, 30, -BTOT, BTOT,30, -BTOT, BTOT); 
//#endif

	weih = new TH1D("weih", "source weight distribution, wounded", 500, -0.1, 20.*Uw);
	weih_bin = new TH1D("weih_bin", "source weight distribution, binary", 500, -0.1, 20.*Ubin);
	wpro = new TH1D("wpro", "wounding profile", 100, 0.000001, 6.);

// histograms for the 2D profiles
	xyhist   = new TH2D("xyhist",  "fixed-axes source density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist -> SetXTitle("x"); xyhist -> SetYTitle("y");

//   x-y histograms of density for single (A, B) nucleus
        xyhist_nuclA   = new TH2D("xyhist_nuclA",  "fixed-axes source density NUCLA", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist_nuclA -> SetXTitle("x"); xyhist_nuclA -> SetYTitle("y");
        xyhist_nuclB   = new TH2D("xyhist_nuclB",  "fixed-axes source density NUCLB", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist_nuclB -> SetXTitle("x"); xyhist_nuclB -> SetYTitle("y");
        rcostheta_nuclA   = new TH2D("rcostheta_nuclA",  "(r,cos(theta)) source density NUCLA", NBIN, 0.0, 10.0, NBIN,-1.0,1.0); 
        rcostheta_nuclA -> SetXTitle("r"); rcostheta_nuclA -> SetYTitle("cos(theta)"); 
        rcostheta_nuclB   = new TH2D("rcostheta_nuclB",  "(r,cos(theta)) source density NUCLB", NBIN, 0.0, 10.0, NBIN,-1.0,1.0); 
        rcostheta_nuclB -> SetXTitle("r"); rcostheta_nuclB -> SetYTitle("cos(theta)"); 

	xyhist_mantle   = new TH2D("xyhist_mantle",  "participant-plane mantle density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist_mantle -> SetXTitle("x"); xyhist_mantle -> SetYTitle("y");
	xyhist_core   = new TH2D("xyhist_core",  "participant-plane core density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist_core -> SetXTitle("x"); xyhist_core -> SetYTitle("y");
	xyhistr  = new TH2D("xyhistr", "participant-plane source density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhistr -> SetXTitle("x"); xyhistr -> SetYTitle("y");       

Int_t ran=NCS*(NUMA+NUMB);
Int_t ranA=NCS*NUMA;

// range for the RDS histogram
Int_t ranr=ran*Uw+1;


// histograms for the dependence of certain quantities on the number of wounded objects
	nx = new TH1D("nx", "<x> vs. N_{w}", ran, 0.5, ran+0.5);
	nx2= new TH1D("nx2", "var(x) vs. N_{w}", ran, 0.5, ran+0.5); 
	ny = new TH1D("ny", "<y> vs. N_{w}", ran, 0.5, ran+0.5);
	ny2= new TH1D("ny2", "var(y) vs. N_{w}", ran, 0.5, ran+0.5); 
	nsize = new TH1D("nsize", "<r> vs. N_{w}", ran, 0.5, ran+0.5);
	nsize2 = new TH1D("nsize2", "var(<r>)/<<r>>^2 vs. N_{w}", ran, 0.5, ran+0.5);
	nepsp = new TH1D("nepsp", "#epsilon^{*}_{2} vs. N_{w}",  ran, 0.5, ran+0.5);
	nepsp4 = new TH1D("nepsp4", "#epsilon^{*}_{4} vs. N_{w}",  ran, 0.5, ran+0.5);
	nepsp2= new TH1D("nepsp2", "var(#epsilon*)/#epsilon*^{2} vs. N_{w}",  ran, 0.5, ran+0.5); 
	nuni = new TH1D("nuni", "event multiplicity vs. N_{w}",  ran, 0.5, ran+0.5);
	nuniRDS = new TH1D("nuniRDS", "event multiplicity vs. RDS",  ranr, 0.5, ranr+0.5);
	nunib = new TH1D("nunib", "inelasticity vs. b", 200, BMIN, BMAX);
	nwb = new TH1D("nwb", "N_{w} vs. b", 200, BMIN, BMAX);
	nw2b= new TH1D("nw2b", "var(N_{w}) vs. b", 200, BMIN, BMAX); 


// histograms for fluctuations of number of wounded objects and RDS
	nwei = new TH1D("nwei", "RDS/N_{w}^{A} vs. N_{w}^{A}", ranA, 0.5, ranA+0.5);
	nwei2= new TH1D("nwei2", "var(RDS)/RDS vs. N_{w}^{A}", ranA, 0.5, ranA+0.5); 
	ntarg = new TH1D("ntarg", "N_{w}^{B} vs. N_{w}^{A}", ranA, 0.5, ranA+0.5);
	ntarg2= new TH1D("ntarg2", "var(N_{w}^{B})/N_{w}^{B} vs. N_{w}^{A}", ranA, 0.5, ranA+0.5); 
	nbinar = new TH1D("nbinar", "N_{bin} vs. N_{w}^{A}", ranA, 0.5, ranA+0.5);
	nbinar2= new TH1D("nbinar2", "var(N_{bin})/N_{bin} vs. N_{w}^{A}", ranA, 0.5, ranA+0.5); 
	nunp = new TH1D("nunp", "event multiplicity vs. N_{w}^{A}", ranA, 0.5, ranA+0.5);
        };

//! fill trees param and phys
    void fill(){param->Fill();phys->Fill();}; 

//! fill the main tree
    void fill_tr(){tree->Fill();};

//! calculate eccentricity, size, etc. and their fluctuations vs. number of wounded objects or b 
    void fill_res(){        
        nx -> Fill(rwAB,xx);
        nx2 -> Fill(rwAB,xx*xx);
        ny -> Fill(rwAB,yy);
        ny2 -> Fill(rwAB,yy*yy);

	nsize ->  Fill(rwAB,sizeav);
	nsize2 -> Fill(rwAB,sizeav*sizeav);

// use smeared eccentricities
        nepsp -> Fill(rwAB,eps);
        nepsp4 -> Fill(rwAB,ep4s);
        nepsp2 -> Fill(rwAB,eps*eps);
        nuni -> Fill(rwAB,1);
        nuniRDS -> Fill(rpa,1);

	nwb -> Fill(b,rwAB);
	nw2b -> Fill(b,rwAB*rwAB);

// WB - change to get the b profile in p-p collisions
        nunib -> Fill(b,1./b);

// for multiplicity fluctuations
        nwei->Fill(rwA,rpa); 
        nwei2->Fill(rwA,rpa*rpa);
	ntarg->Fill(rwA,rwB);
	ntarg2->Fill(rwA,rwB*rwB);
	nbinar->Fill(rwA,rbin);
	nbinar2->Fill(rwA,rbin*rbin);
	nunp->Fill(rwA,1);};

//! write out trees param, phys, full_event, and the main tree
    void write(){param->Write();phys->Write();tree->Write();
                  if(FULL){full_event->Write();};
                };   

//! write out the radial density distribution an the pair distance distribution in the nucleus
    void write_r(){
		radA->Write();
//#if(_clusters_) 
		radA2xy->Write();radA2xz->Write();radA2yz->Write();radA3->Write();
//#endif
		radB->Write();rrelA->Write();rrelB->Write();rrel_u->Write();
                   xyhist_nuclA->Write();xyhist_nuclB->Write();rcostheta_nuclA->Write();rcostheta_nuclB->Write();
                  };
//! write out the overlaid distributions
    void write_w(){weih->Write();weih_bin->Write();};
 
//! write out the wounding profile
    void write_wpro(){wpro->Write();};

//! write out the histograms with the 2-dim distributions and the radial distributions of the Fourier components of the source profiles
    void write_d(){

	xyhist_mantle->Write(); 
	xyhist_core->Write(); 
	xyhistr->Write();

};
 
//! generate histograms of eccentricities and their variance, etc., vs. the number of wounded objects or b
   void gen(){
	nx->Divide(nx,nuni);
        nx->Write();
        nx2->Divide(nx2,nuni);
        nx->Multiply(nx,nx);
        ny->Divide(ny,nuni);
        ny->Write();
        ny2->Divide(ny2,nuni);
        ny->Multiply(ny,ny);
        ny2->Add(ny,-1);

	nsize -> Divide(nsize,nuni);
	nsize -> Write();
	nsize2-> Divide(nsize2,nuni);
	nsize -> Multiply(nsize,nsize);
	nsize2-> Add(nsize,-1);
	nsize2-> Divide(nsize2,nsize);
	nsize2-> Write();

        nepsp->Divide(nepsp,nuni);
        nepsp->Write();
        nepsp4->Divide(nepsp4,nuni);
        nepsp4->Write();

        nepsp2->Divide(nepsp2,nuni);
        nepsp->Multiply(nepsp,nepsp);
        nepsp2->Add(nepsp,-1);
        nepsp2->Divide(nepsp2,nepsp);

	nwb -> Divide(nwb,nunib);
	nwb -> Write();
	nw2b -> Divide(nw2b,nunib);
	nwb -> Multiply(nwb,nwb);
	nw2b -> Add(nwb,-1);

        nx2->Write();
        ny2->Write();
        nepsp2->Write();
        nuni->Write();
        nuniRDS->Write();
	nw2b->Write();
        nunib->Write();

	// for multiplicity fluctuations
	nwei->Divide(nwei,nunp);
	ntarg->Divide(ntarg,nunp);
	nbinar->Divide(nbinar,nunp);
        nwei->Write();
	ntarg->Write();
	nbinar->Write();
	   
 	nwei2->Divide(nwei2,nunp);
        nwei2->Divide(nwei2,nwei);
	nwei2->Add(nwei,-1);
  	nwei2->Write();
	   
 	ntarg2->Divide(ntarg2,nunp);
        ntarg2->Divide(ntarg2,ntarg);
	ntarg2->Add(ntarg,-1);
  	ntarg2->Write();
	   
 	nbinar2->Divide(nbinar2,nunp);
        nbinar2->Divide(nbinar2,nbinar);
	nbinar2->Add(nbinar,-1);
  	nbinar2->Write();
	   
	nunp->Write();};

};



/**************************
  Helper and print blocks
**************************/

//! print the version or brief help
void helper(
   Int_t argc, //!< number of command line parameters
   char* str   //!< string parameter (-v for version, -h for brief help)
           )
{if(argc > 1){
 if(strcmp(str,"-v")==0){cout << "GLISSANDO 3 ver: " << ver << endl; exit(0);};
 if(strcmp(str,"-h")==0){cout << endl << "Usage:" << endl;
 cout << "./glissando3 <input_file> <output_Root_file>  (for random nuclear distributions, _files_=0)" << endl;
 cout << "./glissando3 <input_file> <output_Root_file> <nucl_dictribution_A> <nucl_distribution_B>   (for external nuclear distributions, _files_=1)" << endl;             
 cout << endl
				 <<       "for ver. 3 (current) see Comp. Phys. Comm. 245 (2019) 106850, http://arxiv.org.abs/1901.04484," << endl
                                 << "for ver. 2 see Comp. Phys. Comm. 185 (2014) 1759, http://arxiv.org.abs/1310.5475," << endl
                                 << "for ver. 1 see Comp. Phys. Comm. 180 (2009) 69, http://arxiv.org.abs/0710.5731" << endl
                                 << endl << endl; exit(0);};
};};

//! print the header in the console output
void header(){
cout << endl << "*********************************************************" << endl;
cout << "GLISSANDO 3 ver. " << ver << endl << 
"ver. 3: Computer Physics Communications 245 (2019) 106850, arXiv:1901.04484" << endl <<
"ver. 2: Computer Physics Communications 185 (2014) 1759, arXiv:1310.5475" << endl <<
"ver. 1: Computer Physics Communications 180 (2009)   69, arXiv:0710.5731" << endl; 
//cout << "(tested with ROOT ver. 5.34)" << endl;
cout         << "**********************************************************" << endl;
cout << "Simulation of nuclear collisions in Glauber models" << endl; 
cout << "----------------------------------------------------------" << endl;
#if(_profile_) 
cout << "generation of the nuclear profile and NN correlations" << endl; 
cout << "----------------------------------------------------------" << endl;
#endif
#if(_weight_) 
cout << "generation of the superimposed distribution and NN collision profiles" << endl; 
cout << "---------------------------------------------------------" << endl;
#endif
#if(_rapidity_) 
cout << "generation of rapidity distributions" << endl; 
cout << "---------------------------------------------------------" << endl;
#endif
};


//! print epilog to the output
void epilog(
           ){
// output  basic results to the console
      cout << "Some quantities for the specified acceptance window " << endl 
           << "(+/- gives the e-by-e standard deviation):" << endl;
      if(BMIN*BMIN<0.0001) {cout << "A+B cross section = " << sitot << "mb  (makes sense for min. bias)";};
//      if((NUMA>2) && (BMIN*BMIN<0.0001)) {cout << ", equiv. hard-sphere radius = " << sirad << "fm";};
      cout << endl << "efficiency (accepted/all) = " << 100.*EVENTS/evall << "%" << endl;
      cout << "N_w = " << nwounded.mean()  << "+/-" << sqrt(nwounded.var()) << endl;
      cout << "RDS: " << nweight.mean()  << "+/-" << sqrt(nweight.var()) << endl;

              };

//! start the time measurement and print the stamp
Int_t time_start(){
time_t rawtime; 
struct tm * timeinfo;
time ( &rawtime ); 
timeinfo = localtime ( &rawtime ); 
Int_t ti=Int_t(rawtime);
cout << "Start: " << asctime (timeinfo); // time stamp
cout << "--------------------------------------" << endl;
return ti;
};

//! stop the time measurement and print the stamp
void time_stop(
              Int_t ts //!< time at start
              ){
time_t rawtime; 
struct tm * timeinfo;
cout << endl; time ( &rawtime ); timeinfo = localtime ( &rawtime ); 
Int_t ti=Int_t(rawtime) - ts; Int_t tig=ti/3600; ti=ti-tig*3600; Int_t tim=ti/60; ti=ti-tim*60;
cout << "Finish: " << asctime (timeinfo) << "(" << tig << "h:" << tim <<"m:" << ti << "s)" << endl;
cout << "**************************************" << endl << endl;
}


/*****************************
random generator functions
*****************************/


//! random number generator using the built-in ROOT generator, uniform on (0,1)
Float_t los() {return raa.Uniform(0,1);}; 

//! random number generator for spin (3/2,3/2) projection
Float_t los32(){Int_t x=0;while(x==0){Float_t rr=2.*los()-1.;Float_t b=los();
	       if(b<1.-rr*rr){x=1;return rr;};};}; 

//! random number generator for spin (3/2,1/2) projection
Float_t los12(){Int_t x=0;while(x==0){Float_t rr=2.*los()-1.;Float_t b=los();
	       if(b<(1.+3.*rr*rr)/4.){x=1;return rr;};};}; 

//! random number generator for the Woods-Saxon (or Fermi) distribution - nucleus A
Float_t rlosA(){Int_t x=0;while(x==0){Float_t rr=3.*RWSA*los();Float_t b=RWSA*RWSA*los();
	       if(b<rr*rr*(1.+WFA*rr*rr/RWSA/RWSA)/(exp((rr-RWSA)/AWSA)+1)){x=1;return rr;};};}; // Woods-Saxon/Fermi

//! random number generator for the Woods-Saxon (or Fermi) distribution - nucleus B
Float_t rlosB(){Int_t x=0;while(x==0){Float_t rr=3.*RWSB*los();Float_t b=RWSB*RWSB*los();
	       if(b<rr*rr*(1.+WFB*rr*rr/RWSB/RWSB)/(exp((rr-RWSB)/AWSB)+1)){x=1;return rr;};};}; //Woods-Saxon/Fermi

//! random number generator for the deformed Woods-Saxon or Fermi distribution
/*! random number generator for the Woods-Saxon or Fermi distribution (deformed with beta2, beta4 parameters of the spherical 
  harmonics Y20, Y40) - nucleus A */
Float_t rlosA_def(
                  Float_t* cth_pointerA, /*!< cos(theta) */
                  Float_t beta2, 	 /*!< beta_2 nuclear deformation parameter */
                  Float_t beta4		 /*!< beta_4 nuclear deformation parameter */
                 )
               {Int_t x=0;
                while(x==0)
                {Float_t rr=3.*RWSA*los(); 
                 Float_t ctheta =2*los()-1;
                 Float_t y20=sqrt(5.0/(4.0*PI))*((3.0*ctheta*ctheta-1.0)/2.0); 
                 Float_t y40=sqrt(9.0/(4.0*PI))*((35.0*ctheta*ctheta*ctheta*ctheta-30.0*ctheta*ctheta+3.0)/8.0); 
                 Float_t WS_def=rr*rr*(1.0+WFA*rr*rr/RWSA/RWSA)/(1 + exp( (rr-RWSA)/AWSA-RWSA*(beta2*y20+beta4*y40)/AWSA)  );
                 Float_t b=RWSA*RWSA*los();
//                 cout<<" ctheta= "<<ctheta<<endl;
                 if(b<WS_def) 
{x=1;
// cout<<" rlosA_def, b= "<<b<<" rr= "<<rr<<" ctheta= "<<ctheta<<endl;
*cth_pointerA=ctheta;
return rr;};
                };
               }; // Woods-Saxon DEFORMED/Fermi


//! random number generator for the deformed Woods-Saxon or Fermi distribution
/*! random number generator for the Woods-Saxon or Fermi distribution (deformed with beta2, beta4 parameters of the spherical 
  harmonics Y20, Y40) - nucleus B */
Float_t rlosB_def(
                  Float_t* cth_pointerB, /*!< cos(theta) */
                  Float_t beta2, 	 /*!< beta_2 nuclear deformation parameter */
                  Float_t beta4		 /*!< beta_4 nuclear deformation parameter */
                 )
               {Int_t x=0;
                while(x==0)
                {Float_t rr=3.*RWSB*los(); 
                 Float_t ctheta=2*los()-1;
                 Float_t y20=sqrt(5.0/(4.0*PI))*((3.0*ctheta*ctheta-1.0)/2.0);
                 Float_t y40=sqrt(9.0/(4.0*PI))*((35.0*ctheta*ctheta*ctheta*ctheta-30.0*ctheta*ctheta+3.0)/8.0); 
                 Float_t WS_def=rr*rr*(1.0+WFB*rr*rr/RWSB/RWSB) / (1 + exp( (rr-RWSB)/AWSB-RWSB*(beta2*y20+beta4*y40)/AWSB)  );
                 Float_t b=RWSB*RWSB*los();
//                 cout<<" ctheta= "<<ctheta<<endl;
                 if(b<WS_def)
{x=1;
// cout<<" rlosB_def, b= "<<b<<" rr= "<<rr<<" ctheta= "<<ctheta<<endl;
*cth_pointerB=ctheta;
return rr;};
                };
               }; // Woods-Saxon DEFORMED/Fermi

//! random number generator for distribution with a hole in the middle
Float_t rlos_hole(
		   Float_t s /*!< size scale parameter */
                 ){
Int_t x=0;
while(x==0){
 Float_t rr=10.*los();
 Float_t b=2.2*s*s*s*s*los();
 Float_t hos;
 hos=rr*rr*exp(-rr*rr/(2.*s*s));
 if(b<rr*rr*hos){
 x=1;return rr;
};
};
}; 	   


//! random number generator for distribution in the alpha nucleus - not used
Float_t rlos_alpha(Float_t s,Float_t bp,Float_t dp){
Int_t x=0;
while(x==0){
 Float_t p=2.25; //alpha tail parameter
 Float_t r0=2.5;
 Float_t rr=10.*los();
 Float_t b=2*los();
 Float_t hos;
 if(rr<r0)  {hos=(1.+bp*bp*rr*rr)/(1.+dp*dp*rr*rr*rr*rr)*exp(-rr*rr/(2.*s*s));}; 
 if(rr>=r0) {hos=(1.+bp*bp*r0*r0)/(1.+dp*dp*r0*r0*r0*r0)*exp(p*(r0-rr)-r0*r0/(2.*s*s))*r0*r0/rr/rr;};
 if(b<rr*rr*hos){
 x=1;return rr;
};
};
}; 


//! random number generator for distribution of nucleons in the alpha cluster
Float_t rlos_abf(
                  Float_t sc, /*!< scale parameter */
                  Float_t scp /*!< scale parameter */
                ){
Int_t x=0;
while(x==0){

 Float_t p=2.25/scp;   // alpha tail parameter rescaled
 Float_t r0=2.5*sc;    // best parameters rescaled
 Float_t s=1.028*sc;
 Float_t bp=0.795/sc;
 Float_t dp=0.919/sc/sc;

 Float_t rr=10.*los();
 Float_t b=2.3*los();
 Float_t hos;
 if(rr<r0)  {hos=(1.+bp*bp*rr*rr)/(1.+dp*dp*rr*rr*rr*rr)*exp(-rr*rr/(2.*s*s));}; 
 if(rr>=r0) {hos=(1.+bp*bp*r0*r0)/(1.+dp*dp*r0*r0*r0*r0)*exp(p*(r0-rr)-r0*r0/(2.*s*s))*r0*r0/rr/rr;};
 if(b<rr*rr*hos){
 x=1;return rr;
};
};
}; 



//! random number generator for the harmonic oscillator shell model density - nucleus A
/*! The harmonic oscillator shell distribution used to generate the distance 
between nucleons in light (2<NUMA<17) nuclei (Nuclear Sizes. L. R. B. Elton, Oxford University Press, New York, 1961.)*/		   
Float_t rlosA_hos(){
Int_t x=0;
while(x==0){
 Float_t rr=10.*los();
 Float_t b=10.*los();
 Float_t ddd2=2.5-(4./(Float_t)NUMA);
 ddd2=1./ddd2;
 ddd2=ddd2*(RCHA-RCHP);
 Float_t hos=1.0;
 hos=hos*(1.+(((Float_t)NUMA-4.)/6.)*rr*rr/ddd2);
 hos=hos*exp(-rr*rr/ddd2);
 if(b<rr*rr*hos){
 x=1;return rr;
};
};
}; 

//! random number generator for the harmonic oscillator shell model density - nucleus B
/*! The harmonic oscillator shell distribution used to generate the distance 
between nucleons in light (2<NUMB<17) nuclei (Nuclear Sizes. L. R. B. Elton, Oxford University Press, New York, 1961.)*/		   
Float_t rlosB_hos(){
Int_t x=0;
while(x==0){
 Float_t rr=10.*los();
 Float_t b=10.*los();
 Float_t ddd2=2.5-(4./(Float_t)NUMB);
 ddd2=1./ddd2;
 ddd2=ddd2*(RCHB-RCHP);
 Float_t hos=1.0;
 hos=hos*(1.+(((Float_t)NUMB-4.)/6.)*rr*rr/ddd2);
 hos=hos*exp(-rr*rr/ddd2);
 if(b<rr*rr*hos){
 x=1; return rr;
};
};
}; 
		   
//! random number generator for the Hulthen distribution
/*! The Hulthen distribution used to generate the distance between nucleons in the deuteron */
Float_t rlos_hult(){double const ah=.228;double const bh=1.18;Int_t x=0;while(x==0){Float_t rr=15.*los();
Float_t b=.25*los(); // 0.25 is an upper bound of the probability distribution
 if(b< 2*ah*bh*(ah+bh)*(exp(-2.*ah*rr)+exp(-2.*bh*rr)-2.*exp(-(ah+bh)*rr))/(ah-bh)/(ah-bh)){x=1;return rr;};};}; //Hulthen


//! random number generator for the Gamma distribution
Double_t gamgen(
            Float_t a //!< the parameter a in f(x) = x^(a-1) exp(-x)/Gamma(a)
            ){
Int_t k = Int_t(a);
Double_t b = a - k;
Double_t x1 = 0.0;
Double_t u=0.0;
Double_t v=0.0;
Double_t x2=0.0;
Double_t s;

if(k==0) goto blok;
s = 1.0 ;
for (Int_t i=1;i<=k;i++)
s=s*los();
x1 = -log(s);

if (b>1e-6) goto blok;
return x1;
 
blok:x2=-log(los());

bloka:u=pow(los(),1.0/b);
v=u+pow(los(),1.0/(1.0-b));
if (v>1.0) goto bloka;
return (x1+x2*u/v);
}//gamgen


//! random number generator for the negative binomial distribution
Int_t negbin(Double_t m,Double_t v) {
Double_t p=1.0-(m/v);
Double_t r=m*(1.0-p)/p;
Double_t x=gamgen(r)*p/(1.0-p);
Int_t n=raa.Poisson(x);
return n;
}//negbin
 

//! random radial coordinate of a parton in the nucleon
Float_t rlos_parton() {

#if(_pardis_==0)
//! gamma distribution f(r)=r^2*exp(-r/wid) 
Float_t u = gamgen(3.0)*QSCALE *sqrt(2.*NCS/(3.*NCS-3.));
#endif

#if(_pardis_==1)
//! Gaussian distribution f(r)=r^2*exp(-r^2/(2 wid^2)), 1-1/NCS is the center-of-mass correction 
 Float_t wid=sqrt(QSCALE*((double) NCS)/(((double) NCS)-1.));
  Float_t x = raa.Gaus(0.0,wid);
  Float_t y = raa.Gaus(0.0,wid);
  Float_t z = raa.Gaus(0.0,wid);
  Float_t u=sqrt(x*x+y*y+z*z);
#endif

return u;
};
 
 

//! statistial distribition overlaid on RDS
// no rescaling with the mean u (!)
Float_t dist(
           Int_t m,  //!< case: 0 - none, 1 - Poisson, 2 - Gamma distribution, 3 - Negative binomial distribution
           Float_t u, //!< average of the distribution in cases 1, 2 and 3
           Float_t v //!< variance of the distribution in case 3, v>u

          ){
 switch(m) {
 case 0: { return u; break; }
 case 1: { return raa.Poisson(u);  break; } 
 case 2: { return gamgen(u*u/(v-u))*(v-u)/u; break; }
 case 3: { return negbin(u,v); break;} 
}
}

//! random shift of the source location
/*! The location of the source may by shifted randomly when DW>0 od DBIN>0, with the Gaussian distribution of the width w. */
Float_t disp(
          Float_t w //! average shift of the source location, Gaussian distribution 
          ){return raa.Gaus(0.,w);};


//! linear interpolation for the function Sqrt[Pi] Exp[-1/(2 x^2)] BesselI[0, 1/(2 x^2)]/x in the conformal variable t defined via x=2t/(1-t)
// The table of values generated with Mathematica at 41 equidistant points for t in [0,1]    
Float_t F0(
Float_t t /*!< conformal variable t=x(2+x) for interpolation */
){

static const Float_t tF0[41]={1, 1.00066, 1.00281, 1.00678, 1.01312, 1.02273, 1.03759, 1.06101, 1.09363, 1.12937, 1.15863, 1.17386, 1.17182, 1.15293, 1.11971, 1.07545, 1.02338, 0.966318, 0.90652, 0.84571, 0.785151, 0.725732, 0.668058, 0.612512, 0.559321, 0.508596, 0.460365, 0.414601, 0.37124,
0.33019, 0.291348, 0.254602, 0.219836, 0.186935, 0.155786, 0.126281, 0.0983179, 0.0717972, 0.0466274, 0.0227219, 0};

int i=t*40; if(i>40){i=40;}; if(i<0){i=0;};
Float_t d=40*t-i;
return tF0[i]*(1-d)+tF0[i+1]*d;
};


//! linear interpolation for the function Sqrt[Pi] Exp[-1/(2 x^2)] BesselI[1, 1/(2 x^2)]/x in the conformal variable t defined via x=2t/(1-t)
// The table of values generated with Mathematica at 41 equidistant points for t in [0,1]    
Float_t F1(
Float_t t /*!< conformal variable t=x(2+x) for interpolation */
){

static const Float_t tF1[41]={1, 0.998024, 0.991631, 0.979937, 0.961704, 0.935042, 0.896533, 0.840465, 0.76311, 0.668484, 0.56649, 0.467172, 0.377311, 0.300056, 0.235927, 0.183951, 0.142508, 0.109833, 0.0842698, 0.0643828, 0.0489763, 0.0370815, 0.0279263, 0.0209018, 0.0155307, 0.0114405, 0.00834104, 0.00600654, 0.0042614, 0.00296904, 0.0020232, 0.00134121, 0.000858728, 0.000525698, 0.000303216, 0.000161073, 0.0000758625, 0.0000295003, 0.00000807, 0.00000093, 0};

int i=t*40; 
Float_t d=40*t-i; if(i>40){i=40;}; if(i<0){i=0;};
return tF1[i]*(1-d)+tF1[i+1]*d;
};
		  
#endif

