/** \file counter.h
 * Part of GLISSANDO
 * 
 */

/*! \mainpage
                                                                                         
            GLISSANDO 2 - GLauber Initial State Simulation AND mOre...       \n
                          ver. 2.601, 18 May 2013                              \n
                                                                             \n 

																			 
  Authors: 
           - Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              
           - Maciej Rybczynski   (Maciej.Rybczynski@ujk.edu.pl)            
           - Grzegorz Stefanek   (Grzegorz.Stefanek@ujk.edu.pl)   
           - Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                    \n  
                                                                             \n
  
  Modification of the code to ver. 2.601.

  For the detailed description of ver. 1 program and further references         
  to the description of the model, please, refer our                                                                                                   
  Computer Physics Communications 180(2009)69, arXiv:0710.5731 [nucl-th]     \n
  accessibile from: http://arxiv.org.abs/0710.5731                           \n
                                                                             \n
  Implementation of nuclear correlations in ver. 2 as described in 
  Phys. Rev. C81(2010)064909                                                 \n
                                                                             \n
  Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      \n
  
  GLISSANDO is a Glauber Monte-Carlo generator for early-stages of relativistic
  heavy-ion collisions, written in c++ and interfaced to ROOT. Several models
  are implemented: the wounded-nucleon model, the binary collisions model,
  the mixed model, and the model with hot-spots. The original geometric
  distribution of sources (i.e., wounded nucleons or binary collisions) in
  the transverse plane can be superimposed with a statistical distribution
  simulating the dispersion in the generated transverse energy in each individual
  collision. The program generates inter alia the variable-axes (participant)
  two-dimensional profiles of the density of sources in the transverse plane
  and their Fourier components. These profiles can be used in further analyses
  of physical phenomena, such as the the jet quenching, event-by-event hydrodynamics,
  or analyses of the elliptic flow and its fluctuations. Characteristics of the event
  (multiplicities, eccentricities, Fourier shape coefficients, etc.) are evaluated
  and stored in a ROOT file for further off-line studies. A number of scripts
  is provided for that purpose.
  The code can also be used for the proton-nucleus and deuteron-nucleus collisions. \n
                                                                             \n

  Version 2 of GLISSANDO offers much more functionality than version 1, 
  moreover, it is fully object-oriented, providing the user with the flexibility 
  of inspecting and, if needed, modyfying the code in a simple manner. New 
  features involve:                                                          \n
                                                                             \n
- The possibility of feeding into the simulations the nuclear distributions 
  accounting for the two-body NN correlations (read from external files, see
  Alvioli, Drescher and Strikman, [Phys. Lett. B680, 225, 2009], the distributions 
  can be found at http://www.phys.psu.edu/~malvioli/eventgenerator/ )  
- The use of the Gaussian NN wounding profile (which is more realistic than 
  the commonly-used hard-core wounding profile, see the analysis by Bialas 
  and Bzdak [Acta Phys. Polon. B38, 159,2007]) 
- The generation of the core-mantle (core-corona) distributions (see Bozek 
  [Acta Phys. Polon. B36, 3071,2005] and Werner [Phys. Rev. Lett. 98, 152301, 
  2007], see also Becattini and Manninen [Phys. Lett. B673, 19, 2009] and Bozek 
  [Phys. Rev. C79, 054901, 2009]) 
- The analysis of the triangular shape deformation parameter and profiles, 
  relevant for the triangular flow, see Alver and Roland, [Phys. Rev. C81, 054905, 
  2010] and Alver, Gombeaud, Luzum,and Ollitrault, [arXiv:1007.5469] 
- Generation of rapidity distributions in the wounded-nucleon picture according 
  to the model of Bialas and Czyz [Acta Phys.Polon.B36:905-918,2005], as implemented 
  by Bozek [arXiv:1002.4999]. This allows to obtain the fully 3-dimensional
  distribution of matter in the early Glauber phase of the collision.        
- The generation of initial distributions of nucleons in deformed nuclei 
  according to the deformed Woods-Saxon density with default deformation 
  parameters taken from P.Moller,J.R.Nix,W.D.Mayers, and W.J.Swiatecki 
  [Nucl.Data Tables 59,185,1995].See also W.Broniowski,M.Rybczynski,G.Stefanek
  [Phys.Rev. C87, 044908, 2013; arXiv:1211.2537]                                      \n 
                                                                          \n \n
  The reference manual for ver. 2, generated by Doxygen, is supplied at the home
  page. The full write-up of ver. 2 is under preparation.                    \n
                                                                             \n
  The code can be freely used and redistributed. However, if you decide to  
  make modifications, the authors would appreciate notification for the record.             
  Any publication or display of results obtained using GLISSANDO must        
  include a reference to our published paper.                                

*/

#ifndef _GL_COUNTER
  #define _GL_COUNTER  


/*******************************
 counting classes
*******************************/

//! simplest counting class
class counter {
private:
    Int_t count; //!< number of entries
    Double_t value; //!< sum of values counted
public:
//! constructor
    counter(){count=0;value=0;};
//! destructor
    ~counter(){};
//! reset the counter 
    void reset(){count=0;value=0;}; 
//! add entry to the counter
    void add(
            Double_t s //!< value added
            ){count++; value=value+s;};
//! get the number of entries
    Int_t getN(){return count;};
//! get the sum of values
    Double_t get(){return value;};
//! get the mean value
    Double_t mean(){return value/count;};
};

//! counting class with variance
class counter2 {
private:
  Int_t     count;  //!< number of entries
  Double_t  value,  //!< sum of values
            value2; //!< sum of squares of values
public:
//! constructor
    counter2(){count=0;value=0;value2=0;};
//! destructor
    ~counter2(){};
//! reset the counter 
    void reset(){count=0;value=0;value2=0;};
//! add entry
    void add(
            Double_t s //!< value added
            ){count++; value=value+s; value2=value2+s*s;};
//! get the number of entries
    Int_t getN(){return count;};
//! get the sum of values
    Double_t get(){return value;};
//! get the sum of squares of values
    Double_t get2(){return value2;};
//! get the mean value
    Double_t mean(){return value/count;};
//! get the variance
    Double_t var(){return value2/(count-1)-value/count*value/(count-1);}
//! get the variance multiplied with (N-1)/N
    Double_t vara(){return value2/count-value/count*value/count;};};

//! 2-dimensional counting class
class counter_2D {
private:
  Int_t count;          //!< number of entries
  Double_t     valuex,  //!< sum of x values
               valuey,  //!< sum of y values
               valuex2, //!< sum of x^2
               valuey2, //!< sum of y^2
               valuexy; //!< sum of x*y
public:
//! constructor
    counter_2D(){count=0;valuex=0;valuey=0;valuex2=0;valuey2=0;valuexy=0;};
//! destructor
    ~counter_2D(){};
//! reset the counter 
    void reset(){count=0;valuex=0;valuey=0;valuex2=0;valuey2=0;valuexy=0;};
//! add entry
    void add(
            Double_t sx, //!< x value added
            Double_t sy  //!< y value added
            ){count++; valuex+=sx; 
                       valuey+=sy; 
                       valuex2+=sx*sx;
                       valuey2+=sy*sy;
                       valuexy+=sx*sy;
             };
//! get the number of entries
    Int_t getN(){return count;};
//! get the sum of x values
    Double_t get_x(){return valuex;};
//! get the sum of x values
    Double_t get_y(){return valuey;};
//! get the sum of x2
    Double_t get_x2(){return valuex2;};
//! get the sum of y2
    Double_t get_y2(){return valuey2;};
//! get the sum of xy
    Double_t get_xy(){return valuexy;};
//! get the mean x value
    Double_t mean_x(){return valuex/count;};
//! get the mean y value
    Double_t mean_y(){return valuey/count;};
//! get the variance of x
    Double_t var_x(){return valuex2/(count-1)-valuex/count*valuex/(count-1);};
//! get the variance of y
    Double_t var_y(){return valuey2/(count-1)-valuey/count*valuey/(count-1);};
//! get the covariance
    Double_t cov(){return valuexy/(count-1)-valuex/count*valuey/(count-1);};
//! get the correlation
    Double_t corr(){return cov()/sqrt(var_x()*var_y());};

};

#endif

