/** \file retrieve.cpp
 * auxilliary file, part of GLISSANDO 2
 * 
 */


/*! \mainpage
                                                                                         
            GLISSANDO - GLauber Initial State Simulation AND mOre...         \n
                        ver. 2.07, 25 August 2010                            \n
                                                                             \n 
  Authors: 
           - Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              
           - Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              
           - Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                    \n  
                                                                             \n
  
  Modification of the code to ver. 2 by WB.

  For the detailed description of ver. 1 program and further references         
  to the description of the model, please, refer to our                                                                                                   
  Computer Physics Communications 180(2009)69, arXiv:0710.5731 [nucl-th]     \n
  accessibile from: http://arxiv.org.abs/0710.5731                           \n
                                                                             \n
  Implementation of nuclear correlations in ver. 2 as described in 
  Phys. Rev. C81(2010)064909                                                 \n
                                                                             \n
  Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      \n
  
  GLISSANDO is a Glauber Monte-Carlo generator for early-stages of relativistic 
  heavy-ion collisions, written in c++ and interfaced to ROOT. Several models 
  are implemented: the wounded-nucleon model, the binary collisions model, 
  the mixed model, and the model with hot-spots. The original geometric 
  distribution of sources (i.e., wounded nucleons or binary collisions) in 
  the transverse plane can be superimposed with a statistical distribution 
  simulating the dispersion in the generated transverse energy in each individual 
  collision. The program generates inter alia the fixed axes (standard) 
  and variable-axes (participant) two-dimensional profiles of the density 
  of sources in the transverse plane and their Fourier components. These profiles 
  can be used in further analyses of physical phenomena, such as the the jet 
  quenching, event-by-event hydrodynamics, or analyses of the elliptic flow 
  and its fluctuations. Characteristics of the event (multiplicities, eccentricities, 
  Fourier shape coefficients, etc.) are evaluated and stored in a ROOT file 
  for further off-line studies. A number of scripts is provided for that purpose. 
  The code can also be used for the proton-nucleus and deuteron-nucleus collisions. \n
                                                                             \n

  Version 2 of GLISSANDO offers much more functionality than version 1, 
  moreover, it is fully object-oriented, providing the user with the flexibility 
  of inspecting and, if needed, modyfying the code in a simple manner. New 
  features involve:                                                          \n
                                                                             \n
- The possibility of feeding into the simulations the nuclear distributions 
  accounting for the two-body NN correlations (read from external files, see
  Alvioli, Drescher and Strikman, [Phys. Lett. B680, 225, 2009], the distributions 
  can be found at http://www.phys.psu.edu/~malvioli/eventgenerator/ )  
- The use of the Gaussian NN wounding profile (which is more realistic than 
  the commonly-used hard-core wounding profile, see the analysis by Bialas 
  and Bzadak [Acta Phys. Polon. B38, 159,2007]) 
- The generation of the core-mantle (core-corona) distributions (see Bozek 
  [Acta Phys. Polon. B36, 3071,2005] and Werner [Phys. Rev. Lett. 98, 152301, 
  2007], see also Becattini and Manninen [Phys. Lett. B673, 19, 2009] and Bozek 
  [Phys. Rev. C79, 054901, 2009]) 
- The analysis of the triangular shape deformation parameter and profiles, 
  relevant for the triangular flow, see Alver and Roland, [Phys. Rev. C81, 054905, 
  2010] and Alver, Gombeaud, Luzum,and Ollitrault, [arXiv:1007.5469] 
- Generation of rapidity distributions in the wounded-nucleon picture according 
  to the model of Bialas and Czyz [Acta Phys.Polon.B36:905-918,2005], as implemented 
  by Bozek [arXiv:1002.4999]. This allows to obtain the fully 3-dimensional
  distribution of matter in the early Glauber phase of the collision.        \n 
                                                                          \n \n
  The reference manual for ver. 2, generated by Doxygen, is supplied at the home
  page. The full write-up of ver. 2 is under preparation.                    \n
                                                                             \n
  The code can be freely used and redistributed. However, if you decide to  
  make modifications, the authors would appreciate notification for the record.             
  Any publication or display of results obtained using GLISSANDO must        
  include a reference to our published paper.                                

*/

#include <math.h>
#include <TH1D.h>
#include <TFile.h>
#include <TTree.h>
#include <TChain.h>

#include <fstream>
#include <iostream>
#include <sstream>


using namespace std;

typedef struct {Float_t X,Y,W; UInt_t KK;} SOURCE;

/*! The code serves as an example of using the full spatial distribution of sources, which 
    optionally may be generated in the simulation. */
int main(
   int argc, 
char **argv //!< name of the GLISSANDO output ROOT file
        ){

// Default file name
TString inpfile("glissando.root");
if (argc>1) inpfile = argv[1];
TFile f(inpfile);


  SOURCE buf;

  int numev=0;            // number of event (all events)
  int curev=1;            // current event + 1 (in each file)
  int sw;                 // =1 when the event is changed, 0 otherwise
  int ilow, ihigh;        // range of entries in the event

  // kinematic variables for particle 1 and 2
  Float_t x, y, w, b, xav, xxav=0, xx2av=0;
  int iter, kk, m;
  float mav=0, m2av=0;


// for a chain of files
  TChain *itree = new TChain("full_event");
  TChain *etree = new TChain("events");
  itree->Add(inpfile);
  // itree->Add("glissando_2.root"); // adding more files with data
  etree->Add(inpfile);
  // etree->Add("glissando_2.root");

  itree->SetBranchAddress("full_source", &buf); // set the branch to the sources
  etree->SetBranchAddress("b",&b);              // set the branch to the impact parameter b

  cout << "got data" <<endl;
  cout << "will print sample info only for first 10 and last 10 events" << endl;

// go over all entries
  ihigh=0;
  // the trick is that when the buf.KK, describing the number of event, changes, we know we pass to the new event
  for (iter=0; iter < itree->GetEntries(); iter++)
  { ilow=ihigh;
    m=0; xav=0;
	itree->GetEntry(iter);
    sw=0;
    if (curev != buf.KK){curev = buf.KK;sw=1;numev++;ihigh=iter;};

// do at the end of each event
  if(iter == (itree->GetEntries())-1 ){numev++;ihigh=iter+1;}; // in the last entry
  if(sw || iter == (itree->GetEntries())-1 ){
    etree->GetEntry(numev-1); // get the impact parameter
    if(numev<=10 || numev>=(etree->GetEntries())-10){ // test output for the first and last 10 events
  cout << "event: " << numev << "  range for particle labels: " << ilow << " - " << ihigh-1 << "  b= " << b << endl; 
  cout.flush();
  };

// loop over sources
for(int j1=ilow;j1<ihigh;j1++){
  itree->GetEntry(j1); // here we have access to the sources
 m++; xav=xav+buf.X;
  } // end "loop over sources"

// a sample use of data
  xav=xav/m;
  mav=mav+m;
  m2av=m2av+m*m;
  xxav=xxav+xav;
  xx2av=xx2av+xav*xav;

  } //end "do at the end of each event"


 } // end "go over all entries"

cout << endl;
cout <<"# of sources=" << float(mav)/numev << "+/-" << sqrt((float)m2av/(numev-1)-(float)mav/numev*mav/(numev-1)) << endl;
cout <<"x c.m. coordinate=" << xxav/numev << "+/-" << sqrt(xx2av/(numev-1)-xxav/numev*xxav/(numev-1)) << endl;
} // end main


