/** \file functions.h
 * Part of GLISSANDO
 * 
 */


/*! \mainpage
                                                                                         
            GLISSANDO - GLauber Initial State Simulation AND mOre...         \n
                        ver. 2.07, 25 August 2010                            \n
                                                                             \n 
  Authors: 
           - Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              
           - Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              
           - Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                    \n  
                                                                             \n
  
  Modification of the code to ver. 2 by WB.

  For the detailed description of ver. 1 program and further references         
  to the description of the model, please, refer to our                                                                                                   
  Computer Physics Communications 180(2009)69, arXiv:0710.5731 [nucl-th]     \n
  accessibile from: http://arxiv.org.abs/0710.5731                           \n
                                                                             \n
  Implementation of nuclear correlations in ver. 2 as described in 
  Phys. Rev. C81(2010)064909                                                 \n
                                                                             \n
  Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      \n
  
  GLISSANDO is a Glauber Monte-Carlo generator for early-stages of relativistic 
  heavy-ion collisions, written in c++ and interfaced to ROOT. Several models 
  are implemented: the wounded-nucleon model, the binary collisions model, 
  the mixed model, and the model with hot-spots. The original geometric 
  distribution of sources (i.e., wounded nucleons or binary collisions) in 
  the transverse plane can be superimposed with a statistical distribution 
  simulating the dispersion in the generated transverse energy in each individual 
  collision. The program generates inter alia the fixed axes (standard) 
  and variable-axes (participant) two-dimensional profiles of the density 
  of sources in the transverse plane and their Fourier components. These profiles 
  can be used in further analyses of physical phenomena, such as the the jet 
  quenching, event-by-event hydrodynamics, or analyses of the elliptic flow 
  and its fluctuations. Characteristics of the event (multiplicities, eccentricities, 
  Fourier shape coefficients, etc.) are evaluated and stored in a ROOT file 
  for further off-line studies. A number of scripts is provided for that purpose. 
  The code can also be used for the proton-nucleus and deuteron-nucleus collisions. \n
                                                                             \n

  Version 2 of GLISSANDO offers much more functionality than version 1, 
  moreover, it is fully object-oriented, providing the user with the flexibility 
  of inspecting and, if needed, modyfying the code in a simple manner. New 
  features involve:                                                          \n
                                                                             \n
- The possibility of feeding into the simulations the nuclear distributions 
  accounting for the two-body NN correlations (read from external files, see
  Alvioli, Drescher and Strikman, [Phys. Lett. B680, 225, 2009], the distributions 
  can be found at http://www.phys.psu.edu/~malvioli/eventgenerator/ )  
- The use of the Gaussian NN wounding profile (which is more realistic than 
  the commonly-used hard-core wounding profile, see the analysis by Bialas 
  and Bzadak [Acta Phys. Polon. B38, 159,2007]) 
- The generation of the core-mantle (core-corona) distributions (see Bozek 
  [Acta Phys. Polon. B36, 3071,2005] and Werner [Phys. Rev. Lett. 98, 152301, 
  2007], see also Becattini and Manninen [Phys. Lett. B673, 19, 2009] and Bozek 
  [Phys. Rev. C79, 054901, 2009]) 
- The analysis of the triangular shape deformation parameter and profiles, 
  relevant for the triangular flow, see Alver and Roland, [Phys. Rev. C81, 054905, 
  2010] and Alver, Gombeaud, Luzum,and Ollitrault, [arXiv:1007.5469] 
- Generation of rapidity distributions in the wounded-nucleon picture according 
  to the model of Bialas and Czyz [Acta Phys.Polon.B36:905-918,2005], as implemented 
  by Bozek [arXiv:1002.4999]. This allows to obtain the fully 3-dimensional
  distribution of matter in the early Glauber phase of the collision.        \n 
                                                                          \n \n
  The reference manual for ver. 2, generated by Doxygen, is supplied at the home
  page. The full write-up of ver. 2 is under preparation.                    \n
                                                                             \n
  The code can be freely used and redistributed. However, if you decide to  
  make modifications, the authors would appreciate notification for the record.             
  Any publication or display of results obtained using GLISSANDO must        
  include a reference to our published paper.                                

*/

#include <TRandom3.h>
#include <iostream>

using namespace std; 

/*******************
    variables
*******************/


extern TRandom3 raa; //!< random number generator from ROOT 

extern float ver;    //!< version of the code 

//! structure for output of the full event - transverse coordinates, weight, number of event
typedef struct {
                Float_t X, //!< x coordinate
                        Y, //!< y coordinate
                        W; //!< z coordinate
                UInt_t KK; //!< number of the event
                } SOURCE; 

static SOURCE tSource;      //!< structure used when FULL=1 

extern int EVENTS,          //!< number of generated events 
           NBIN,            //!< number of bins for histogramming in x, y, and r 
           FBIN,            //!<  number of bins for histogramming in the azimuthal angle
           NUMA,            //!< mass number of nucleus A 
           NUMB,            //!< mass number of nucleus B 
           WMIN,            //!< minimum number of wounded nucleons to record the event 
           MODEL,           //!< switch for the superimposed multiplicity distribution: 0 - uniform, 1 - Poisson, 2 - Gamma
           DOBIN,           //!< 1 - count binary collisions even in the pure wounded-nucleon model. 0 - do not
           W0,              //!< minimum allowed number of wounded nucleons in the acceptance window
           W1,              //!< maximum allowed number of wounded nucleons in the acceptance window
           SHIFT,           //!< 1 - shift the coordinates of the fireball to c.m. in the fixed-axes case (preferred), 0 - do not
           RET,             //!< 0 - fix-last algorithm (preferred), 1 - return-to-beginning algorithm for the generation of the nuclear distribution
           FULL,            //!< 1 - generate the full event tree (large output file), 0 - do not
           FILES,           //!< 1 - read distribution from files, 0 - do not
           GAUSS,           //!< 1 - Gaussian wounding profile, 0 - hard-sphere wounding profile
           NUMRAP;          //!< number of particles per unit weight generated in the whole rapidity range


extern UInt_t ISEED,        //!< read seed for the ROOT random number generator, if 0 - random seed generated
              ISEED1;       //!< copy of ISEED

extern float BMIN,             //!< minimum value of the impact parameter in the acceptance window
             BMAX,             //!< maximum value of the impact parameter in the acceptance window
             RDS0,             //!< minimum value of the relative deposited strength (RDS) in the acceptance window 
             RDS1,             //!< maximum value of the relative deposited strength (RDS) in the acceptance window 
             BTOT,             //!< maximum impact parameter value for histogramming
             RWSA,             //!< Woods-Saxon radius for nucleus A
             AWSA,             //!< Woods-Saxon width for nucleus A
             RWSB,             //!< Woods-Saxon radius for nucleus B
             AWSB,             //!< Woods-Saxon width for nucleus B
             WFA,              //!< the w parameter for the Fermi distribution for nucleus A
             WFB,              //!< the w parameter for the Fermi distribution for nucleus B
             SNN,              //!< NN "wounding" cross section in millibarns
             SBIN,             //!< NN binary cross section in millibarns
             ALPHA,            //!< the mixing parameter: 0 - wounded, 1 - binary, 0.145 - mixed (PHOBOS)
             Uw,               //!< Poisson or Gamma parameters for superimposed distribution, wounded nucleons
             Ubin,             //!< Poisson or Gamma parameters for superimposed distribution, binary collisions
             PI,               //!< the number pi
             CD,               //!< closest allowed distance (expulsion distance) between nucleons in the nucleus in fm (simulation of repulsion)
             DW,               //!< dispersion of the location of the source for wounded nucleons (in fm)
             DBIN,             //!< dispersion of the location of the source for binary collisions (in fm)
             GA,               //!< Gaussian wounding profile parameter (hight at the origin)
             RAPRANGE,         //!< range in rapidity
             ETA0,     	       //!< 2*ETA0 is the width of the plateau in eta
             ETAM,             //!< parameter of the Bialas-Czyz-Bozek model
             SIGETA,           //!< parameter controling the witdth of the rapidity distribution 
             MAXYRAP;          //!< maximum absolute value of the y coordinate in the x-y-rapidity histogram  

/*******************************
 counting classes
*******************************/

//! simple counting class
class counter {
private:
    int count; //!< number of entries
    long double value; //!< sum of values counted
public:
//! reset the counter 
    void reset(){count=0;value=0;}; 
//! add entry
    void add(
            long double s //!< value added
    ){count++; value=value+s;};
//! get the number of entries
    int getN(){return count;};
//! get the sum of values
    long double get(){return value;};
//! get the mean value
    long double mean(){return value/count;};
};

//! counting class with variance
class counter2 {
private:
  int count; //!< number of entries
  long double  value,  //!< sum of values
               value2; //!< sum of squares of values
public:
//! reset the counter 
    void reset(){count=0;value=0;value2=0;};
//! add entry
    void add(
            long double s //!< value added
            ){count++; value=value+s; value2=value2+s*s;};
//! get the number of entries
    int getN(){return count;};
//! get the sum of values
    long double get(){return value;};
//! get the sum of squares of values
    long double get2(){return value2;};
//! get the mean value
    long double mean(){return value/count;};
//! get the variance
    long double var(){return value2/(count-1)-value/count*value/(count-1);}
//! get the variance multiplied with (N-1)/N
   long double vara(){return value2/count-value/count*value/count;};};

/*************************************
 declaration of counters and variables 
*************************************/

extern counter2 
estd,		  //!< counter for epsilon standard (fixed-axes), <r^2 cos(2 phi)>/<r^2>
epart,            //!< counter for epsilon participant (variable-axes), <r^2 cos(2 phi)>/<r^2>
estd3,            //!< counter for fixed-axes <r^2 cos(3 phi)>/<r^2>
epart3,           //!< counter for variable-axes <r^2 cos(3 phi)>/<r^2>
estd4,            //!< counter for fixed-axes <r^2 cos(4 phi)>/<r^2>
epart4,           //!< counter for variable-axes <r^2 cos(4 phi)>/<r^2>
estd5,            //!< counter for fixed-axes <r^2 cos(5 phi)>/<r^2>
epart5,           //!< counter for variable-axes <r^2 cos(5 phi)>/<r^2>
estd6,            //!< counter for fixed-axes <r^2 cos(6 phi)>/<r^2>
epart6,           //!< counter for variable-axes <r^2 cos(6 phi)>/<r^2>
nwounded,         //!< counter for number of wounded nucleons
nbinary,          //!< counter for number of binary collisions
nhot,             //!< counter for number of hot-spots
nweight;          //!< counter for relative deposited strength (RDS)

extern int evall, //!< number of all attempted event
           kk;    //!< number of the current event

extern float 
d,             //!< the wounding distance
dbin;          //!< the binary-collision distance 

extern Float_t 
b,         //!< impact parameter 
sitot,     //!< the total A+B cross section in the acceptance window
sirad,     //!< equivalent hard-sphere radius for the cross section
rwA,       //!< number of wounded nucleons in A
rwB,       //!< number of wounded nucleons in B
rwAB,      //!< number of all wounded nucleons
rbin,      //!< number of binary collisions
rhotspot,  //!< number of hot-spots
rpa,       //!< relative deposited strength (RDS)
sizeav,	   //!< size
es,        //!< epsilon standard (fixed-axes), <r^2 cos(2 phi)>/<r^2>
ess,       //!< fixed-axes sine moment, <r^2 sin(2 phi)>/<r^2>
ep,        //!< epsilon participant (variable-axes), <r^2 cos(2 phi)>/<r^2>
eps,       //!< variable-axes sine moment, <r^2 sin(2 phi)>/<r^2> 
es3,       //!< fixed-axes <r^2 cos(3 phi)>/<r^2>
es3s,      //!< fixed-axes sine moment, <r^2 sin(3 phi)>/<r^2>
ep3,       //!< variable-axes <r^2 cos(3 phi)>/<r^2>
ep3s,      //!< variable-axes sine moment, <r^2 sin(3 phi)>/<r^2>
es4,       //!< fixed-axes <r^2 cos(4 phi)>/<r^2>
es4s,      //!< fixed-axes sine moment, <r^2 sin(4 phi)>/<r^2>
ep4,       //!< variable-axes <r^2 cos(4 phi)>/<r^2>
ep4s,      //!< variable-axes sine moment, <r^2 sin(4 phi)>/<r^2>
es5,       //!< fixed-axes <r^2 cos(5 phi)>/<r^2>
es5s,      //!< fixed-axes sine moment, <r^2 sin(5 phi)>/<r^2>
ep5,       //!< variable-axes <r^2 cos(5 phi)>/<r^2>
ep5s,      //!< variable-axes sine moment, <r^2 sin(5 phi)>/<r^2>
es6,       //!< fixed-axes <r^2 cos(6 phi)>/<r^2>
es6s,      //!< fixed-axes sine moment, <r^2 sin(6 phi)>/<r^2>
ep6,       //!< variable-axes <r^2 cos(6 phi)>/<r^2>
ep6s,      //!< variable-axes sine moment, <r^2 sin(6 phi)>/<r^2>
phirot,    //!< rotation angle maximizing the second Fourier moment 
phirot3,   //!< rotation angle maximizing the third Fourier moment 
phirot4,   //!< rotation angle maximizing the fourth Fourier moment 
phirot5,   //!< rotation angle maximizing the fifth Fourier moment 
phirot6,   //!< rotation angle maximizing the sixth Fourier moment 
xx,        //!< center-of-mass x coordinate
yy,        //!< center-of-mass y coordinate
xeps,      //!< average es 
xseps,     //!< standard deviation of es
xepp,      //!< average ep 
xsepp;     //!< standard deviation of ep

/***********
  Blocks
***********/

//! print the help
void helper(int argc, char* str);
//! print the header in the output
void header();
//! echo parameters to the output
void echopar();
//! print epilog to the output
void epilog();
//! start the time measurement
int  time_start();
//! stop the time measurement
void time_stop(int ts);
//! read parameter values from the input file
void readpar(TString inpfile);
//! reset the counters used to store physical quantities in the event
void reset_counters();

/***************************
 random generator functions
***************************/

//! random number generator using the built-in ROOT generator, uniform on (0,1)
float los() ; 

//! random number generator for the Woods-Saxon distribution - nucleus A
float rlosA() ;

//! random number generator for the Woods-Saxon distribution - nucleus B
float rlosB() ;

//! random number generator for the Hulthen distribution
float rlos_hult() ;

//! random number generator for the rapidity distribution - wounded nucleons from nucleus A
float los_rap_A() ;

//! random number generator for the rapidity distribution - wounded nucleons from nucleus B
float los_rap_B() ;

//! random number generator for the rapidity distribution - binary collisions
float los_rap_bin() ;

// random number generator for the Poisson distribution
// int ipois(float u) ;

//! random number generator for the Gamma distribution
float gamgen(float a) ;

//negative binomial distribution - not used
//int negbin(float m,float v) ;

//! distribution of RDS overlayed over the number of sources
float dist(int m, float u) ;

//! random shift of the source location
float disp(float x);


//! class storing the trees and histograms
/*!
Class for storage of ROOT structures (trees, histograms) used for later off-line analysis within ROOT of other codes 
*/
class tr_his_c {
public:

// trees
    TTree *param,      //!< parameters 
          *phys,       //!< A+B cross section and other physical results
          *full_event, //!< full info on the event (positions and RDS of the sources)
          *tree;       //!< basic physical results

// histograms
	TH2D *xyhist,        //!< cartesian fixed-axes distribution
             *xyhist_mantle, //!< cartesian fixed-axes mantle distribution
             *xyhist_core,   //!< cartesian fixed-axes core distribution
             *xyhistr,       //!< cartesian variable-axes distribution

             *c0hist, //!< polar fixed-axes distribution of cos(phi)
             *c2hist, //!< polar fixed-axes distribution of cos(2 phi)
             *c3hist, //!< polar fixed-axes distribution of cos(3 phi)
             *c4hist, //!< polar fixed-axes distribution of cos(4 phi) 
             *c5hist, //!< polar fixed-axes distribution of cos(5 phi)
             *c6hist, //!< polar fixed-axes distribution of cos(6 phi)

	     *c0rhist, //!< polar variable-axes distribution of cos(phi)
             *c2rhist, //!< polar variable-axes distribution of cos(2 phi) 
             *c3rhist, //!< polar variable-axes distribution of cos(3 phi) 
             *c4rhist, //!< polar variable-axes distribution of cos(4 phi) 
             *c5rhist, //!< polar variable-axes distribution of cos(5 phi) 
             *c6rhist, //!< polar variable-axes distribution of cos(6 phi)

	     *s3hist,  //!< polar fixed-axes distribution of sin(3 phi) 
             *s3rhist; //!< polar variable-axes distribution of sin(3 phi)
 
// histograms for the dependence of certain quantities on the total number of wounded nucleons
	TH1D *nx,      //!< center-of-mass x coordinate of the source distribution vs. Nw
             *nx2,     //!< square of cm x coordinate, then its variance, vs. Nw 
             *ny,      //!< center-of-mass y coordinate of the source distribution vs. Nw
             *ny2,     //!< square of cm y coordinate, then its variance, vs. Nw
             *nsize,   //!< size vs. Nw
             *nsize2,  //!< square of size, then its variance/size^2, vs. Nw
             *neps,    //!< fixed-axes eccentricity vs. Nw
             *neps2,   //!< square of fixed-axes eccentricity, then its variance, vs. Nw
             *neps4,   //!< fixed-axes fourth moment vs. Nw
             *nepsp,   //!< variable-axes eccentricity vs. Nw
             *nepsp2,  //!< square of variable-axes eccentricity, then its variance, vs. Nw
             *nepsp4,  //!< variable-axes fourth moment vs. Nw
             *nuni,    //!< frequency of Nw, i.e. histogram of unity vs. Nw
             *nepsb,   //!< fixed-axes eccentricity vs. b 
             *neps2b,  //!< square of fixed-axes eccentricity, then its variance, vs. b
             *nepspb,  //!< variable-axes eccentricity vs. b 
             *nepsp2b, //!< square of variable-axes eccentricity, then its variance, vs. b
             *nunib,   //!< frequency of b, i.e. histogram of unity vs. b
             *nwb,     //!< number of wounded nucleons in nucleus B vs. total number of wounded nucleons
             *nw2b;    //!< square of the number of wounded nucleons in nucleus B, then its variance, vs. total number of wounded nucleons
 
// histograms for fluctuations of number of wounded nucleons and RDS
	TH1D *nwei,   //!< RDS vs. number of wounded nucleons in nucleus A
             *nwei2,  //!< square of RDS, then its variance, vs. number of wounded nucleons in nucleus A
             *ntarg,  //!< number of wounded nucleons in nucles B vs. number of wounded nucleons in nucleus A
             *ntarg2, //!< square of the number of wounded nucleons in nucles B, then its variance, vs. number of wounded nucleons in nucleus A
             *nbinar, //!< number of binary collisions vs. number of wounded nucleons in nucleus A 
             *nbinar2,//!< square of the number of binary collisions, then its variance, vs. number of wounded nucleons in nucleus A  
             *nunp;   //!< frequency of the number of wounded nucleons in nucleus A


// histograms for nuclear profiles, correlations, and the weight distribution
    TH1D  *rad,      //!< one-body radial distribution in the nucleus
          *rrel,     //!< distance between the pair of nucleons in the nucleus  
          *rrel_u,   //!< uncorrelated distance between the pair of nucleons in the nucleus (one nucleon from A, the other one from B)
          *weih,     //!< the distribution overlaid on the wounded nucleons
          *weih_bin, //!< the distribution overlaid over binary collisions
          *wpro;     //!< the wounding profile

// histogram for radial distributions
TH1D *c0hp,  //!< fixed-axes radial profile f_0
     *c2hp,  //!< fixed-axes radial profile f_2
     *c3hp,  //!< fixed-axes radial profile f_3
     *s3hp,  //!< fixed-axes radial profile for the sine moment, g_3
     *c4hp,  //!< fixed-axes radial profile f_4
     *c5hp,  //!< fixed-axes radial profile f_5
     *c6hp,  //!< fixed-axes radial profile f_6
     *c0rhp, //!< variable-axes radial profile f_0
     *c2rhp, //!< variable-axes radial profile f_2 
     *s3rhp, //!< variable-axes radial profile f_3 
     *c3rhp, //!< variable-axes radial profile for the sine moment, g_3 
     *c4rhp, //!< variable-axes radial profile f_4
     *c5rhp, //!< variable-axes radial profile f_5 
     *c6rhp; //!< variable-axes radial profile f_6

//! initialize the histograms
    void init(){
        param = new TTree("param","param tree"); //< tree storing parameters
        param->Branch("EVENTS",&EVENTS,"EVENTS/I");
        param->Branch("NBIN",&NBIN,"NBIN/I");
        param->Branch("FBIN",&FBIN,"FBIN/I");
        param->Branch("NUMA",&NUMA,"NUMA/I");
        param->Branch("NUMB",&NUMB,"NUMB/I");
        param->Branch("WMIN",&WMIN,"WMIN/I");
        param->Branch("MODEL",&MODEL,"MODEL/I");
        param->Branch("W0",&W0,"W0/I");
        param->Branch("W1",&W1,"W1/I");
        param->Branch("RDS0",&RDS0,"RDS0/F");
        param->Branch("RDS1",&RDS1,"RDS1/F");
        param->Branch("ISEED",&ISEED,"ISEED/i");
        param->Branch("BMIN",&BMIN,"BMIN/F");
        param->Branch("BMAX",&BMAX,"BMAX/F");
        param->Branch("BTOT",&BTOT,"BTOT/F");
        param->Branch("RWSA",&RWSA,"RWSA/F");
        param->Branch("AWSA",&AWSA,"AWSA/F");
        param->Branch("RWSB",&RWSB,"RWSB/F");
        param->Branch("AWSB",&AWSB,"AWSB/F");
        param->Branch("WFA",&WFA,"WFA/F");
        param->Branch("WFB",&WFB,"WFB/F");
        param->Branch("SNN",&SNN,"SNN/F");
        param->Branch("SBIN",&SBIN,"SBIN/F");
        param->Branch("ALPHA",&ALPHA,"ALPHA/F");
        param->Branch("DOBIN",&DOBIN,"DOBIN/I");
        param->Branch("Uw",&Uw,"Uw/F");
        param->Branch("Ubin",&Ubin,"Ubin/F");
        param->Branch("CD",&CD,"CD/F");
        param->Branch("SHIFT",&SHIFT,"SHIFT/I");
        param->Branch("RET",&RET,"RET/I");      
        param->Branch("DW",&DW,"DW/F");
        param->Branch("DBIN",&DBIN,"DBIN/F");
        param->Branch("GAUSS",&GAUSS,"GAUSS/I");
        param->Branch("GA",&GA,"GA/F");
        param->Branch("FILES",&FILES,"FILES/I");
        param->Branch("ver",&ver,"ver/F");

        phys = new TTree("phys","physical results"); //< tree storing some physical quantities 
        phys->Branch("sitot",&sitot,"sitot/F");
        phys->Branch("eps_fixed",&xeps,"seps/F");
        phys->Branch("eps_variable",&xepp,"sepp/F");
        phys->Branch("sigma_eps_fixed",&xseps,"xseps/F");
        phys->Branch("sigma_eps_variable",&xsepp,"xsepp/F");

        full_event = new TTree("full_event","full event"); //< tree storing full info on the events
        full_event -> Branch("full_source",&tSource,"X:Y:W:KK");
     
        tree =  new TTree("events","event tree");         //< tree storing basic information on events
        tree->Branch("nwA",&rwA,"nwA");    // wounded nucleons in A
        tree->Branch("nwB",&rwB,"nwB");    // wounded nucleons in B
        tree->Branch("nwAB",&rwAB,"nwAB"); // all wounded nucleons
        tree->Branch("nbin",&rbin,"nbin"); // binary collisions
        tree->Branch("npa",&rpa,"npa");    // source RDS
        tree->Branch("b",&b,"b");          // impact parameter

// fixed-axes and variable-axes cos and sin moments (some commented out)
	tree->Branch("size",&sizeav,"size"); // <r>
        tree->Branch("es",&es,"es");         // < r^2 cos(2 phi) >
        tree->Branch("ep",&ep,"ep");         // < r^2 cos(2 (phi-phi*)) >
        tree->Branch("es3",&es3,"es3");      // < r^2 cos(3 phi) >
        tree->Branch("ep3",&ep3,"ep3");      // < r^2 cos(3 (phi-phi*)) >
        tree->Branch("es4",&es4,"es4");      // < r^2 cos(4 phi) >
        tree->Branch("ep4",&ep4,"ep4");      // < r^2 cos(4 (phi-phi*)) >
        tree->Branch("es5",&es5,"es5");      // < r^2 cos(5 phi) >
        tree->Branch("ep5",&ep5,"ep5");      // < r^2 cos(5 (phi-phi*)) >
        tree->Branch("es6",&es6,"es6");      // < r^2 cos(6 phi) >
        tree->Branch("ep6",&ep6,"ep6");      // < r^2 cos(6 (phi-phi*)) >

// fixed-axes and variable-axes sin moments
        tree->Branch("ess",&ess,"ess");    // < r^2 sin(2 phi) >
        tree->Branch("eps",&eps,"eps");    // < r^2 sin(2 (phi-phi*)) >
        tree->Branch("es3s",&es3s,"es3s"); // < r^2 sin(3 phi) >
        tree->Branch("ep3s",&ep3s,"ep3s"); // < r^2 sin(3 (phi-phi*)) >
        tree->Branch("es4s",&es4s,"es4s"); // < r^2 sin(4 phi) >
        tree->Branch("ep4s",&ep4s,"ep4s"); // < r^2 sin(4 (phi-phi*)) >
        tree->Branch("es5s",&es5s,"es5s"); // < r^2 sin(5 phi) >
        tree->Branch("ep5s",&ep5s,"ep5s"); // < r^2 sin(5 (phi-phi*)) >
        tree->Branch("es6s",&es6s,"es6s"); // < r^2 sin(6 phi) >
        tree->Branch("ep6s",&ep6s,"ep6s"); // < r^2 sin(6 (phi-phi*)) >

// other quantities
        tree->Branch("phir",&phirot,"phir");    // the rotation angle phi*
        tree->Branch("phir3",&phirot3,"phir3"); // the rotation angle phi3*
        tree->Branch("phir4",&phirot4,"phir4"); // the rotation angle phi4*
        tree->Branch("phir5",&phirot5,"phir5"); // the rotation angle phi5*
        tree->Branch("phir6",&phirot6,"phir6"); // the rotation angle phi6*
        tree->Branch("xx",&xx,"x");             // x c.m. coordinate
        tree->Branch("yy",&yy,"y");             // y c.m. coordinate

        rad =  new TH1D("rad", "one-body distribution", 600, 0., 15.);
        rrel = new TH1D("rrel", "relative distance - correlated", 2500, 0.000001, 25.);
        rrel_u = new TH1D("rrel_u", "relative distance - uncorrelated", 2500, 0.000001, 25.);
	weih = new TH1D("weih", "source weight distribution, wounded", 500, -0.1, 1.5);
	weih_bin = new TH1D("weih_bin", "source weight distribution, binary", 500, -0.1, 1.5);
	wpro = new TH1D("wpro", "wounding profile", 100, 0.000001, 6.);

// histograms for the 2D profiles
	xyhist   = new TH2D("xyhist",  "fixed-axes source density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist -> SetXTitle("x"); xyhist -> SetYTitle("y");
	xyhist_mantle   = new TH2D("xyhist_mantle",  "fixed-axes mantle density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist_mantle -> SetXTitle("x"); xyhist_mantle -> SetYTitle("y");
	xyhist_core   = new TH2D("xyhist_core",  "fixed-axes core density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhist_core -> SetXTitle("x"); xyhist_core -> SetYTitle("y");
	xyhistr  = new TH2D("xyhistr", "variable-axes source density", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT);
        xyhistr -> SetXTitle("x"); xyhistr -> SetYTitle("y");

// histograms for the 2D profiles
// fixed-axes cos
	c0hist  = new TH2D("c0hist", "f_{0}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c0hist -> SetXTitle("r"); c0hist -> SetYTitle("#phi   ");
	c2hist  = new TH2D("c2hist", "f_{2}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c2hist -> SetXTitle("r"); c2hist -> SetYTitle("#phi   ");
	c3hist  = new TH2D("c3hist", "f_{3}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c3hist -> SetXTitle("r"); c3hist -> SetYTitle("#phi   ");
	c4hist  = new TH2D("c4hist", "f_{4}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c4hist -> SetXTitle("r"); c4hist -> SetYTitle("#phi   ");
	c5hist  = new TH2D("c5hist", "f_{5}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c5hist -> SetXTitle("r"); c5hist -> SetYTitle("#phi   ");
	c6hist  = new TH2D("c6hist", "f_{6}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c6hist -> SetXTitle("r"); c6hist -> SetYTitle("#phi   ");

// variable-axes cos
	c0rhist  = new TH2D("c0rhist", "f^{*}_{0}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c0rhist -> SetXTitle("r"); c0rhist -> SetYTitle("#phi   ");
	c2rhist  = new TH2D("c2rhist", "f^{*}_{2}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c2rhist -> SetXTitle("r"); c2rhist -> SetYTitle("#phi   ");
	c3rhist  = new TH2D("c3rhist", "f^{*}_{3}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c3rhist -> SetXTitle("r"); c3rhist -> SetYTitle("#phi   ");
	c4rhist  = new TH2D("c4rhist", "f^{*}_{4}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c4rhist -> SetXTitle("r"); c4rhist -> SetYTitle("#phi   ");
	c5rhist  = new TH2D("c5rhist", "f^{*}_{5}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c5rhist -> SetXTitle("r"); c5rhist -> SetYTitle("#phi   ");
	c6rhist  = new TH2D("c6rhist", "f^{*}_{6}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        c6rhist -> SetXTitle("r"); c6rhist -> SetYTitle("#phi   ");

// fixed-axes and variable-axes sin
	s3hist   = new TH2D("s3hist",      "g_{3}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        s3hist -> SetXTitle("r"); s3hist -> SetYTitle("#phi   ");
	s3rhist  = new TH2D("s3rhist", "g^{*}_{3}", NBIN, 0.001, BTOT,FBIN,-PI,PI);
        s3rhist -> SetXTitle("r"); s3rhist -> SetYTitle("#phi   ");

// histograms for the dependence of certain quantities on the number of wounded nucleons
	nx = new TH1D("nx", "<x> vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	nx2= new TH1D("nx2", "var(x) vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 
	ny = new TH1D("ny", "<y> vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	ny2= new TH1D("ny2", "var(y) vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 
	nsize = new TH1D("nsize", "<r> vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	nsize2 = new TH1D("nsize2", "var(<r>)/<<r>>^2 vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	neps = new TH1D("neps", "#epsilon_{2} vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	neps4 = new TH1D("neps4", "#epsilon_{4} vs. N_{w}", NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	neps2= new TH1D("neps2", "var(#epsilon)/#epsilon^{2} vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 
	nepsp = new TH1D("nepsp", "#epsilon^{*}_{2} vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	nepsp4 = new TH1D("nepsp4", "#epsilon^{*}_{4} vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	nepsp2= new TH1D("nepsp2", "var(#epsilon*)/#epsilon*^{2} vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5); 
	nuni = new TH1D("nuni", "event multiplicity vs. N_{w}",  NUMA+NUMB, 0.5, NUMA+NUMB+0.5);
	nepsb = new TH1D("nepsb", "#epsilon vs. b", 200, BMIN, BMAX);
	neps2b= new TH1D("neps2b", "var(#epsilon)/#epsilon^{2} vs. b", 200, BMIN, BMAX); 
	nepspb = new TH1D("nepspb", "#epsilon* vs. b", 200, BMIN, BMAX);
	nepsp2b= new TH1D("nepsp2b", "var(#epsilon*)/#epsilon*^{2} vs. b", 200, BMIN, BMAX); 
	nunib = new TH1D("nunib", "event multiplicity vs. b", 200, BMIN, BMAX);
	nwb = new TH1D("nwb", "N_{w} vs. b", 200, BMIN, BMAX);
	nw2b= new TH1D("nw2b", "var(N_{w}) vs. b", 200, BMIN, BMAX); 

// histograms for fluctuations of number of wounded nucleons and RDS
	nwei = new TH1D("nwei", "RDS vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);
	nwei2= new TH1D("nwei2", "var(RDS)/RDS vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5); 
	ntarg = new TH1D("ntarg", "N_{w}^{B} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);
	ntarg2= new TH1D("ntarg2", "var(N_{w}^{B})/N_{w}^{B} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5); 
	nbinar = new TH1D("nbinar", "N_{bin} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);
	nbinar2= new TH1D("nbinar2", "var(N_{bin})/N_{bin} vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5); 
	nunp = new TH1D("nunp", "event multiplicity vs. N_{w}^{A}",  NUMA, 0.5, NUMA+0.5);
        };

//! fill trees param and phys
    void fill(){param->Fill();phys->Fill();}; 

//! fill the main tree
    void fill_tr(){tree->Fill();};

//! projects the 2-dim histograms with polar distributions on 1-dim histograms
    void proj(){
	c0hp = c0hist->ProjectionX("c0hp");
	c2hp = c2hist->ProjectionX("c2hp");
	c3hp = c3hist->ProjectionX("c3hp");
	s3hp = s3hist->ProjectionX("s3hp");
	c4hp = c4hist->ProjectionX("c4hp");        
	c5hp = c5hist->ProjectionX("c5hp");        
	c6hp = c6hist->ProjectionX("c6hp");
	c0rhp = c0rhist->ProjectionX("c0rhp");
	c2rhp = c2rhist->ProjectionX("c2rhp");
	s3rhp = s3rhist->ProjectionX("s3rhp");
	c3rhp = c3rhist->ProjectionX("c3rhp");        
	c4rhp = c4rhist->ProjectionX("c4rhp");
	c5rhp = c5rhist->ProjectionX("c5rhp");        
	c6rhp = c6rhist->ProjectionX("c6rhp");
	};

//! calculates eccentricity, size, etc. and their fuctuations vs. number of wounded nucleons or b 
    void fill_res(){        
        nx -> Fill(rwAB,xx);
        nx2 -> Fill(rwAB,xx*xx);

        ny -> Fill(rwAB,yy);
        ny2 -> Fill(rwAB,yy*yy);

	nsize ->  Fill(rwAB,sizeav);
	nsize2 -> Fill(rwAB,sizeav*sizeav);

        neps -> Fill(rwAB,es);
        neps4 -> Fill(rwAB,es4);
        neps2 -> Fill(rwAB,es*es);
        nepsp -> Fill(rwAB,ep);
        nepsp4 -> Fill(rwAB,ep4);
        nepsp2 -> Fill(rwAB,ep*ep);
        nuni -> Fill(rwAB,1);

        nepsb -> Fill(b,es);
        neps2b -> Fill(b,es*es);
        nepspb -> Fill(b,ep);
        nepsp2b -> Fill(b,ep*ep);

	nwb -> Fill(b,rwAB);
	nw2b -> Fill(b,rwAB*rwAB);
        nunib -> Fill(b,1);

// for multiplicity fluctuations
        nwei->Fill(rwA,rpa);
        nwei2->Fill(rwA,rpa*rpa);
	ntarg->Fill(rwA,rwB);
	ntarg2->Fill(rwA,rwB*rwB);
	nbinar->Fill(rwA,rbin);
	nbinar2->Fill(rwA,rbin*rbin);
	nunp->Fill(rwA,1);};

//! write out trees param, phys, full_event, and the main tree
    void write(){param->Write();phys->Write();tree->Write();
                  if(FULL){full_event->Write();};
                };   

//! write out the radial density distribution an the pair distance distribution in the nucleus
    void write_r(){rad->Write();rrel->Write();rrel_u->Write();};

//! write out the overlaid distributions
    void write_w(){weih->Write();weih_bin->Write();};
 
//! write out the wounding profile
    void write_wpro(){wpro->Write();};

//! write out the histograms with the 2-dim distributions and the radial distributions of the Fourier components of the source profiles
    void write_d(){
	c0hp->Write(); 
	c2hp->Write(); 
	c3hp->Write(); 
	s3hp->Write();
	c4hp->Write(); 
	c5hp->Write(); 
	c6hp->Write(); 
	c0rhp->Write();
	c2rhp->Write(); 
	c3rhp->Write();
	s3rhp->Write(); 
	c4rhp->Write(); 
	c5rhp->Write();
	c6rhp->Write(); 

	xyhist->Write(); 
	xyhist_mantle->Write(); 
	xyhist_core->Write(); 
	xyhistr->Write();

	c0hist->Write(); 
	c0rhist->Write();};
 
//! generate histograms of eccentricities and their variance, etc., vs. the number of wounded nucleons or b
   void gen(){
	nx->Divide(nx,nuni);
        nx->Write();
        nx2->Divide(nx2,nuni);
        nx->Multiply(nx,nx);
        nx2->Add(nx,-1);

        ny->Divide(ny,nuni);
        ny->Write();
        ny2->Divide(ny2,nuni);
        ny->Multiply(ny,ny);
        ny2->Add(ny,-1);

	nsize -> Divide(nsize,nuni);
	nsize -> Write();
	nsize2-> Divide(nsize2,nuni);
	nsize -> Multiply(nsize,nsize);
	nsize2-> Add(nsize,-1);
	nsize2-> Divide(nsize2,nsize);
	nsize2-> Write();

        neps->Divide(neps,nuni);


        neps->Write();
        neps4->Divide(neps4,nuni);
        neps4->Write();

        neps2->Divide(neps2,nuni);
        neps->Multiply(neps,neps);
        neps2->Add(neps,-1);
        neps2->Divide(neps2,neps);

        nepsp->Divide(nepsp,nuni);
        nepsp->Write();
        nepsp4->Divide(nepsp4,nuni);
        nepsp4->Write();

        nepsp2->Divide(nepsp2,nuni);
        nepsp->Multiply(nepsp,nepsp);
        nepsp2->Add(nepsp,-1);
        nepsp2->Divide(nepsp2,nepsp);
        	
        nepsb->Divide(nepsb,nunib);
        nepsb->Write();
        neps2b->Divide(neps2b,nunib);
        nepsb->Multiply(nepsb,nepsb);
        neps2b->Add(nepsb,-1);
        neps2b->Divide(neps2b,nepsb);

        nepspb->Divide(nepspb,nunib);
        nepspb->Write();
        nepsp2b->Divide(nepsp2b,nunib);
        nepspb->Multiply(nepspb,nepspb);
        nepsp2b->Add(nepspb,-1);
        nepsp2b->Divide(nepsp2b,nepspb);

	nwb -> Divide(nwb,nunib);
	nwb -> Write();
	nw2b -> Divide(nw2b,nunib);
	nwb -> Multiply(nwb,nwb);
	nw2b -> Add(nwb,-1);

        nx2->Write();
        ny2->Write();
        neps2->Write();
        nepsp2->Write();
        nuni->Write();
        neps2b->Write();
        nepsp2b->Write();
	nw2b->Write();
        nunib->Write();

	// for multiplicity fluctuations
	nwei->Divide(nwei,nunp);
	ntarg->Divide(ntarg,nunp);
	nbinar->Divide(nbinar,nunp);
        nwei->Write();
	ntarg->Write();
	nbinar->Write();
	   
	nwei2->Divide(nwei2,nunp);
        nwei2->Divide(nwei2,nwei);
	nwei2->Add(nwei,-1);
  	nwei2->Write();
	   
	ntarg2->Divide(ntarg2,nunp);
        ntarg2->Divide(ntarg2,ntarg);
	ntarg2->Add(ntarg,-1);
  	ntarg2->Write();
	   
	nbinar2->Divide(nbinar2,nunp);
        nbinar2->Divide(nbinar2,nbinar);
	nbinar2->Add(nbinar,-1);
  	nbinar2->Write();
	   
	nunp->Write();};
};

