/** \file distrib.h
 * Part of GLISSANDO
 * 
 */


/*! \mainpage
                                                                                         
            GLISSANDO - GLauber Initial State Simulation AND mOre...         \n
                        ver. 2.07, 25 August 2010                            \n
                                                                             \n 
  Authors: 
           - Wojciech Broniowski (Wojciech.Broniowski@ifj.edu.pl)              
           - Maciej Rybczynski   (Maciej.Rybczynski@pu.kielce.pl)              
           - Piotr Bozek         (Piotr.Bozek@ifj.edu.pl)                    \n  
                                                                             \n
  
  Modification of the code to ver. 2 by WB.

  For the detailed description of ver. 1 program and further references         
  to the description of the model, please, refer to our                                                                                                   
  Computer Physics Communications 180(2009)69, arXiv:0710.5731 [nucl-th]     \n
  accessibile from: http://arxiv.org.abs/0710.5731                           \n
                                                                             \n
  Implementation of nuclear correlations in ver. 2 as described in 
  Phys. Rev. C81(2010)064909                                                 \n
                                                                             \n
  Homepage: http://www.pu.kielce.pl/homepages/mryb/GLISSANDO/index.html      \n
  
  GLISSANDO is a Glauber Monte-Carlo generator for early-stages of relativistic 
  heavy-ion collisions, written in c++ and interfaced to ROOT. Several models 
  are implemented: the wounded-nucleon model, the binary collisions model, 
  the mixed model, and the model with hot-spots. The original geometric 
  distribution of sources (i.e., wounded nucleons or binary collisions) in 
  the transverse plane can be superimposed with a statistical distribution 
  simulating the dispersion in the generated transverse energy in each individual 
  collision. The program generates inter alia the fixed axes (standard) 
  and variable-axes (participant) two-dimensional profiles of the density 
  of sources in the transverse plane and their Fourier components. These profiles 
  can be used in further analyses of physical phenomena, such as the the jet 
  quenching, event-by-event hydrodynamics, or analyses of the elliptic flow 
  and its fluctuations. Characteristics of the event (multiplicities, eccentricities, 
  Fourier shape coefficients, etc.) are evaluated and stored in a ROOT file 
  for further off-line studies. A number of scripts is provided for that purpose. 
  The code can also be used for the proton-nucleus and deuteron-nucleus collisions. \n
                                                                             \n

  Version 2 of GLISSANDO offers much more functionality than version 1, 
  moreover, it is fully object-oriented, providing the user with the flexibility 
  of inspecting and, if needed, modyfying the code in a simple manner. New 
  features involve:                                                          \n
                                                                             \n
- The possibility of feeding into the simulations the nuclear distributions 
  accounting for the two-body NN correlations (read from external files, see
  Alvioli, Drescher and Strikman, [Phys. Lett. B680, 225, 2009], the distributions 
  can be found at http://www.phys.psu.edu/~malvioli/eventgenerator/ )  
- The use of the Gaussian NN wounding profile (which is more realistic than 
  the commonly-used hard-core wounding profile, see the analysis by Bialas 
  and Bzadak [Acta Phys. Polon. B38, 159,2007]) 
- The generation of the core-mantle (core-corona) distributions (see Bozek 
  [Acta Phys. Polon. B36, 3071,2005] and Werner [Phys. Rev. Lett. 98, 152301, 
  2007], see also Becattini and Manninen [Phys. Lett. B673, 19, 2009] and Bozek 
  [Phys. Rev. C79, 054901, 2009]) 
- The analysis of the triangular shape deformation parameter and profiles, 
  relevant for the triangular flow, see Alver and Roland, [Phys. Rev. C81, 054905, 
  2010] and Alver, Gombeaud, Luzum,and Ollitrault, [arXiv:1007.5469] 
- Generation of rapidity distributions in the wounded-nucleon picture according 
  to the model of Bialas and Czyz [Acta Phys.Polon.B36:905-918,2005], as implemented 
  by Bozek [arXiv:1002.4999]. This allows to obtain the fully 3-dimensional
  distribution of matter in the early Glauber phase of the collision.        \n 
                                                                          \n \n
  The reference manual for ver. 2, generated by Doxygen, is supplied at the home
  page. The full write-up of ver. 2 is under preparation.                    \n
                                                                             \n
  The code can be freely used and redistributed. However, if you decide to  
  make modifications, the authors would appreciate notification for the record.             
  Any publication or display of results obtained using GLISSANDO must        
  include a reference to our published paper.                                

*/

#include <TH1D.h>
#include <TH3D.h>


//! Distribution of sources in space. 
/*!
Class for storage and basic operations (translation, rotation) of a distribution of "sources" in space. 
A source is a point with real weight and some additional integer property, e.g., charge.   
 */

class distr {
 public:
//! number of sources (points)
    int    n;
//! x space coordinate 
    float *x;
//! y space coordinate  
    float *y;
//! z space coordinate    
    float *z;
//! some integer property, here called "charge" 
/*! Depending on the situation, c is the electric charge of the nucleon in the nucleus, number of collisions of the nucleon, etc. */ 
    int   *c; 
//! weight  
/*! Depending on the situation, the weigh may describe the amount of the deposited energy, entropy, etc. */ 
       float *w;  
//! center-of-mass x coordinate of the distribution
    float xcm, 
//! center-of-mass y coordinate of the distribution
    ycm, 
//! center-of-mass y coordinate of the distribution 
    zcm, 
//! mean squared radius of the distribution    
    msr, 
//! mean squared transverse radius of the distribution
    msrt, 
//! sum of weights of the distribution
    sumw;

//! constructor
    distr(
          int k  /*!< number of sources, k>0 */  
         ){ 
           n=k;
           x=NULL;y=NULL;z=NULL;x=new float[n];y=new float[n];z=new float[n];c=new int[n];w=new float[n];
	   for(int i=0;i<n;i++){x[i]=0;y[i]=0;z[i]=0;c[i]=0;w[i]=1;};};

//! default constructor, 208 sources
/*! 208 correspons to the number on nucleons in the 208Pb nucleus */
    distr(void){
           n=208;
           x=NULL;y=NULL;z=NULL;x=new float[n];y=new float[n];z=new float[n];c=new int[n];w=new float[n];
	   for(int i=0;i<n;i++){x[i]=0;y[i]=0;z[i]=0;c[i]=0;w[i]=1;};};

//! copying constructor
    distr(const distr & w1){ 
           n = w1.n;
           x=NULL;y=NULL;z=NULL;x=new float[n];y=new float[n];z=new float[n];c=new int[n];w=new float[n];
	   for(int i=0;i<n;i++){x[i]=w1.x[i];y[i]=w1.y[i];z[i]=w1.z[i];c[i]=w1.c[i];w[i]=w1.w[i];};}    

//! destructor
   ~distr(){}; 

//! generate sum of weights
    float sum_w(){ 
           sumw=0;for(int i=0;i<n;i++){sumw+=w[i];};return sumw;};

//! generate the center-of-mass x coordinate, no weights
    float cmx(){ 
           xcm=0;for(int i=0;i<n;i++){xcm+=x[i];};xcm/=n;return xcm;};

//! generate the center-of-mass y coordinate, no weights
    float cmy(){
           ycm=0;for(int i=0;i<n;i++){ycm+=y[i];};ycm/=n;return ycm;};

//! generate the center-of-mass z coordinate, no weights
    float cmz(){
           zcm=0;for(int i=0;i<n;i++){zcm+=z[i];};zcm/=n;return zcm;};

//! generate the center-of-mass x coordinate with weights
    float cmx_w(){
           sum_w();xcm=0;for(int i=0;i<n;i++){xcm+=x[i]*w[i];};xcm/=sumw;return xcm;};

//! generate the center-of-mass y coordinate with weights
    float cmy_w(){
           sum_w();ycm=0;for(int i=0;i<n;i++){ycm+=y[i]*w[i];};ycm/=sumw;return ycm;};

//! generate the center-of-mass z coordinate with weights
    float cmz_w(){
           sum_w();zcm=0;for(int i=0;i<n;i++){zcm+=z[i]*w[i];};zcm/=sumw;return zcm;};

//! translate in the x direction
    void shift_x(
                 float xt /*!< displacement in the x direction */ 
                ){
           for(int i=0;i<n;i++){x[i]+=xt;};};

//! translate in the y direction
    void shift_y(
                 float yt /*!< displacement in the y direction */ 
                ){
           for(int i=0;i<n;i++){y[i]+=yt;};};

//! translate in the z direction
    void shift_z(
                 float zt /*!< displacement in the z direction */ 
                ){
           for(int i=0;i<n;i++){z[i]+=zt;};};

//! translate to the cm reference frame in the x direction, no weights
    void shift_cmx(){ 
           cmx();for(int i=0;i<n;i++){x[i]-=xcm;};};

//! translate to the cm reference frame in the y direction, no weights
    void shift_cmy(){
           cmy();for(int i=0;i<n;i++){y[i]-=ycm;};};

//! translate to the cm reference frame in the z direction, no weights
    void shift_cmz(){
           cmz();for(int i=0;i<n;i++){z[i]-=zcm;};};

//! translate to the cm reference frame in the x direction with weights
    void shift_cmx_w(){
           cmx_w();for(int i=0;i<n;i++){x[i]-=xcm;};};

//! translate to the cm reference frame in the y direction with weights
    void shift_cmy_w(){
           cmy_w();for(int i=0;i<n;i++){y[i]-=ycm;};};

//! translate to the cm reference frame in the z direction with weights
    void shift_cmz_w(){
           cmz_w();for(int i=0;i<n;i++){z[i]-=zcm;};};

//! mean squared radius, no weights
    float msrad(){ 
           cmx();cmy();cmz();
           msr=0;for(int i=0;i<n;i++){msr+=(x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm)+(z[i]-zcm)*(z[i]-zcm);};msr/=n;return msr;};

//! mean squared transverse radius, no weights
    float msrad_t(){
           cmx();cmy();
           msrt=0;for(int i=0;i<n;i++){msrt+=(x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm);};msrt/=n;return msrt;};

//! mean squared radius with weights
    float msrad_w(){
           cmx_w();cmy_w();cmz_w();
           msr=0;for(int i=0;i<n;i++){msr+=w[i]*((x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm)+(z[i]-zcm)*(z[i]-zcm));};msr/=sumw;return msr;};

//! mean squared transverse radius with weights
    float msrad_t_w(){
           cmx_w();cmy_w();
           msrt=0;for(int i=0;i<n;i++){msrt+=w[i]*((x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm));};msrt/=sumw;return msrt;};

//! size - the weighted average of the distance from the orgin (in the cm frame)
    float size(){
           cmx_w();cmy_w();
           float siz=0;for(int i=0;i<n;i++){siz+=w[i]*sqrt((x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm));};siz/=sumw;return siz;};

//! rotation angle maximizing the m-th cosine Fourier moment
/*! for m=2 this is the angle for passing to the "participant" frame */
    float phrot(
                int m /*!< rank of the Fourier moment, m=0,2,3,4,5,... (m=1 does not make sence in the cm frame) */
               ){
           float sc=0; float ss=0;
           for(int i=0;i<n;i++){float r2=x[i]*x[i]+y[i]*y[i];float ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*sin(m*ph)*w[i];};
                                return atan2(ss,sc)/m;};

//! rotate in the transverse plane by the specified angle 
    void rotate(
                float ph /*!< rotation angle in the transverse plane */
               ){
           for(int i=0;i<n;i++){float xp=x[i]*cos(ph)-y[i]*sin(ph);float yp=y[i]*cos(ph)+x[i]*sin(ph);x[i]=xp;y[i]=yp;};};

//! m-th cosine Fourier moment in the azimuthal angle in the transverse plane, limited charge range
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    float eps(
              int m,    /*!< rank of the Fourier moment, m=0,2,3,4,5,... (m=1 does not make sence in the cm frame) */
              int imin, /*!< lowest charge for the sources included */
              int imax  /*!< highest charge for the sources included */
             ){
                      float sc=0; float ss=0;
                      for(int i=0;i<n;i++){if((c[i]>=imin) && (c[i]<=imax)){float r2=x[i]*x[i]+y[i]*y[i];float ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*w[i];};};
                                return sc/ss;};

//! m-th cosine Fourier moment in the azimuthal angle in the transverse plane, no limit on charge
/*! Most popular is the second moment, known as eccentricity. The third moment is relevant for the triangular flow. */   
    float eps(
              int m /*!< rank of the Fourier moment, m=0,2,3,4,5,... (m=1 does not make sence in the cm frame) */
             ){ 
                      float sc=0; float ss=0;
                      for(int i=0;i<n;i++){float r2=x[i]*x[i]+y[i]*y[i];float ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*w[i];};
                                return sc/ss;};

//! m-th sine Fourier moment in the azimuthal angle in the transverse plane, limited charge 
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    float eps_s(
                int m,    /*!< rank of the Fourier moment, m=0,2,3,4,5,... (m=1 does not make sence in the cm frame) */
                int imin, /*!< lowest charge for the sources included */
                int imax  /*!< highest charge for the sources included */
               ){
                      float sc=0; float ss=0;
                      for(int i=0;i<n;i++){if((c[i]>=imin) && (c[i]<=imax)){float r2=x[i]*x[i]+y[i]*y[i];float ph=atan2(x[i],y[i]);
                                sc+=r2*sin(m*ph)*w[i]; ss+=r2*w[i];};};
                                return sc/ss;};

//! m-th sine Fourier moment in the azimuthal angle in the transverse plane, no limit on charge   
    float eps_s(
                int m /*!< rank of the Fourier moment, m=2,3,4,5,... (m=1 does not make sence in the cm frame) */
               ){
                      float sc=0; float ss=0;
                      for(int i=0;i<n;i++){float r2=x[i]*x[i]+y[i]*y[i];float ph=atan2(x[i],y[i]);
                                sc+=r2*sin(m*ph)*w[i]; ss+=r2*w[i];};
                                return sc/ss;};
  
//! fill the histogram of the transverse distribution in cartesian coordinates, limited charge  
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    void fill_xy(
                TH2D *xyh, /*!< 2-dim cartesian histogram in the transverse plane */ 
                float fac, /*!< normalizatin factor */
                int imin,  /*!< lowest charge for the sources included */
                int imax   /*!< highest charge for the sources included */
                ){
           for(int i=0;i<n;i++){if((c[i]*c[i]>=imin*imin) && (c[i]*c[i]<=imax*imax)){xyh->Fill(x[i],y[i],w[i]*fac);};};};

//! fill the histogram of the transverse distribution in cartesian coordinates, no limit on charge  
    void fill_xy(
                TH2D *xyh,  /*!< 2-dim cartesian histogram in the transverse plane */ 
                float fac   /*!< normalizatin factor */
                ){
           for(int i=0;i<n;i++){xyh->Fill(x[i],y[i],w[i]*fac);};};

//! fill the histogram of the transverse distribution in polar coordinates, limited charge  
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    void fill_polar(
                   TH2D *polh,  /*!< 2-dim polar histogram in the transverse plane */ 
                   int m,       /*!< rank of the Fourier moment */
                   float fac,   /*!< normalization factor */
                   int imin,    /*!< lowest charge for the sources included */
                   int imax     /*!< highest charge for the sources included */
                   ){
           for(int i=0;i<n;i++){if((c[i]>=imin) && (c[i]<=imax)){float r=sqrt(x[i]*x[i]+y[i]*y[i]); float ph=atan2(x[i],y[i]);
                                if(r>0){polh->Fill(r,ph,fac*w[i]*cos(m*ph)/r);};};};};

//! fill the histogram of the transverse sine distribution in polar coordinates, limited charge  
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    void fill_polar_s(
                   TH2D *polh,  /*!< 2-dim polar sine histogram in the transverse plane */ 
                   int m,       /*!< rank of the sine Fourier moment */
                   float fac,   /*!< normalization factor */
                   int imin,    /*!< lowest charge for the sources included */
                   int imax     /*!< highest charge for the sources included */
                   ){
           for(int i=0;i<n;i++){if((c[i]*c[i]>=imin*imin) && (c[i]*c[i]<=imax*imax)){float r=sqrt(x[i]*x[i]+y[i]*y[i]); float ph=atan2(x[i],y[i]);
                                if(r>0){polh->Fill(r,ph,fac*w[i]*sin(m*ph)/r);};};};};


//! fill the histogram of the transverse distribution in polar coordinates, no limit on charge
    void fill_polar(
                   TH2D *polh,   /*!< 2-dim polar histogram in the transverse plane */ 
                   int m,        /*!< rank of the Fourier moment */
                   float fac     /*!< normalization factor */
                   ){
           for(int i=0;i<n;i++){float r=sqrt(x[i]*x[i]+y[i]*y[i]); float ph=atan2(x[i],y[i]);
                                if(r>0){polh->Fill(r,ph,fac*w[i]*cos(m*ph)/r);};};};

//! fill the histogram of the transverse sine distribution in polar coordinates, no limit on charge
    void fill_polar_s(
                   TH2D *polh,   /*!< 2-dim polar sine histogram in the transverse plane */ 
                   int m,        /*!< rank of the sine Fourier moment */
                   float fac     /*!< normalization factor */
                   ){
           for(int i=0;i<n;i++){float r=sqrt(x[i]*x[i]+y[i]*y[i]); float ph=atan2(x[i],y[i]);
                                if(r>0){polh->Fill(r,ph,fac*w[i]*cos(m*ph)/r);};};};


//! substitution overloading
    distr& operator=(const distr& w1); 
};

//! substitution overloading
distr& distr::operator=(const distr &w1) 
{	
  if(this != &w1){                           
           n = w1.n;
           x=NULL;y=NULL;z=NULL;x=new float[n];y=new float[n];z=new float[n];c=new int[n];w=new float[n];
	   for(int i=0;i<n;i++){x[i]=w1.x[i];y[i]=w1.y[i];z[i]=w1.z[i];c[i]=w1.c[i];w[i]=w1.w[i];};
	   return *this;};	        
};


//! nucleus class
/*! Class to store distributions of nucleons in nuclei. */
class nucleus : public distr {
  
  private:
    bool g;
    float cth, sth, phi, r;

  public:
//! constructor  
  nucleus(
          int k /*!< number of nucleons in the nucleus (the mass naumber of the nucleus), 
                     k>0, k=1 - proton, k=2 - deuteron, k>2 - other nucleus */
         ) : distr(k){};

//! copying constructor 
  nucleus(const nucleus& w1) : distr(w1){};

//! substitution overloading
  nucleus& operator = (const nucleus& w1)
         {distr::operator=(w1);return *this;};

//! set the distribution for the proton 
/*! The proton is just placed in the origin */
  void set_proton(){x[0]=0.; y[0]=0.; z[0]=0.;}; 

//! set the distribution for the deuteron 
/*! Use the Hulthen distribution. */
  void set_deuteron(){x[0]=0.; y[0]=0.; z[0]=0.;
       phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlos_hult(); 
       x[1]=r*cos(phi)*sth; y[1]=r*sin(phi)*sth; z[1]=r*cth;}; 

//! set randomly the distribution of nucleons in the nucleus A, use the rlosA() function, no correlations
  void set_random_A(){for(int i=0;i<n;i++){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosA(); 
       x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 

//! set randomly the distribution of nucleons in the nucleus B, use the rlosB() function, no correlations 
  void set_random_B(){for(int i=0;i<n;i++){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosB(); 
       x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 

//! set randomly the distribution of nucleons in the nucleus A with expulsion, use the rlosA() function
/*! The nucleon positions are subsequntly generated acording to the spherically-symmetric 
    Woods-Saxon distribution. If the nucleon happens to be generated closer than the expulsion distance d 
    to any of the previously generated nucleons, it is generated anew, until it is "good". 
    Since this leads to some swelling, the original distribution must be a bit narrower to 
    cancel neutrilize this effect (see our original paper for a detailed discussion). 
    The expulsion simulates in a simple manner the nuclear repulsion and generates the hard-core two-body 
    correlations. */
  void set_random_A(
                   float d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                   ){int j=0;while(j<n){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosA(); 
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;};};};



//! set randomly the distribution of nucleons in the nucleus B with expulsion, use the rlosB() function
/*! Same as set_random_A for the case of the nucleus B, which in general is different from A */
  void set_random_B(
                   float d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                   ){int j=0;while(j<n){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosB(); 
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;};};};


//! set the distribution of nucleons in the nucleus from the tables generated earlier
/*! The nucleon position (x,y,z) is taken from the tables read earlier. The pointers x, y, z are 
    originally positioned at 
    px, py, pz at a place corresponding to the beginning of a randomly selected nucleus. 
    The tables with nuclear distributions have to be prepared externally, 
    see, e.g., http://www.phys.psu.edu/~malvioli/eventgenerator/ 
    for distributions involving correlations.
    */
  void set_file(
                float *px, /*!< x coordinate of distributions read from files */ 
                float *py, /*!< y coordinate of distributions read from files */ 
                float *pz, /*!< z coordinate of distributions read from files */ 
                int sn,    /*!< number of entries in the file (should be the mass number time the number of stored nuclei) */
                int nu     /*!< mass number of the nucleus */ 
               ){  
                   int count=raa.Integer(sn/nu)*nu;
                   x = (px+count); y = (py+count); z = (pz+count); // set the pointers
//		   for(int i=0;i<n;i++){x[i]=px[count]; y[i]=py[count]; z[i]=pz[count];count++;};
   }; 

//! set the distribution of nucleons in the nucleus from the tables generated earlier, killing correlations
/*! The nucleon position (x,y,z) is taken from the tables read earlier. The pointers x, y, z are positioned at 
    px, py, pz at a place corresponding to a completely randomly selected nucleon. This procedure kills any 
    correlations and is (probably) equivalent to the mixing technique.  
    The tables with nuclear distributions have to be prepared externally.
    */
  void set_file_uncor(
                float *px, /*!< x coordinate of distributions read from files */ 
                float *py, /*!< y coordinate of distributions read from files */ 
                float *pz, /*!< z coordinate of distributions read from files */ 
                int sn,    /*!< number of entries in the file (should be the mass number time the number of stored nuclei) */
                int nu     /*!< mass number of the nucleus */ 
               ){  
                   int count=raa.Integer(sn-nu);
                   x = (px+count); y = (py+count); z = (pz+count); // set the pointers
//		   for(int i=0;i<n;i++){x[i]=px[count]; y[i]=py[count]; z[i]=pz[count];count++;};
   }; 

//! distance between two nucleons 
  float dist2(
             int j1, /*!< index of nucleon 1 */
             int j2  /*!< index of nucleon 2 */
             ){return (x[j1]-x[j2])*(x[j1]-x[j2])+(y[j1]-y[j2])*(y[j1]-y[j2])+(z[j1]-z[j2])*(z[j1]-z[j2]);};

//! the pair of nucleons is "good" when the distance between the nucleons is larger than the expulsion distance d
  bool good_pair(
                int j1,  /*!< index of nucleon 1 */
                int j2,  /*!< index of nucleon 2 */
                float d  /*!< expulsion distance - nucleons cannot be closer to each other than d*/
                ){if(dist2(j1,j2)>d*d){return true;} else {return false;};};
  
//! nucleon j is "good" when the distance to all nucleons of index i with i<j is larger than the expulsion distance d
  bool good_down(
                int j,   /*!< index of the nucleon */
                float d  /*!< expulsion distance */
                ){if(j==0){return true;} else {g=true; for(int i=0;i<j;i++){g=g && good_pair(i,j,d);};return g;};};

//! configuration of nucleons in the nucleus is "good" when the distances between all nucleons are larger than the expulsion distance d
  bool good_all(
               float d   /*!< expulsion distance */
               ){g=true; for(int i=1;i<n;i++){g=g && good_down(i,d);};return g;};

}; //class


//! collision class
/*! 
Class performing the collision of two nuclei. Here the "source" is the position of the wounded niucleon (a nucleon that 
collided at least onece) or the location of the binary collision, taken as the average position in the nucleon pair. 
The weight, called RDS (relative deposited strength) depends on the adopted model. The "charge" is 0 for the binary 
collision, and equal to i for the wounded nucleons, where i>0 is the number of collisions experienced by the nucleon.  
*/
class collision : public distr {

  public:

  float rd2; /*!< square of the distance between the nucleons */
  int wc;    /*!< counter of the sources */
  int *wwA, /*!< wwA[i] is the number of collisions of the i-th nucleon from nucleus A with the nucleons from nucleus B */ 
      *wwB; /*!< wwB[i] is the number of collisions of the i-th nucleon from nucleus A with the nucleons from nucleus A */
  int nwA,  /*!< number of wounded (collided at least once) nucleons in nucleus A */
      nwB,  /*!< number of wounded (collided at least once) nucleons in nucleus B */
      nwAB, /*!< total number of wounded nucleons (nwA+nwB) generated in the event */
      nzw,  /*!< number of sources with nonzero weight */
      nbin, /*!< number of binary collisions in the event */
      nhotspot; /*!< number of hot spots */
  float rpa; /*!< total weight (RDS) */

  #if(_weight_)
  TH1D *w_distr,      /*!< histogram for the collision profile for the wounded nucleons */
       *w_distr_bin;  /*!< histogram for the collision profile for the binary collisions */
  #endif

//! constructor
  collision(
           int nnA,  /*!< mass number of nucleus A */ 
           int nnB   /*!< mass number of nucleus B */ 
           ){
            int nn=nnA+nnB+nnA*nnB;
  #if(_weight_)
           w_distr = new TH1D("w_distr", "wounding profile (normalized to wounding cross section)", 100, 0.000001, 6.);
           w_distr->SetXTitle("NN distance squared [fm^{2}]");
           w_distr_bin = new TH1D("w_distr_bin", "binary profile (normalized to binary cross section)", 100, 0.000001, 6.);
           w_distr_bin->SetXTitle("NN distance squared [fm^{2}]");
  #endif
           x=NULL;y=NULL;x=new float[nn];y=new float[nn];c=new int[nn];w=new float[nn];
           wwA=NULL;wwB=NULL;wwA=new int[nnA];wwB=new int[nnB];};

//! default constructor
/*! assumes 208Pb-208Pb collisions */
  collision(){
  #if(_weight_)
           w_distr = new TH1D("w_distr", "wounding profile (normalized to wounding cross section)", 100, 0.000001, 6.);
           w_distr->SetXTitle("NN distance squared [fm^{2}]");
           w_distr_bin = new TH1D("w_distr_bin", "binary profile (normalized to binary cross section)", 100, 0.000001, 6.);
           w_distr_bin->SetXTitle("NN distance squared [fm^{2}]");
  #endif
           x=NULL;y=NULL;x=new float[43680];y=new float[43680];c=new int[43680];w=new float[43680];
           wwA=NULL;wwB=NULL;wwA=new int[208];wwB=new int[208];};

//! collission between two nuclei
/*! gen_RDS performs the collision of two nuclei, generating the wounded nucleon and the binary collisions, as well as
their RDS (relative deposited strength, or weight). It is the core function of the code, implementing the specific mechanism of the collision. */
  void gen_RDS(
            const nucleus &nA,  /*!< nucleus A, nA>0, nA=1 - proton, nA=2 - deuteron, nA>2 - other nuclei */
            const nucleus &nB,  /*!< nucleus B, nB>0, nB=1 - proton, nB=2 - deuteron, nB>2 - other nuclei  */
            float d2,           /*!< wounding distance squared */
            float dbin2,        /*!< binary distance squared */
            float mb            /*!< ratio of the wounding to binary cross sections */
            ){

   for(int i=0;i<nA.n;i++){wwA[i]=0;};
   for(int j=0;j<nB.n;j++){wwB[j]=0;};
   nwA=0; nwB=0; nzw=0; rpa=0; nbin=0; nhotspot=0; 
   wc=-1;

   for(int i=0;i<nA.n;i++){
	for(int j=0;j<nB.n;j++){
		rd2=(nA.x[i]-nB.x[j])*(nA.x[i]-nB.x[j])+(nA.y[i]-nB.y[j])*(nA.y[i]-nB.y[j]);

			#if(_gauss_!=1)
        		   if(rd2 < d2)
			#else
			   if(los() < GA*exp(-GA*rd2/(d2)))
			#endif

                {wwA[i]++;wwB[j]++;
                 #if(_weight_) 
                 w_distr->Fill(rd2,1.);
                 #endif
    };};};


   for(int i=0;i<nA.n;i++){if(wwA[i]>0) 
	       	       {
                        nwA++; // count the wounded nucleons in A
			wc++; // count the sources
                        c[wc]=wwA[i];
 			x[wc]=nA.x[i]; // store x-coordinate of the source
			if(DW>0){x[wc]+=disp(DW);}; // displace randomly
 			y[wc]=nA.y[i]; // store y-coordinate of the source
			if(DW>0){y[wc]+=disp(DW);}; // displace randomly
			w[wc]=(1-ALPHA)/2.0*dist(MODEL,Uw); // wounded nucleon RDS
           		rpa+=w[wc];
	        	if(w[wc]>0){nzw++;}; // count wounded if RDS > 0
                        };};

   for(int i=0;i<nB.n;i++){if(wwB[i]>0) 
	       	       {nwB++; // count the wounded nucleons in B
			wc++; // count the sources
                        c[wc]=-wwB[i]; // by convention, negative numbers for nucleus B
 			x[wc]=nB.x[i]; // store x-coordinate of the source
			if(DW>0){x[wc]+=disp(DW);}; // displace randomly
 			y[wc]=nB.y[i]; // store y-coordinate of the source
			if(DW>0){y[wc]+=disp(DW);}; // displace randomly
			w[wc]=(1-ALPHA)/2.0*dist(MODEL,Uw); // wounded nucleon RDS
           		rpa+=w[wc];
	        	if(w[wc]>0){nzw++;}; // count wounded in nzw if RDS > 0
                       };};

if(ALPHA>0 || DOBIN==1){
   for(int i=0;i<nA.n;i++){
	for(int j=0;j<nB.n;j++){
		rd2=(nA.x[i]-nB.x[j])*(nA.x[i]-nB.x[j])+(nA.y[i]-nB.y[j])*(nA.y[i]-nB.y[j]);

			#if(_gauss_!=1)
        		   if(rd2 < dbin2)
			#else
			   if(los() < GA*exp(-GA*rd2/dbin2))
			#endif

			       {
                        #if(_weight_)
                                w_distr_bin->Fill(rd2,1.);
                        #endif
                                nbin++; // count the binary collisions
                                wc++;   // count the sources
                                c[wc]=0; // be convention, 0 for binary collisions
				x[wc]=(nA.x[i]+nB.x[j])/2;
				if(DBIN>0){// displace randomly
					x[wc]+=disp(DBIN);
				};
                        	y[wc]=(nA.y[i]+nB.y[j])/2;
				if(DBIN>0){// displace randomly
					y[wc]+=disp(DBIN);
				};
		  		if(los()>mb){
					w[wc]=0;
				} 
				else {
					w[wc]=ALPHA/mb*dist(MODEL,Ubin); // binary RDS
					nhotspot++; 
					rpa+=w[wc];  
					if(w[wc]>0){nzw++;};  // count binary in nzw if RDS > 0
				};
			   };
		};
	};
};

  n=wc+1; nwAB=nwA+nwB;
};

}; //class


//! collision_rap class
/*! Collision class with an extra feature of generating the rapidity distribution. It can be used to create fully 3-dimensional 
distribution of sources, overlaying the (spatial) rapidity distribution over the transvers distribution. */
class collision_rap : public collision {

public:

TH3D *rap_distr;      /*!< histogram for storing the rapidity distribution */
 
//! constructor  
  collision_rap(
           int nnA,  /*!< mass number of nucleus A */ 
           int nnB   /*!< mass number of nucleus B */ 
           ) : collision(nnA,nnB)
          {
           rap_distr = new TH3D("rap_distr", "3 dim. x-y-rapidity distribution of particles", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT,NBIN,-5,5);
           rap_distr->SetXTitle("x [fm]");  rap_distr->SetYTitle("y [fm]");  rap_distr->SetZTitle("#eta");  
         };

//! default constructor
  collision_rap() : collision()
          {
           rap_distr = new TH3D("rap_distr", "3 dim. x-y-rapidity distribution of particles", NBIN, -BTOT, BTOT,NBIN,-BTOT,BTOT,NBIN,-5,5);          
           rap_distr->SetXTitle("x [fm]");  rap_distr->SetYTitle("y [fm]");  rap_distr->SetZTitle("#eta");  
          };

//! generate the rapidity distribution
void gen_rap(
            int part,  /*!< number of particles per unit RDS */
            float maxy /*!< maximum absolute value of the transverse y coordinate for histogramming in the x-y-rapidity histogram */
            ){
float eta; // rapidity
for(int i=0;i<n;i++) // loop over all n sources
  {
    int np=w[i]*part;
    for(int j=0;j<np;j++)
       {
         if(c[i]>0)
           eta=los_rap_A();
         else if(c[i]<0)          
           eta=los_rap_B();
         else
           eta=los_rap_bin();
  
         if(y[i]*y[i]< maxy*maxy){rap_distr->Fill(x[i],y[i],eta,1.);};       
       };
  };
   
             }
}; //class


