/** \file epsilon_c.C
 * Script generating the plots of participant eccentricities and their scaled standard deviations as functions of centrality (part of GLISSANDO 2)
 */

// #include "label.C"

//! produces plots of the mean and the scaled standard deviation of the participant eccentricities as functions of centrality
/*! Produces plots of the mean and the scaled standard deviation of the participant eccentricities as functions of centrality. 
The centrality is determined from the number of wounded nucleons. */

void epsilon_c(
 char const *p //!< name of the ROOT input file
            ) {

TString empty("");

// Default file name
//TString inpfile("glissando.root");
char const *inpfile = "glissando.root";

 if (p!=empty) inpfile = p;
 cout << "reads from: " << inpfile << endl;;  


TFile *f = new TFile(inpfile);
TH1D *ho = ((TH1D*)f->Get("nepsp2")); 
TH1D *n  = ((TH1D*)f->Get("nuniRDS")); 

TAxis *xaxis = ho->GetXaxis();
Int_t ima=(xaxis->GetNbins());
Float_t ima_w=(xaxis->GetBinWidth(1));
Float_t zas=ima_w*ima;
Float_t sc=(n->Integral());


Float_t bins[1300];
Int_t  binnum = ima; 

bins[0]=0;

for (Int_t i=1;i<=ima;i++) {
    bins[i]=bins[i-1]+(n->GetBinContent(ima-i+1))/sc;
// cout << ima-i << " " << bins[i] << endl;
};

TH1D* h = new TH1D("h","t", binnum, bins);

for (Int_t i=1;i<=ima;i++) {
    Float_t v =(ho->GetBinContent(i));
    if (v>0){
        h->SetBinContent(ima-i+1,sqrt(v));    
    }
        else{
            h->SetBinContent(ima-i+1,0);        
        }
}


Int_t reb;
//reb=1 - number of bins as in original histigram
//reb=x - x times smaller number of bins
cout << "Warning: this script makes sense for the minimum-bias case" << endl << 
        "Give the rebinning parameter (a small natural number): ";
cin >> reb;
cout << endl;

TH1D *hnew = (TH1D*)h -> Rebin(reb,"hnew");
hnew -> Scale(1./reb);

Float_t title_offset=0.88;
Float_t label_offset=0.01;
Float_t title_size=0.05;
Float_t label_size=0.05;

TCanvas *c2 = new TCanvas("c2","sigma epsilon",0,0,1000,600);
c2->cd(1);
c2->Range(0,0,25,18);
c2->SetFillColor(0);

hnew->SetStats(kFALSE);

hnew->SetTitle("");
hnew->SetXTitle("centrality");
hnew->GetXaxis()->SetLabelSize(label_size);
hnew->GetYaxis()->SetLabelSize(label_size);
hnew->GetXaxis()->SetLabelOffset(label_offset);
hnew->GetYaxis()->SetLabelOffset(label_offset);
hnew->GetXaxis()->SetTitleSize(title_size);
hnew->GetYaxis()->SetTitleSize(title_size);
hnew->GetXaxis()->SetTitleOffset(title_offset);
hnew->GetYaxis()->SetTitleOffset(title_offset);

hnew->GetYaxis()->SetRangeUser(0.,0.7);
hnew->SetYTitle("#Delta#epsilon_{2}/#epsilon_{2}    ");
hnew->SetLineColor(kBlue);
hnew->SetLineWidth(2);
hnew->SetLineStyle(1);
hnew->Draw("HIST");


Float_t ymax=sqrt(4/3.14159265-1);
TLine *liml = new TLine(0,ymax,1.0,ymax);
liml->Draw("SAME");

Int_t NUMA, NUMB;
Int_t PARTONS;
Float_t ECM_qq,ECM_NN,ALPHA;

TTree *param_NN = (TTree*)f->Get("param");
param_NN->SetBranchAddress("PARTONS",&PARTONS);param_NN->GetEntry(0);
param_NN->SetBranchAddress("NUMA",&NUMA);      param_NN->GetEntry(0);
param_NN->SetBranchAddress("NUMB",&NUMB);   param_NN->GetEntry(0);
param_NN->SetBranchAddress("ALPHA",&ALPHA); param_NN->GetEntry(0);
param_NN->SetBranchAddress("ECM",&ECM_NN);  param_NN->GetEntry(0);

cout << "NN, center-of-mass-energy, sqrt(s_NN): " << ECM_NN << " GeV" << endl;

char tEVENTS[60];
sprintf(tEVENTS,"GLISSANDO 3   %d+%d @ %.0f GeV",NUMA,NUMB,ECM_NN);

char tOPIS[60];
if (ALPHA!=0.0)
    sprintf(tOPIS,"mixed model with ALPHA=%.3f",ALPHA);
else if (PARTONS==0) 
      sprintf(tOPIS,"wounded nucleon model");
  else 
      sprintf(tOPIS,"wounded quark model");

Float_t x1=0.45, x2=0.60;
Float_t off1=0.85, off2=0.78;

TLatex *t1 = new TLatex();
t1->SetNDC();
t1->SetTextAlign(13);
t1->SetTextFont(32);
t1->SetTextColor(kBlack);
t1->SetTextSizePixels(18);
t1->SetTextAlign(12);
t1->DrawLatex(x1,off1,tEVENTS);
t1->DrawLatex(x2,off2,tOPIS);
c2->Modified();
c2->cd();
c2->SetSelected(c2);

TLegend *leg = new TLegend(0.6,0.50,0.8,0.62);
leg->SetFillColor(0);
leg->SetBorderSize(0);
leg->AddEntry(hnew, "  quarks", "l");
leg->Draw("SAME");

c2->SaveAs("sigma_epsilon_c.pdf");

TH1D *hepso = ((TH1D*)f->Get("nepsp")); 
TH1D* heps = new TH1D("n","t", binnum, bins);

for (Int_t i=1;i<=ima;i++) {
    Float_t v =(hepso->GetBinContent(i));
    heps->SetBinContent(ima-i+1,v);
}


TH1D *hepsnew = (TH1D*)heps -> Rebin(reb,"hepsnew");
hepsnew -> Scale(1./reb);
hepsnew->SetStats(kFALSE);

TCanvas *c4 = new TCanvas("c4","epsilon",0,0,1000,600);
c4->cd(1);
c4->Range(0,0,25,18);
c4->SetFillColor(0);

hepsnew->SetTitle("");
hepsnew->SetXTitle("centrality");
hepsnew->GetXaxis()->SetLabelSize(label_size);
hepsnew->GetYaxis()->SetLabelSize(label_size);
hepsnew->GetXaxis()->SetLabelOffset(label_offset);
hepsnew->GetYaxis()->SetLabelOffset(label_offset);
hepsnew->GetXaxis()->SetTitleSize(title_size);
hepsnew->GetYaxis()->SetTitleSize(title_size);
hepsnew->GetXaxis()->SetTitleOffset(title_offset);
hepsnew->GetYaxis()->SetTitleOffset(title_offset);
hepsnew->GetYaxis()->SetRangeUser(0,1.05);
hepsnew->SetYTitle("#epsilon_{2}    ");
hepsnew->SetLineColor(kBlue);
hepsnew->SetLineStyle(1);

hepsnew->Draw("HIST");

TLatex *t2 = new TLatex();
t2->SetNDC();
t2->SetTextAlign(13);
   t2->SetTextFont(32);
   t2->SetTextColor(kBlack);
   t2->SetTextSizePixels(18);
   t2->SetTextAlign(12);
   t2->DrawLatex(x1,off1,tEVENTS);
   t2->DrawLatex(x2,off2,tOPIS);

   c4->Modified();
   c4->cd();
   c4->SetSelected(c4);

c4->SaveAs("epsilon_c.pdf");

}
