/** \file distrib.h
 * Part of GLISSANDO 3
 * 
 */

#ifndef _GL_DISTRIB
#define _GL_DISTRIB  

#include <TH1D.h>
#include <TH3D.h>

#include "counter.h"


//! Distribution of sources in space. 
/*!
Class for storage and basic operations (translation, rotation) of a distribution of "sources" in space. 
A source is a point with real weight and some additional integer property, e.g., charge.   
 */

class distr {
 public:
//! number of sources (points)
    Int_t    n;  
//! x space coordinate 
    Float_t *x;
//! y space coordinate  
    Float_t *y;
//! z space coordinate    
    Float_t *z;

//! some integer property, here called "charge" 
/*! Depending on the situation, c is the electric charge of the nucleon in the nucleus, number of collisions of the nucleon, etc. */ 
    Int_t   *c; 
//! weight  
/*! Depending on the situation, the weight may describe the amount of the deposited energy, entropy, etc., in the source */ 
    Float_t *w;  
//! center-of-mass x coordinate of the distribution
    Float_t xcm, 
//! center-of-mass y coordinate of the distribution
    ycm, 
//! center-of-mass z coordinate of the distribution 
    zcm, 
//! mean squared radius of the distribution    
    msr,
//! mean squared transverse radius of the distribution
    msrt, 
//! sum of weights of the distribution
    sumw;

//! constructor
    distr(
          Int_t k  /*!< number of sources, k>0 */  
         ){ 
           n=k;
           x=NULL;y=NULL;z=NULL;x=new Float_t[n];y=new Float_t[n];z=new Float_t[n];
           c=new Int_t[n];w=new Float_t[n];
	   for(Int_t i=0;i<n;i++){x[i]=0;y[i]=0;z[i]=0;c[i]=0;w[i]=1;};};

//! default constructor, 208 sources
/*! 208 corresponds to the number on nucleons in the 208Pb nucleus */
    distr(void){
           n=208;
           x=NULL;y=NULL;z=NULL;x=new Float_t[n];y=new Float_t[n];z=new Float_t[n];
           c=new Int_t[n];w=new Float_t[n];
	   for(Int_t i=0;i<n;i++){x[i]=0;y[i]=0;z[i]=0;c[i]=0;w[i]=1;};};

//! copying constructor
    distr(const distr & w1){ 
           n = w1.n;
           x=NULL;y=NULL;z=NULL;x=new Float_t[n];y=new Float_t[n];z=new Float_t[n];
           c=new Int_t[n];w=new Float_t[n];
	   for(Int_t i=0;i<n;i++){x[i]=w1.x[i];y[i]=w1.y[i];z[i]=w1.z[i];
                                  c[i]=w1.c[i];w[i]=w1.w[i];};};    

//! destructor
//   ~distr(){
// delete x; delete y; delete z; delete c; delete w;
//           }; 


//! generate sum of weights
    Float_t sum_w(){ 
           sumw=0;for(Int_t i=0;i<n;i++){sumw+=w[i];};return sumw;};

//! generate the center-of-mass x coordinate, no weights
    Float_t cmx(){ 
           xcm=0;for(Int_t i=0;i<n;i++){xcm+=x[i];};xcm/=n;return xcm;};

//! generate the center-of-mass y coordinate, no weights
    Float_t cmy(){
           ycm=0;for(Int_t i=0;i<n;i++){ycm+=y[i];};ycm/=n;return ycm;};

//! generate the center-of-mass z coordinate, no weights
    Float_t cmz(){
           zcm=0;for(Int_t i=0;i<n;i++){zcm+=z[i];};zcm/=n;return zcm;};

//! generate the center-of-mass x coordinate with weights
    Float_t cmx_w(){
           sum_w();xcm=0;for(Int_t i=0;i<n;i++){xcm+=x[i]*w[i];};xcm/=sumw;return xcm;};

//! generate the center-of-mass y coordinate with weights
    Float_t cmy_w(){
           sum_w();ycm=0;for(Int_t i=0;i<n;i++){ycm+=y[i]*w[i];};ycm/=sumw;return ycm;};

//! generate the center-of-mass z coordinate with weights
    Float_t cmz_w(){
           sum_w();zcm=0;for(Int_t i=0;i<n;i++){zcm+=z[i]*w[i];};zcm/=sumw;return zcm;};

//! translate in the x direction
    void shift_x(
                 Float_t xt /*!< displacement in the x direction */ 
                ){
           for(Int_t i=0;i<n;i++){x[i]+=xt;};};

//! translate in the y direction
    void shift_y(
                 Float_t yt /*!< displacement in the y direction */ 
                ){
           for(Int_t i=0;i<n;i++){y[i]+=yt;};};

//! translate in the z direction
    void shift_z(
                 Float_t zt /*!< displacement in the z direction */ 
                ){
           for(Int_t i=0;i<n;i++){z[i]+=zt;};};

//! translate to the cm reference frame in the x direction, no weights
    void shift_cmx(){ 
           cmx();for(Int_t i=0;i<n;i++){x[i]-=xcm;};};

//! translate to the cm reference frame in the y direction, no weights
    void shift_cmy(){
           cmy();for(Int_t i=0;i<n;i++){y[i]-=ycm;};};

//! translate to the cm reference frame in the z direction, no weights
    void shift_cmz(){
           cmz();for(Int_t i=0;i<n;i++){z[i]-=zcm;};};

//! translate to the cm reference frame in the x direction with weights
    void shift_cmx_w(){
           cmx_w();for(Int_t i=0;i<n;i++){x[i]-=xcm;};};

//! translate to the cm reference frame in the y direction with weights
    void shift_cmy_w(){
           cmy_w();for(Int_t i=0;i<n;i++){y[i]-=ycm;};};

//! translate to the cm reference frame in the z direction with weights
    void shift_cmz_w(){
           cmz_w();for(Int_t i=0;i<n;i++){z[i]-=zcm;};};

//! mean squared x, no weights
    Float_t msx(){ 
           cmx();
           msr=0;for(Int_t i=0;i<n;i++){msr+=(x[i]-xcm)*(x[i]-xcm);};msr/=n;return msr;};

//! mean squared y, no weights
    Float_t msy(){ 
           cmy();
           msr=0;for(Int_t i=0;i<n;i++){msr+=(y[i]-ycm)*(y[i]-ycm);};msr/=n;return msr;};

//! mean x*y, no weights
    Float_t mxy(){ 
           cmx();cmy();
           msr=0;for(Int_t i=0;i<n;i++){msr+=(x[i]-xcm)*(y[i]-ycm);};msr/=n;return msr;};

//! mean squared radius, no weights
    Float_t msrad(){ 
           cmx();cmy();cmz();
           msr=0;for(Int_t i=0;i<n;i++){msr+=(x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm)+(z[i]-zcm)*(z[i]-zcm);};msr/=n;return msr;};

//! mean squared transverse radius, no weights
    Float_t msrad_t(){
           cmx();cmy();
           msrt=0;for(Int_t i=0;i<n;i++){msrt+=(x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm);};msrt/=n;return msrt;};

//! mean squared radius with weights
    Float_t msrad_w(){
           cmx_w();cmy_w();cmz_w();
           msr=0;for(Int_t i=0;i<n;i++){msr+=w[i]*((x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm)+(z[i]-zcm)*(z[i]-zcm));};msr/=sumw;return msr;};

//! mean squared transverse radius with weights
    Float_t msrad_t_w(){
           cmx_w();cmy_w();
           msrt=0;for(Int_t i=0;i<n;i++){msrt+=w[i]*((x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm));};msrt/=sumw;return msrt;};

//! mean squared transverse radius with weights and smearing
    Float_t msrad_t_w(
             Float_t d  /*!< smearing parameter */
             ){
           cmx_w();cmy_w();
           msrt=0;for(Int_t i=0;i<n;i++){msrt+=w[i]*((x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm)+d*d);};msrt/=sumw;return msrt;};

//! size - the weighted average of the distance from the orgin (in the cm frame)
    Float_t size(){
           cmx_w();cmy_w();
           Float_t siz=0;for(Int_t i=0;i<n;i++){siz+=w[i]*sqrt((x[i]-xcm)*(x[i]-xcm)+(y[i]-ycm)*(y[i]-ycm));};siz/=sumw;return siz;};

//! rotation angle maximizing the m-th cosine Fourier moment with weight r^2
/*! for m=2 this is the angle for passing to the "participant" frame */
    Float_t phrot(
                Int_t m /*!< rank of the Fourier moment, m=0,2,3,4,5,... */
               ){
           Float_t sc=0; Float_t ss=0;
           for(Int_t i=0;i<n;i++){Float_t r2=x[i]*x[i]+y[i]*y[i];Float_t ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*sin(m*ph)*w[i];};
                                return atan2(ss,sc)/m;};

//! rotation angle maximizing the m-th cosine Fourier moment with weight r^k
/*! for m=2 this is the angle for passing to the "participant" frame */
    Float_t phrot(
                Int_t m,   /*!< rank of the Fourier moment, m=0,2,3,4,5,... */
                Float_t k  /*!< power of transverse radius in the weight */
               ){
           Float_t sc=0; Float_t ss=0;
           for(Int_t i=0;i<n;i++){Float_t r2=pow(x[i]*x[i]+y[i]*y[i],k/2.);Float_t ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*sin(m*ph)*w[i];};
                                return atan2(ss,sc)/m;};

//! rotate in the transverse plane by the specified angle 
    void rotate(
                Float_t ph /*!< rotation angle in the transverse plane */
               ){
           for(Int_t i=0;i<n;i++){Float_t xp=x[i]*cos(ph)-y[i]*sin(ph);Float_t yp=y[i]*cos(ph)+x[i]*sin(ph);x[i]=xp;y[i]=yp;};};


//! rotate in the ZX plane by the theta angle
    void rotate_polar(
                      Float_t costh /*!< cosine of the rotation angle (theta) in the ZX plane */
               ){
            Float_t sinth=sqrt(1.-costh*costh);
            for(Int_t i=0;i<n;i++){Float_t zp=z[i]*costh-x[i]*sinth;
                                   Float_t xp=z[i]*sinth+x[i]*costh;
                                   z[i]=zp;x[i]=xp;};};

//! uncluster
    void uncluster(){
           for(Int_t i=0;i<n;i++){Float_t phu=2.*PI*los(); Float_t cthu=2.*los()-1.; Float_t sthu=sqrt(1.-cthu*cthu);
                                  Float_t ru=sqrt(x[i]*x[i]+y[i]*y[i]+z[i]*z[i]);
                                  x[i]=ru*cos(phu)*sthu; y[i]=ru*sin(phu)*sthu; z[i]=ru*cthu;};};
	   
//! m-th cosine Fourier moment in the azimuthal angle in the transverse plane, weight r^2, limited charge range
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    Float_t eps(
              Int_t m,    /*!< rank of the Fourier moment, m=0,2,3,4,5,... */
              Int_t imin, /*!< lowest charge for the sources included */
              Int_t imax  /*!< highest charge for the sources included */
             ){
                      Float_t sc=0; Float_t ss=0;
                      for(Int_t i=0;i<n;i++){if((c[i]>=imin) && (c[i]<=imax)){Float_t r2=x[i]*x[i]+y[i]*y[i];Float_t ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*w[i];};};
                                return sc/ss;};

//! m-th cosine Fourier moment in the azimuthal angle in the transverse plane, weight r^2, no limit on charge
/*! Most popular is the second moment, known as eccentricity. The third moment is relevant for the triangular flow. */   
    Float_t eps(
              Int_t m /*!< rank of the Fourier moment, m=0,2,3,4,5,...  */
             ){ 
                      Float_t sc=0; Float_t ss=0;
                      for(Int_t i=0;i<n;i++){Float_t r2=x[i]*x[i]+y[i]*y[i];Float_t ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*w[i];};
                                return sc/ss;};

//! m-th cosine Fourier moment in the azimuthal angle in the transverse plane, weight r^k, no limit on charge
/*! Most popular is the second moment, known as eccentricity. The third moment is relevant for the triangular flow. */   
    Float_t eps(
              Int_t m,  /*!< rank of the Fourier moment, m=0,2,3,4,5,... */
              Float_t k /*!< power of transverse radius in the weight */
             ){ 
                      Float_t sc=0; Float_t ss=0;
                      for(Int_t i=0;i<n;i++){Float_t r2=pow(x[i]*x[i]+y[i]*y[i],k/2.);Float_t ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*w[i];};
                                return sc/ss;};

//! m-th cosine Fourier moment in the azimuthal angle in the transverse plane, weight r^k, limited charge range
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    Float_t eps(
              Int_t m,    /*!< rank of the Fourier moment, m=0,2,3,4,5,... */
              Float_t k,  /*!< power of transverse radius in the weight */
              Int_t imin, /*!< lowest charge for the sources included */
              Int_t imax  /*!< highest charge for the sources included */
             ){
                      Float_t sc=0; Float_t ss=0;
                      for(Int_t i=0;i<n;i++){if((c[i]>=imin) && (c[i]<=imax)){Float_t r2=pow(x[i]*x[i]+y[i]*y[i],k/2.);Float_t ph=atan2(x[i],y[i]);
                                sc+=r2*cos(m*ph)*w[i]; ss+=r2*w[i];};};
                                return sc/ss;};


//! x-component of the Q vector  in the transverse plane, weight r^k, no limit on charge, with smearing  
/*! The smearing formula is exact. See the Glissando 3 paper for details */
 
   Float_t qx(
              Int_t m,   /*!< rank of the Fourier moment, m=0,2,3,4,5,...  */
              Int_t k,   /*!< power of transverse radius in the weight */
              Float_t d  /*!< smearing parameter */
             ){ 
                      Float_t sc=0, ss=0, r0, r20, r2, ph, t;
                      for(Int_t i=0;i<n;i++){
                                r0=sqrt(x[i]*x[i]+y[i]*y[i]);
				if(r0==0){r0=r0+0.000001;};
                                r20=pow(r0,k);
                                if(d==0){r2=r20;}
                                   else {t=d/(2*r0+d);
					switch(k){

					case 2: r2=pow(d,2)+pow(r0,2); break;
					case 4: r2=2*pow(d,4) + 4*pow(d,2)*pow(r0,2) + pow(r0,4); break;		
					case 6: r2=6*pow(d,6) + 18*pow(d,4)*pow(r0,2) + 9*pow(d,2)*pow(r0,4) + pow(r0,6); break;
					case 8: r2=24*pow(d,8) + 96*pow(d,6)*pow(r0,2) + 72*pow(d,4)*pow(r0,4) + 16*pow(d,2)*pow(r0,6) + pow(r0,8); break;

					case 3: r2=((3*pow(d,4) + 6*pow(d,2)*pow(r0,2) + 2*pow(r0,4))*F0(t))/(4.*r0) + (r0*(2*pow(d,2) + pow(r0,2))*F1(t))/2.; break;
					case 5: r2=((15*pow(d,6)+45*pow(d,4)*pow(r0,2)+28*pow(d,2)*pow(r0,4)+4*pow(r0,6))*F0(t) + 
     							pow(r0,2)*(23*pow(d,4)+24*pow(d,2)*pow(r0,2)+4*pow(r0,4))*F1(t))/(8.*r0); break;
					case 7: r2=((105*pow(d,8)+420*pow(d,6)*pow(r0,2)+376*pow(d,4)*pow(r0,4)+104*pow(d,2)*pow(r0,6)+8*pow(r0,8))*F0(t))/(16.*r0) + 
   							(r0*(44*pow(d,6)+71*pow(d,4)*pow(r0,2)+24*pow(d,2)*pow(r0,4)+2*pow(r0,6))*F1(t))/4.; break;

        					 };
                                         };
                                ph=atan2(x[i],y[i]);
                                sc+=r20*cos(m*ph)*w[i]; ss+=r2*w[i];};
                                return sc/(ss+0.000001);};



//! y-component of the Q vector in the transverse plane, weight r^k, no limit on charge, with smearing  
/*! The smearing formula is exact. See the Glissando 3 paper for details */
 
   Float_t qy(
              Int_t m,   /*!< rank of the Fourier moment, m=0,2,3,4,5,...  */
              Int_t k,   /*!< power of transverse radius in the weight */
              Float_t d  /*!< smearing parameter */
             ){ 
                      Float_t sc=0, ss=0, r0, r20, r2, ph, t;
                      for(Int_t i=0;i<n;i++){
                                r0=sqrt(x[i]*x[i]+y[i]*y[i]);
				if(r0==0){r0=r0+0.000001;};
                                r20=pow(r0,k);
                                if(d==0){r2=r20;}
                                   else {t=d/(2*r0+d);
					switch(k){

					case 2: r2=pow(d,2)+pow(r0,2); break;
					case 4: r2=2*pow(d,4) + 4*pow(d,2)*pow(r0,2) + pow(r0,4); break;		
					case 6: r2=6*pow(d,6) + 18*pow(d,4)*pow(r0,2) + 9*pow(d,2)*pow(r0,4) + pow(r0,6); break;
					case 8: r2=24*pow(d,8) + 96*pow(d,6)*pow(r0,2) + 72*pow(d,4)*pow(r0,4) + 16*pow(d,2)*pow(r0,6) + pow(r0,8); break;

					case 3: r2=((3*pow(d,4) + 6*pow(d,2)*pow(r0,2) + 2*pow(r0,4))*F0(t))/(4.*r0) + (r0*(2*pow(d,2) + pow(r0,2))*F1(t))/2.; break;
					case 5: r2=((15*pow(d,6)+45*pow(d,4)*pow(r0,2)+28*pow(d,2)*pow(r0,4)+4*pow(r0,6))*F0(t) + 
     							pow(r0,2)*(23*pow(d,4)+24*pow(d,2)*pow(r0,2)+4*pow(r0,4))*F1(t))/(8.*r0); break;
					case 7: r2=((105*pow(d,8)+420*pow(d,6)*pow(r0,2)+376*pow(d,4)*pow(r0,4)+104*pow(d,2)*pow(r0,6)+8*pow(r0,8))*F0(t))/(16.*r0) + 
   							(r0*(44*pow(d,6)+71*pow(d,4)*pow(r0,2)+24*pow(d,2)*pow(r0,4)+2*pow(r0,6))*F1(t))/4.; break;
        					 };
                                         };
                                ph=atan2(x[i],y[i]);
                                sc+=r20*sin(m*ph)*w[i]; ss+=r2*w[i];};
                                return sc/(ss+0.000001);};


//! fill the histogram of the transverse distribution in cartesian coordinates, limited charge  
/*! Limited charge range may be used to get the core and mantle (corona) distributions. */
    void fill_xy(
                TH2D *xyh,   /*!< 2-dim cartesian histogram in the transverse plane */ 
                Float_t fac, /*!< normalization factor */
                Int_t imin,  /*!< lowest charge for the sources included */
                Int_t imax   /*!< highest charge for the sources included */
                ){
           for(Int_t i=0;i<n;i++){if((c[i]>=imin) && (c[i]<=imax)){xyh->Fill(x[i],y[i],w[i]*fac);};};};

//! fill the histogram of the transverse distribution in cartesian coordinates, no limit on charge  
    void fill_xy(
                TH2D *xyh,    /*!< 2-dim cartesian histogram in the transverse plane */ 
                Float_t fac   /*!< normalization factor */
                ){
           for(Int_t i=0;i<n;i++){xyh->Fill(x[i],y[i],w[i]*fac);};};

//! substitution overloading
    distr& operator=(const distr& w1); 
};

//! substitution overloading
distr& distr::operator=(const distr &w1) 
{
  if(this != &w1){                           
           n = w1.n;
           x=NULL;y=NULL;z=NULL;x=new Float_t[n];y=new Float_t[n];z=new Float_t[n];c=new Int_t[n];w=new Float_t[n];
	   for(Int_t i=0;i<n;i++){x[i]=w1.x[i];y[i]=w1.y[i];z[i]=w1.z[i];c[i]=w1.c[i];w[i]=w1.w[i];};
	   return *this;};	        
};


//! nucleus class
/*! Class to store distributions of nucleons (or partons) in nuclei. */
class nucleus : public distr {
  
  private:
    bool g;
    Float_t cth, sth, phi, r;

  public:
//! constructor  
  nucleus(
          Int_t k /*!< number of nucleons (or partons) in the nucleus (the mass naumber of the nucleus), 
                     k>0, k=1 - proton, k=2 - deuteron, k>2 - other nucleus */
         ) : distr(k){};

//! copying constructor 
  nucleus(const nucleus& w1) : distr(w1){};

//! substitution overloading
  nucleus& operator = (const nucleus& w1)
         {distr::operator=(w1);return *this;};

//! destructor
//  ~nucleus(){}; 

//! set randomly the distribution of np partons around centre of nucleon A
void set_parton_A(
            const nucleus &nucA,  /*!< nucleus A */
            Int_t np              /*!< number of partons in nucleon */
                 )
    {
           Float_t sx, sy, sz;
	   for (Int_t i=0;i<nucA.n;i++){
		  for (Int_t j=0;j<np;j++){
		        phi=2*PI*los(); 
			cth=2*los()-1; 
			sth=sqrt(1.-cth*cth); 
			r=rlos_parton(); 
			x[np*i+j]=r*cos(phi)*sth; 
			y[np*i+j]=r*sin(phi)*sth; 
			z[np*i+j]=r*cth;  
		                          };
		sx = 0.0; sy=0.0; sz=0.0;
		  for (Int_t j=0;j<np;j++){
		  sx+=x[np*i+j]; sy+=y[np*i+j]; sz+=z[np*i+j];
		                         };
                sx/=np; sy/=np; sz/=np;
		  for (Int_t j=0;j<np;j++){
		  x[np*i+j] = x[np*i+j] - sx + nucA.x[i];
		  y[np*i+j] = y[np*i+j] - sy + nucA.y[i];
		  z[np*i+j] = z[np*i+j] - sz + nucA.z[i];
		                          };
	    };		  
};

//! set randomly the distribution of partons around centres of nucleons inside nucleus B
void set_parton_B(
            const nucleus &nucB,  /*!< nucleus B */
            Int_t np              /*!< number of partons in nucleon */
                 )
                  {
           Float_t sx, sy, sz;
	   for (Int_t i=0;i<nucB.n;i++){
		  for (Int_t j=0;j<np;j++){
			phi=2*PI*los(); 
			cth=2*los()-1; 
			sth=sqrt(1.-cth*cth); 
			r=rlos_parton(); 
	        	x[np*i+j]=r*cos(phi)*sth; 
			y[np*i+j]=r*sin(phi)*sth; 
			z[np*i+j]=r*cth;  
		                         };
		sx = 0.0; sy=0.0; sz=0.0;
		  for (Int_t j=0;j<np;j++){
		  sx+=x[np*i+j]; sy+=y[np*i+j]; sz+=z[np*i+j];
		                         };
                sx/=np; sy/=np; sz/=np;
		  for (Int_t j=0;j<np;j++){
		  x[np*i+j] = x[np*i+j] - sx + nucB.x[i];
		  y[np*i+j] = y[np*i+j] - sy + nucB.y[i];
		  z[np*i+j] = z[np*i+j] - sz + nucB.z[i];
		                                    };
	    };		  
};

//! set the distribution for the proton 
/*! The proton is just placed at the origin */
  void set_proton(){x[0]=0.; y[0]=0.; z[0]=0.;}; 

//! set the distribution for the deuteron 
/*! Use the Hulthen distribution. */
  void set_deuteron(){x[0]=0.; y[0]=0.; z[0]=0.;
       phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlos_hult(); 
       x[1]=r*cos(phi)*sth; y[1]=r*sin(phi)*sth; z[1]=r*cth;}; 


/*! set the distributions of the polarized deuteron */
  void set_deuteron_s1(){ // m=1 case
           double const ah=.228; double const bh=1.18; // Hulthen parameters
           x[0]=0.; y[0]=0.; z[0]=0.; // first nucleon at 0
  Int_t xx=0;while(xx==0){
                        Float_t rr=15.*los(); // relative radius
                        Float_t c=2.*los()-1; // cos(theta)
                        Float_t b=.5*los();  // upper bound of the probability distribution from Mathematica is 0.4
//                        Float_t hult=2*ah*bh*(ah+bh)*(exp(-2.*ah*rr)+exp(-2.*bh*rr)-2.*exp(-(ah+bh)*rr))/(ah-bh)/(ah-bh); // Hulthen distribution
//                        Float_t fd0=sqrt(hult), fd2=sqrt(hult);  

Float_t fd0=(4.33568102/exp(3.46888099*rr*rr) - 4.86170097/exp(3.36265132*rr*rr) + 
             0.27170156/exp(0.44391265*rr*rr) + 0.27790561/exp(0.12953766*rr*rr) - 
              0.3417714/exp(0.07689243*rr*rr) + 0.44597882/exp(0.05815706*rr*rr) - 
             0.34573458/exp(0.03533723*rr*rr) + 0.26478838/exp(0.02964324*rr*rr) + 
             0.00843757/exp(0.00738593*rr*rr))*rr;

Float_t fd2=(-0.23984123/exp(2.85287828*rr*rr) + 0.11562677/exp(0.60245431*rr*rr) + 
               0.0907134/exp(0.23131272*rr*rr) - 0.57255823/exp(0.09054895*rr*rr) + 
 	      0.30327344/exp(0.08431495*rr*rr) + 0.30310748/exp(0.0843142*rr*rr) + 
               0.3026254/exp(0.084312*rr*rr) - 0.57983043/exp(0.06756505*rr*rr) + 
               0.2757009/exp(0.05939491*rr*rr) + 0.00137949/exp(0.01153549*rr*rr))*rr;

                        Float_t cond=(4*pow(fd0,2) + 2*sqrt(2)*(-1 + 3*pow(c,2))*fd0*fd2 + (5 - 3*pow(c,2))*pow(fd2,2))/8.; // from Mathematica
                            if(b<cond){xx=1;phi=2*PI*los(); cth=c; sth=sqrt(1.-cth*cth); x[1]=rr*cos(phi)*sth; y[1]=rr*sin(phi)*sth; z[1]=rr*cth;};
                         };
                          }; 

 void set_deuteron_s0(){ // m=0 case
           double const ah=.228;double const bh=1.18; // Hulthen parameters
           x[0]=0.; y[0]=0.; z[0]=0.; // first nucleon at 0
  Int_t xx=0;while(xx==0){
                        Float_t rr=15.*los(); // relative radius
                        Float_t c=2.*los()-1; // cos(theta)
                        Float_t b=.5*los();  // upper bound of the probability distribution from Mathematica is 0,4
//                        Float_t hult=2*ah*bh*(ah+bh)*(exp(-2.*ah*rr)+exp(-2.*bh*rr)-2.*exp(-(ah+bh)*rr))/(ah-bh)/(ah-bh); // Hulthen distribution
//                        Float_t fd0=sqrt(hult), fd2=sqrt(hult);  

Float_t fd0=(4.33568102/exp(3.46888099*rr*rr) - 4.86170097/exp(3.36265132*rr*rr) + 
             0.27170156/exp(0.44391265*rr*rr) + 0.27790561/exp(0.12953766*rr*rr) - 
              0.3417714/exp(0.07689243*rr*rr) + 0.44597882/exp(0.05815706*rr*rr) - 
             0.34573458/exp(0.03533723*rr*rr) + 0.26478838/exp(0.02964324*rr*rr) + 
             0.00843757/exp(0.00738593*rr*rr))*rr;

Float_t fd2=(-0.23984123/exp(2.85287828*rr*rr) + 0.11562677/exp(0.60245431*rr*rr) + 
               0.0907134/exp(0.23131272*rr*rr) - 0.57255823/exp(0.09054895*rr*rr) + 
 	      0.30327344/exp(0.08431495*rr*rr) + 0.30310748/exp(0.0843142*rr*rr) + 
               0.3026254/exp(0.084312*rr*rr) - 0.57983043/exp(0.06756505*rr*rr) + 
               0.2757009/exp(0.05939491*rr*rr) + 0.00137949/exp(0.01153549*rr*rr))*rr;

                        Float_t cond=(2*pow(fd0,2) + 2*sqrt(2)*(1 - 3*pow(c,2))*fd0*fd2 + (1 + 3*pow(c,2))*pow(fd2,2))/4.; // from Mathematica
                            if(b<cond){xx=1;phi=2*PI*los(); cth=c; sth=sqrt(1.-cth*cth); x[1]=rr*cos(phi)*sth; y[1]=rr*sin(phi)*sth; z[1]=rr*cth;};
                         };
                          }; 


//! set randomly the distribution of nucleons in nucleus A, use the rlosA() function, no correlations
  void set_random_A(){for(Int_t i=0;i<n;i++){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosA(); 
       x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 

//! set randomly the distribution of nucleons in nucleus A, use the rlosA_hos() function, no correlations
void set_random_A_hos(){for(Int_t i=0;i<n;i++){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosA_hos(); 
     x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 	   

//! set randomly the distribution of nucleons in nucleus B, use the rlosB() function, no correlations 
  void set_random_B(){for(Int_t i=0;i<n;i++){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosB(); 
       x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 

//! set randomly the distribution of nucleons in nucleus B, use the rlosB_hos() function, no correlations 
  void set_random_B_hos(){for(Int_t i=0;i<n;i++){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosB_hos(); 
       x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 

//! set randomly the distribution of nucleons in nucleus A with deformation, no correlations
  void set_random_A_def(){for(Int_t i=0;i<n;i++){phi=2*PI*los();   
                                                 r=rlosA_def(&cth,BETA2A,BETA4A); 
                                                 sth=sqrt(1.-cth*cth);
       x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 

//! set randomly the distribution of nucleons in nucleus B with deformation, no correlations 
  void set_random_B_def(){for(Int_t i=0;i<n;i++){phi=2*PI*los();   
                                                 r=rlosB_def(&cth,BETA2B,BETA4B); 
                                                 sth=sqrt(1.-cth*cth);
       x[i]=r*cos(phi)*sth; y[i]=r*sin(phi)*sth; z[i]=r*cth;};}; 

  void set_random_A_def(
                       Float_t d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                       ){Int_t j=0;while(j<n){phi=2*PI*los();
                                              r=rlosA_def(&cth,BETA2A,BETA4A);
                                              sth=sqrt(1.-cth*cth);
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){
 j++;
}; };};


  void set_random_B_def(
                       Float_t d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                       ){Int_t j=0;while(j<n){phi=2*PI*los();
                                              r=rlosB_def(&cth,BETA2B,BETA4B);
                                              sth=sqrt(1.-cth*cth);
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;}; };};


/*! set triangular tritium */
  void set_tritium(
                           Float_t scale /*!< scale of the triangle */
 			) 
{x[0]=-0.5*scale; y[0]=0; z[0]=0;
 x[1]= 0.5*scale; y[1]=0; z[1]=0;
 x[2]=0; y[2]=sqrt(3.)/2.*scale; z[2]=0;}

 
/*! set alpha cluster */
  void set_alpha_cluster(   
                           Float_t d,     /*!< expulsion distance, nucleons cannot be closer to each other than d */
                           Float_t sigma, /*!< not used */
                           Float_t bp,    /*!< not used */
                           Float_t dp     /*!< not used */
 			) 
 {Int_t j=0;while(j<n){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); 
//       r=rlos_alpha(sigma,bp,dp); 
       r=rlos_abf(1.,1.); 
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;};};};



/*! set clustered 7Be (dumbbell of alpha and 3He) */
  void set_berillium_7_cluster(
                           Float_t scale,    /*!< scale parameter for the size of the nucleus */ 
                           Float_t d,        /*!< expulsion distance, nucleons cannot be closer to each other than d */
                           Float_t sigma,    /*!< standard deviation of x,y,z coordinates of nucleons in the alpha cluster */
                           Float_t sigmabis  /*!< standard deviation of x,y,z coordinates of nucleons in the 3He cluster */
			){
	Float_t pol[2][3];
    	pol[0][0]=scale * 0.;     // x
        pol[0][1]=scale * 0.;     // y
        pol[0][2]=scale * 0.5;    // z
	pol[1][0]=scale * 0.;     // x
        pol[1][1]=scale * 0;      // y
        pol[1][2]=scale * (-0.5); // z

	n=7;
	Int_t j=0;
        while (j<n){
		Int_t i=j%2;
                Float_t ss=sigma+i*(sigmabis-sigma);

		Float_t g1=raa.Gaus(0.0,ss);  
		Float_t g2=raa.Gaus(0.0,ss);
		Float_t g3=raa.Gaus(0.0,ss);
 
		x[j]=pol[i][0]+g1;
 		y[j]=pol[i][1]+g2;
	        z[j]=pol[i][2]+g3;
                c[j]=j+1;              // label the nucleon
		if(good_down(j,d)){
		j++; 
		}	   
        	}
		}


/*! set clustered 8Be (dumbbell) */
  void set_berillium_8_cluster(
                           Float_t scale, /*!< scale parameter for the size of the nucleus */ 
                           Float_t d,     /*!< expulsion distance, nucleons cannot be closer to each other than d */
                           Float_t sigma  /*!< standard deviation of x,y,z coordinates of nucleons in the cluster */
			){
	Float_t pol[2][3];
    	pol[0][0]=scale * 0.;     // x
        pol[0][1]=scale * 0.;     // y
        pol[0][2]=scale * 0.5;    // z
	pol[1][0]=scale * 0.;     // x
        pol[1][1]=scale * 0;      // y
        pol[1][2]=scale * (-0.5); // z

	n=8;
	Int_t j=0;
        while (j<n){

/* modified clusters */
       phi=2*PI*los(); 
       cth=2.*los()-1.; sth=sqrt(1.-cth*cth); 
       r=rlos_abf(sigma,1.); 
       Float_t g1=r*cos(phi)*sth; 
       Float_t g2=r*sin(phi)*sth; 
       Float_t g3=r*cth;

		Int_t i=j%2;
		x[j]=pol[i][0]+g1;
 		y[j]=pol[i][1]+g2;
	        z[j]=pol[i][2]+g3;
		if(good_down(j,d)){
		j++;
		}	   
        	}
		}


/*! set clustered 9Be (dumbbell + extra neutron) */
  void set_berillium_9_cluster(
                           Float_t scale, /*!< scale parameter for the size of the nucleus */ 
                           Float_t d, /*!< expulsion distance, nucleons cannot be closer to each other than d */
                           Float_t sigma, /*!< standard deviation of x,y,z coordinates of nucleons in the cluster */
                           Float_t sigmabis/*!< standard deviation of x,y,z coordinates of nucleon no. 9 */
			){
	Float_t pol[2][3];
    	pol[0][0]=scale * 0.;     // x
        pol[0][1]=scale * 0.;     // y
        pol[0][2]=scale * 0.5;    // z
	pol[1][0]=scale * 0.;     // x
        pol[1][1]=scale * 0;      // y
        pol[1][2]=scale * (-0.5); // z

	n=9;
	Int_t j=0;
        while (j<n){
		Int_t i=j%2;
                if(j<8){
		Float_t g1=raa.Gaus(0.0,sigma);
		Float_t g2=raa.Gaus(0.0,sigma);
		Float_t g3=raa.Gaus(0.0,sigma);
		x[j]=pol[i][0]+g1;
 		y[j]=pol[i][1]+g2;
	        z[j]=pol[i][2]+g3;
                        }
                else 
                       {
                phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlos_hole(sigmabis);
                x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; 
                      }
                c[j]=j+1;              // label the nucleon
		if(good_down(j,d)){
		j++;
		}	   
        	}
		}


/*! set clustered 12C (triangle) */
void set_carbon_cluster(
                           Float_t scale, /*!< scale parameter for the size of the nucleus */ 
                           Float_t d, /*!< expulsion distance, nucleons cannot be closer to each other than d */
                           Float_t sigma/*!< standard deviation of x,y,z coordinates of nucleons in the cluster */
			){
	Float_t pol[3][3];
    	pol[0][0]=scale * 0.; // x
        pol[0][1]=scale * 0.57735; // y=1/sqrt(3) * scale
        pol[0][2]=scale * 0.; // z
	pol[1][0]=scale * 0.5; // x
        pol[1][1]=scale * (-0.288675); // y=-1/(2*(sqrt(3)) * scale
        pol[1][2]=scale * 0.; // z
	pol[2][0]=scale * (-0.5); // x
        pol[2][1]=scale * (-0.288675); // y=-1/(2*(sqrt(3)) * scale
        pol[2][2]=scale * 0.; // z

	n=12;
	Int_t j=0;
        while (j<n){
     
/* modified clusters */
       phi=2*PI*los(); 
       cth=2.*los()-1.; sth=sqrt(1.-cth*cth); 
       r=rlos_abf(sigma,1.); 
       Float_t g1=r*cos(phi)*sth; 
       Float_t g2=r*sin(phi)*sth; 
       Float_t g3=r*cth;

		Int_t i=j%3;
		x[j]=pol[i][0]+g1;
 		y[j]=pol[i][1]+g2;
        z[j]=pol[i][2]+g3;
		if(good_down(j,d)){
		j++;
		}	   
       	}
		}

/*! set clustered 16O (tetrahedron) */
void set_oxygen_cluster(
                           Float_t scale, /*!< scale parameter for the size of the nucleus */ 
                           Float_t d, /*!< expulsion distance, nucleons cannot be closer to each other than d */
			   Float_t sigma /*!< standard deviations of x,y,z coordinates of nucleons in the cluster */
			){
	Float_t pol[4][3];
    	pol[0][0]=scale * 0.; // x
        pol[0][1]=scale * 0.57735; // y=1/sqrt(3) * scale
        pol[0][2]=scale * (-0.204124); // z=-1/(2*Sqrt[6]) * scale
	pol[1][0]=scale * 0.5; // x
        pol[1][1]=scale * (-0.288675); // y=-1/(2*(sqrt(3)) * scale
        pol[1][2]=scale * (-0.204124); // z=-1/(2*Sqrt[6]) * scale
	pol[2][0]=scale * (-0.5); // x
        pol[2][1]=scale * (-0.288675); // y=-1/(2*(sqrt(3)) * scale
        pol[2][2]=scale * (-0.204124); // z=-1/(2*Sqrt[6]) * scale
	pol[3][0]=scale * 0.; // x
	pol[3][1]=scale * 0.; // y
	pol[3][2]=scale * 0.612372; // z=Sqrt[3/2]/2 *scale

	n=16;
	Int_t j=0;
        while (j<n){
     
/* modified clusters */
       phi=2*PI*los(); 
       cth=2.*los()-1.; sth=sqrt(1.-cth*cth); 
       r=rlos_abf(sigma,1.); 
       Float_t g1=r*cos(phi)*sth; 
       Float_t g2=r*sin(phi)*sth; 
       Float_t g3=r*cth;

		Int_t i=j%4;
		x[j]=pol[i][0]+g1;
 		y[j]=pol[i][1]+g2;
	        z[j]=pol[i][2]+g3;
		if(good_down(j,d)){
		j++;
		}	   
        	}
		}

/*! set clustered 16O (square) */
void set_oxygen_cluster_square(
                           Float_t scale, /*!< scale parameter for the size of the nucleus */ 
                           Float_t d, /*!< expulsion distance, nucleons cannot be closer to each other than d */
			   Float_t sigma /*!< standard deviation of x,y,z coordinates of nucleons in the cluster */
			  ){
	Float_t pol[4][3];
    	pol[0][0]=scale * 0.5; // x
        pol[0][1]=scale * 0.5; // y
        pol[0][2]=scale * 0. ; // z
	pol[1][0]=scale * 0.5; // x
        pol[1][1]=scale * (-0.5); // y
        pol[1][2]=scale * 0. ; // z
	pol[2][0]=scale * (-0.5); // x
        pol[2][1]=scale * 0.5; // y
        pol[2][2]=scale * 0.; // z
	pol[3][0]=scale * (-0.5); // x
	pol[3][1]=scale * (-0.5); // y
	pol[3][2]=scale * 0.; 

	n=16;
	Int_t j=0;
        while (j<n){
     
/* modified clusters */
       phi=2*PI*los(); 
       cth=2.*los()-1.; sth=sqrt(1.-cth*cth); 
       r=rlos_abf(sigma,1.); 
       Float_t g1=r*cos(phi)*sth; 
       Float_t g2=r*sin(phi)*sth; 
       Float_t g3=r*cth;

		Int_t i=j%4;
		x[j]=pol[i][0]+g1;
 		y[j]=pol[i][1]+g2;
	        z[j]=pol[i][2]+g3;
		if(good_down(j,d)){
		j++;
		}	   
        	}
		}


//! set randomly the distribution of nucleons in the nucleus A with expulsion, use the rlosA() function
/*! The nucleon positions are subsequntly generated acording to the spherically-symmetric 
    Woods-Saxon distribution. If the nucleon happens to be generated closer than the expulsion distance d 
    to any of the previously generated nucleons, it is generated anew, until it is "good". 
    Since this leads to some swelling, the original distribution must be a bit narrower to 
    cancel neutrilize this effect (see our original paper for a detailed discussion). 
    The expulsion simulates in a simple manner the nuclear repulsion and generates the hard-core two-body 
    correlations. */
  void set_random_A(
                   Float_t d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                   )
       {Int_t j=0;while(j<n){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosA(); 
        x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;};};};


//! similar as the above function but for harmonic oscillator shell model, use use the rlosA_hos() function.
  void set_random_A_hos(
                   Float_t d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                   ){Int_t j=0;while(j<n){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosA_hos(); 
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;};};};

//! set randomly the distribution of nucleons in the nucleus B with expulsion, use the rlosB() function
/*! Same as set_random_A for the case of the nucleus B, which in general is different from A */
  void set_random_B(
                   Float_t d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                   ){Int_t j=0;while(j<n){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosB(); 
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;};};};

//! similar as above function but for harmonic oscillator shell model, use use the rlosB_hos() function.
  void set_random_B_hos(
                   Float_t d /*!< expulsion distance, nucleons cannot be closer to each other than d */
                   ){Int_t j=0;while(j<n){phi=2*PI*los(); cth=2*los()-1; sth=sqrt(1.-cth*cth); r=rlosB_hos(); 
       x[j]=r*cos(phi)*sth; y[j]=r*sin(phi)*sth; z[j]=r*cth; if(good_down(j,d)){j++;};};};

//! set the distribution of nucleons in the nucleus from the tables generated earlier
/*! The nucleon position (x,y,z) is taken from the tables read earlier. The pointers x, y, z are 
    originally positioned at 
    px, py, pz at a place corresponding to the beginning of a randomly selected nucleus. 
    The tables with nuclear distributions have to be prepared externally, 
    see, e.g., http://www.phys.psu.edu/~malvioli/eventgenerator/ 
    for distributions involving correlations.
    */
  void set_file(
                Float_t *px, /*!< x coordinate of distributions read from files */ 
                Float_t *py, /*!< y coordinate of distributions read from files */ 
                Float_t *pz, /*!< z coordinate of distributions read from files */ 
                Int_t sn,    /*!< number of entries in the file (should be the mass number time the number of stored nuclei) */
                Int_t nu     /*!< mass number of the nucleus */ 
               ){  
                   Int_t count=raa.Integer(sn/nu)*nu;
                   x = (px+count); y = (py+count); z = (pz+count); // set the pointers
   }; 

//! set the distribution of nucleons in the nucleus from the tables generated earlier, killing correlations
/*! The nucleon position (x,y,z) is taken from the tables read earlier. The pointers x, y, z are positioned at 
    px, py, pz at a place corresponding to a completely randomly selected nucleon. This procedure kills any 
    correlations and is (probably) equivalent to the mixing technique.  
    The tables with nuclear distributions have to be prepared externally.
    */
  void set_file_uncor(
                Float_t *px, /*!< x coordinate of distributions read from files */ 
                Float_t *py, /*!< y coordinate of distributions read from files */ 
                Float_t *pz, /*!< z coordinate of distributions read from files */ 
                Int_t sn,    /*!< number of entries in the file (should be the mass number time the number of stored nuclei) */
                Int_t nu     /*!< mass number of the nucleus */ 
               ){  
                   Int_t count=raa.Integer(sn-nu);
                   x = (px+count); y = (py+count); z = (pz+count); // set the pointers
   }; 

//! distance between two nucleons 
  Float_t dist2(
             Int_t j1, /*!< index of nucleon 1 */
             Int_t j2  /*!< index of nucleon 2 */
             ){return (x[j1]-x[j2])*(x[j1]-x[j2])+(y[j1]-y[j2])*(y[j1]-y[j2])+(z[j1]-z[j2])*(z[j1]-z[j2]);};


//! the pair of nucleons is "good" when the distance between the nucleons is larger than the expulsion distance d
  bool good_pair(
                Int_t j1,  /*!< index of nucleon 1 */
                Int_t j2,  /*!< index of nucleon 2 */
                Float_t d  /*!< expulsion distance - nucleons cannot be closer to each other than d*/
                ){if(dist2(j1,j2)>d*d){return true;} else {return false;};};
  
//! nucleon j is "good" when the distance to all nucleons of index i with i<j is larger than the expulsion distance d
  bool good_down(
                Int_t j,   /*!< index of the nucleon */
                Float_t d  /*!< expulsion distance */
                ){if(j==0){return true;} else {g=true; for(Int_t i=0;i<j;i++){g=g && good_pair(i,j,d);};return g;};};

//! configuration of nucleons in the nucleus is "good" when the distances between all nucleons are larger than the expulsion distance d
  bool good_all(
               Float_t d   /*!< expulsion distance */
               ){g=true; for(Int_t i=1;i<n;i++){g=g && good_down(i,d);};return g;};

}; //class

#endif
