/** \file collision.h
 * Part of GLISSANDO 3
 * 
 */

#ifndef _GL_COLLISION
  #define _GL_COLLISION  

#include "distrib.h"
#include <TMath.h>

//! collision class
/*! 
Class performing the collision of two nuclei.
*/

class collision : public distr {

  public:

  Float_t rd2; /*!< square of the distance between the colliding objects (nucleons or partons) */
  Int_t wc;    /*!< counter of the sources */
  Int_t *wwA,  /*!< wwA[i] is the number of collisions of the i-th object from nucleus A with the objects from nucleus B */ 
        *wwB;  /*!< wwB[i] is the number of collisions of the i-th object from nucleus A with the objects from nucleus A */
  Int_t *wwAB;
  Int_t nwA,   /*!< number of wounded (collided at least once) objects in nucleus A */
        nwB,   /*!< number of wounded (collided at least once) objects in nucleus B */
        nwAB,  /*!< total number of wounded objects (nwA+nwB) generated in the event */
        nzw,   /*!< number of sources with non-zero weight */
        nbin,  /*!< number of binary collisions in the event */
        nhotspot, /*!< number of hot spots */
        specA, /*!< number of spectators in A */
        specB; /*!< number of spectators in B */
        
  Float_t xcmA, /*!< x cm cordinate of the spectators from the A nucleus */
          xcmB, /*!< x cm cordinate of the spectators from the B nucleus */
          ycmA, /*!< y cm cordinate of the spectators from the A nucleus */
          ycmB; /*!< y cm cordinate of the spectators from the B nucleus */
   
  Float_t rpa; /*!< weight of the source (RDS) */
  
  #if(_weight_)
  TH1D *w_distr,      /*!< histogram for the collision profile for the wounded objects */
       *w_distr_bin;  /*!< histogram for the collision profile for the binary collisions */
  #endif

//! constructor
  collision(
           Int_t nnA,  /*!< number of objects in nucleus A */ 
           Int_t nnB   /*!< number of objects in nucleus B */ 
           ){

            Int_t nn=nnA+nnB+nnA*nnB; /*!< maximum number of wounded objects + binary */
  #if(_weight_)
           w_distr = new TH1D("w_distr", "wounding profile p(b^{2})", 100, 0.000001, BTOT*BTOT);
           w_distr->SetXTitle("NN distance squared [fm^{2}]");
           w_distr_bin = new TH1D("w_distr_bin", "binary profile  p(b^{2})", 100, 0.000001, BTOT*BTOT);
           w_distr_bin->SetXTitle("NN distance squared [fm^{2}]");
  #endif
           x=NULL;y=NULL;c=NULL;w=NULL;
           x=new Float_t[nn];y=new Float_t[nn];c=new Int_t[nn];w=new Float_t[nn];
           wwA=NULL;wwB=NULL;wwAB=NULL;
           wwA=new Int_t[nnA];wwB=new Int_t[nnB];wwAB=new Int_t[nnA*nnB];};


//! default constructor
/*! The default constructor assumes 208Pb-208Pb collisions with nucleons. */
  collision(){
  #if(_weight_)
           w_distr = new TH1D("w_distr", "wounding profile  p(b^{2})", 100, 0.000001, BMAX*BMAX);
           w_distr->SetXTitle("NN distance squared [fm^{2}]");
           w_distr_bin = new TH1D("w_distr_bin", "binary profile  p(b^{2})", 100, 0.000001, BMAX*BMAX);
           w_distr_bin->SetXTitle("NN distance squared [fm^{2}]");
  #endif
           x=NULL;y=NULL;c=NULL;w=NULL;
           x=new Float_t[43680];y=new Float_t[43680];c=new Int_t[43680];w=new Float_t[43680];
           wwA=NULL;wwB=NULL;wwAB=NULL;
           wwA=new Int_t[208];wwB=new Int_t[208];wwAB=new Int_t[208*208];};

//! copying constructor 
  collision(const collision& w1) : distr(w1){};

//! substitution overloading
  collision& operator = (const collision& w1)
         {distr::operator=(w1);return *this;};

//! destructor
// ~collision(){
// delete x; delete y; delete z; delete c; delete w;
//              }; 



//! collision between two nuclei
/*! gen_RDS performs the collision of two nuclei, generating the wounded nucleon and the binary collisions, as well as
their RDS (relative deposited strength, or weight). It is the core function of the code, 
implementing the specific model mechanism of the collision. */
  void gen_RDS(
            const nucleus &nA,    /*!< nucleus A, nA>0, nA=1 - proton, nA=2 - deuteron, nA>2 - other nuclei */
            const nucleus &nB,    /*!< nucleus B, nB>0, nB=1 - proton, nB=2 - deuteron, nB>2 - other nuclei */
            Float_t d2,           /*!< wounding distance squared */
            Float_t dbin2,        /*!< binary-collison distance squared */
            Float_t mb            /*!< ratio of the wounding to binary cross sections (for hotspots) */
            ){

    Float_t a     = 1./OMEGA;
    Float_t z;

   for(Int_t i=0;i<nA.n;i++){wwA[i]=0;};
   for(Int_t j=0;j<nB.n;j++){wwB[j]=0;};
   for(Int_t k=0;k<nA.n*nB.n;k++){wwAB[k]=0;};

   nwA=0; nwB=0; nzw=0; rpa=0; nbin=0; nhotspot=0; 
   wc=-1; // to start later from 0 (after wc++ a few lines below)

   for(Int_t i=0;i<nA.n;i++){
	for(Int_t j=0;j<nB.n;j++){
		rd2=(nA.x[i]-nB.x[j])*(nA.x[i]-nB.x[j])+(nA.y[i]-nB.y[j])*(nA.y[i]-nB.y[j]);

			#if(_nnwp_==0)
        		   if(rd2 < d2)
			#elif(_nnwp_==1)
			   if(los() < GA*exp(-GA*rd2/(d2)))
			#elif(_nnwp_==2)
			   z = GAMA*rd2/(d2*OMEGA);
			   if(los() < GAMA*(1.0 - TMath::Gamma(a,z))) 
			#endif

                {wwA[i]++;wwB[j]++;wwAB[i+nA.n*j]++;
                 #if(_weight_) 
                 w_distr->Fill(rd2,1);
                 #endif
    };};};


Float_t over; // overlaid weight


/******************************************************
 wounded from A
*******************************************************/

for(Int_t i=0;i<nA.n;i++){if(wwA[i]>0) 
	       	       {
                        nwA++; // count the wounded objects in A
			wc++; // count the sources
                        c[wc]=wwA[i];

 			x[wc]=nA.x[i]; // store x-coordinate of the source
			if(DW>0){x[wc]+=disp(DW);}; // displace randomly
 			y[wc]=nA.y[i]; // store y-coordinate of the source
			if(DW>0){y[wc]+=disp(DW);}; // displace randomly
// note that RDS is half of the number of wounded nucleons at alpha=0
			w[wc]=(1-ALPHA)/2.0; // wounded objects RDS

#if(_rdsconv_==2)
// pass to the convention used with wounded quarks, i.e., count the wounded objects in RDS
w[wc]=2.*w[wc];
#endif

w[wc]=w[wc]*dist(MODEL,Uw,Vw); // overlay the distribution

rpa+=w[wc]; // add to the weight of the event

	        	if(w[wc]>0){nzw++;}; // count wounded if RDS > 0
                        };};

/******************************************************
 wounded from B
*******************************************************/

for(Int_t i=0;i<nB.n;i++){if(wwB[i]>0) 
	       	       {
                        nwB++; // count the wounded objects in B
			wc++; // count the sources
                        c[wc]=-wwB[i]; // by convention, negative numbers for nucleus B

 			x[wc]=nB.x[i]; // store x-coordinate of the source
			if(DW>0){x[wc]+=disp(DW);}; // displace randomly
 			y[wc]=nB.y[i]; // store y-coordinate of the source
			if(DW>0){y[wc]+=disp(DW);}; // displace randomly
// note that RDS is half of the number of wounded objects an alpha=0 
			w[wc]=(1-ALPHA)/2.0; // wounded objects RDS

#if(_rdsconv_==2)
// pass to the convention used with wounded quarks, i.e., count the wounded objects in RDS
w[wc]=2.*w[wc];
#endif

w[wc]=w[wc]*dist(MODEL,Uw,Vw); // overlay the distribution

rpa+=w[wc]; // add to the weight of the event


	        	if(w[wc]>0){nzw++;}; // count wounded in nzw if RDS > 0
                       };};

/******************************************************
 spectators
*******************************************************/

// coordinates of the spectator "clouds"

   specA=0; xcmA=0; ycmA=0;
   for(Int_t i=0;i<nA.n;i++){if(wwA[i]==0) 
	       	       {specA++;
                        xcmA+=nA.x[i];
                        ycmA+=nA.y[i];
                        };};

   if(specA>0){xcmA/=specA; ycmA/=specA;};

   specB=0; xcmB=0; ycmB=0;
   for(Int_t i=0;i<nB.n;i++){if(wwB[i]==0) 
	       	       {specB++;
                        xcmB+=nB.x[i];
                        ycmB+=nB.y[i];
                        };};

   if(specB>0){xcmB/=specB; ycmB/=specB;};

/******************************************************
 binary
*******************************************************/

if(ALPHA>0 || DOBIN==1){
   for(Int_t i=0;i<nA.n;i++){
	for(Int_t j=0;j<nB.n;j++){
	        	rd2=(nA.x[i]-nB.x[j])*(nA.x[i]-nB.x[j])+(nA.y[i]-nB.y[j])*(nA.y[i]-nB.y[j]);

#if(_bindep_==0)
			#if(_nnwp_==0)
        		   if(rd2 < d2)
			#elif(_nnwp_==1)
			   if(los() < GA*exp(-GA*rd2/(d2)))
			#elif(_nnwp_==2)
			   z = GAMA*rd2/(d2*OMEGA);
			   if(los() < GAMA*(1.0 - TMath::Gamma(a,z)))   
			#endif
#endif

#if(_bindep_==1)
if(wwAB[i+nA.n*j]>0)
#endif
	   		    {
                        #if(_weight_)
                                w_distr_bin->Fill(rd2,1);
                        #endif
                                nbin++; // count the binary collisions
                                wc++;   // count the sources
                                c[wc]=0; // by convention, 0 for binary collisions
				x[wc]=(nA.x[i]+nB.x[j])/2;
				if(DBIN>0){// displace randomly
					x[wc]+=disp(DBIN);
				};
                        	y[wc]=(nA.y[i]+nB.y[j])/2;
				if(DBIN>0){// displace randomly
					y[wc]+=disp(DBIN);
				};
		  		if(los()>mb){
					w[wc]=0;
				} 
				else {
					w[wc]=ALPHA/mb*dist(MODEL,Ubin,Vbin); // binary RDS
					nhotspot++; 
					rpa+=w[wc];  
					if(w[wc]>0){nzw++;};  // count binary in nzw if RDS > 0
				};
			   };
		};
	};
};


  n=wc+1; nwAB=nwA+nwB;
};


//! translate to the cm reference frame in the x direction with weights, including the spactator coordinates
    void shift_cmx_w_c(){
           cmx_w();for(Int_t i=0;i<n;i++){x[i]-=xcm;};xcmA-=xcm;xcmB-=xcm;};
    void shift_cmy_w_c(){
           cmy_w();for(Int_t i=0;i<n;i++){y[i]-=ycm;};ycmA-=ycm;ycmB-=ycm;};


#if(_evout_==2)
//! output of the distribution of points to an external text file (long output)
    void writerds(
                  ofstream& eveout /*!< external file for the full event data */
                 ){
      eveout << rpa << " " << nwAB << " " << specA << " " << atan2(ycmA,xcmA) << " " << specB << " " << atan2(ycmB,xcmB) << endl;
      for(int i=0;i<nwAB;i++){eveout << x[i] << " " << y[i] << " " << c[i] << " " << endl;}
      eveout << nbin << endl;
      for(int i=nwAB;i<nwAB+nbin;i++){eveout << x[i] << " " << y[i] << " " << c[i] << endl;}
    };
#endif

#if(_evout_==1)
//! output short results to an external text file
    void writerds(
                  ofstream& eveout
                 ){
      eveout << nwA << " " << nwB << " " << nbin << " " << rpa << endl;
                  };
#endif

}; //class


#endif

